/*
*
*	File: Hhea.java
*
*
*	ADOBE CONFIDENTIAL
*	___________________
*
*	Copyright 2004-2005 Adobe Systems Incorporated
*	All Rights Reserved.
*
*	NOTICE: All information contained herein is, and remains the property of
*	Adobe Systems Incorporated and its suppliers, if any. The intellectual
*	and technical concepts contained herein are proprietary to Adobe Systems
*	Incorporated and its suppliers and may be covered by U.S. and Foreign
*	Patents, patents in process, and are protected by trade secret or
*	copyright law. Dissemination of this information or reproduction of this
*	material is strictly forbidden unless prior written permission is obtained
*	from Adobe Systems Incorporated.
*
*/

package com.adobe.fontengine.font.opentype;

import java.util.Map;

import com.adobe.fontengine.font.FontByteArray;
import com.adobe.fontengine.font.InvalidFontException;
import com.adobe.fontengine.font.Subset;
import com.adobe.fontengine.font.UnsupportedFontException;
import com.adobe.fontengine.font.opentype.OTByteArray.OTByteArrayBuilder;

/** Gives access to the 'hhea' table.
 * 
 * <h4>Version handling</h4>
 * 
 * <p>'hhea' tables have a major/minor version number.
 * This implementation:
 * <ul> 
 * <li>fully supports version 1.0 tables,</li>
 * <li>interprets 1.x tables as 1.0 tables,</li>
 * <li>rejects other versions with an <code>UnsupportedFontException</code> 
 * at construction time.</li>
 * </ul>
 * 
 * <h4>Synchronization</h4>
 * 
 * <p>Like all tables, these objects are immutable.</p> 
 */

final public class Hhea extends Table {

  protected Hhea (FontByteArray buffer) 
  throws java.io.IOException, InvalidFontException, UnsupportedFontException {
    super (buffer);

    int majorVersion = getTableMajorVersion ();
    if (majorVersion != 1) {
      throw new UnsupportedFontException 
                   ("'hhea' tables with major version " 
                        + majorVersion + "are not supported"); }
  }

  /** Get the major version of the table. */
  public int getTableMajorVersion () throws InvalidFontException {
    return this.data.getuint16 (0);
  }
  
  /** Get the minor version of the table. */
  public int getTableMinorVersion () throws InvalidFontException {
    return this.data.getuint16 (2);
  }

  /** Get the number of hMetrics. */
  public int getNumberOfHMetrics () throws InvalidFontException {
    return this.data.getuint16 (34);
  }
  
  /** Get the caretSlopeRise. */
  public int getRise () throws InvalidFontException {
      return this.data.getint16(18);
  }
  
  /** Get the caretSlopeRun. */
  public int getRun () throws InvalidFontException {
      return this.data.getint16(20);
  }
  
  /** Get the Ascender. */
  public int getAscender () throws InvalidFontException {
    return this.data.getint16 (4); }
  
  /** Get the Descender. */
  public int getDescender () throws InvalidFontException {
    return this.data.getint16 (6); }
  
  /** Get the LineGap. */
  public int getLineGap () throws InvalidFontException {
    return this.data.getint16 (8); }
  
  //----------------------------------------------- subsetting and streaming ---
  
  public void subsetAndStream (Subset subset, Map tables) {
	OTByteArrayBuilder newData = OTByteArray.getOTByteArrayBuilderInstance(this.data);
    newData.setuint16 (34, subset.getNumGlyphs ());      
    tables.put (new Integer (Tag.table_hhea), newData); 
  }  
  
  public void stream(Map tables) {
	  OTByteArrayBuilder newData = this.getDataAsByteArray();
	  tables.put (new Integer (Tag.table_hhea), newData);
  }
}