/*
 * File: Kern.java
 * 
 *	ADOBE CONFIDENTIAL
 *	___________________
 *
 *	Copyright 2004-2005 Adobe Systems Incorporated
 *	All Rights Reserved.
 *
 *	NOTICE: All information contained herein is, and remains the property of
 *	Adobe Systems Incorporated and its suppliers, if any. The intellectual
 *	and technical concepts contained herein are proprietary to Adobe Systems
 *	Incorporated and its suppliers and may be covered by U.S. and Foreign
 *	Patents, patents in process, and are protected by trade secret or
 *	copyright law. Dissemination of this information or reproduction of this
 *	material is strictly forbidden unless prior written permission is obtained
 *	from Adobe Systems Incorporated.
 *
 */

package com.adobe.fontengine.font.opentype;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;

import com.adobe.fontengine.font.FontByteArray;
import com.adobe.fontengine.font.InvalidFontException;
import com.adobe.fontengine.font.Subset;
import com.adobe.fontengine.font.UnsupportedFontException;
import com.adobe.fontengine.font.opentype.OTByteArray.OTByteArrayBuilder;

/** Gives access to the 'kern' table.
 * 
 * <h4>Version handling</h4>
 * 
 * <p>Versioning of 'kern' table is a bit complicated because OpenType
 * defines a format with a minor version number only, and Apple TrueType
 * defines a format with a major/minor version number. This
 * implementation:
 * <ul> 
 * <li>fully supports 0 OpenType tables,</li>
 * <li>fully supports 1.0 Apple TrueType tables,</li>
 * <li>interprets 1.x tables as 1.0 tables,</li>
 * <li>rejects other versions with an <code>UnsupportedFontException</code> 
 * at construction time.</li>
 * </ul>

 * 
 * <h4>Synchronization</h4>
 * 
 * <p>Like all tables, these objects are immutable.</p> 
 */
final public class Kern extends Table {
  private final KernInterpretter interpretter;
  private static final MSKernTable msKernTable = new MSKernTable();
  private static final AppleKernTable appleKernTable = new AppleKernTable();
  
  protected Kern (FontByteArray buffer) 
  throws java.io.IOException, InvalidFontException, UnsupportedFontException {
    super (buffer);
    int majorVersion = getTableMajorVersion ();
    
    switch (majorVersion) {
      case 0: { // ms format of kern table
    	  interpretter = msKernTable;
           break; }

      case 1: {
    	if (data.getuint16(2) != 0)
          throw new UnsupportedFontException ("'kern' table version 1.x not supported yet");
    	else // apple format of kern table
    		interpretter = appleKernTable;
    		break; }

      default: {
        throw new UnsupportedFontException ("'kern' table major version = " 
            + majorVersion + " not supported"); }}}

  
  /** Get the major version of the table.
   * 
   * This is really the minor version for OpenType tables. */
  public int getTableMajorVersion () throws InvalidFontException {
    return this.data.getuint16 (0);
  }

  /** Get the kerning pair for two glyphs.
   */
  public int[] getKernVector (int leftGid, int rightGid)
  throws InvalidFontException, UnsupportedFontException {
    int[] kernVector = new int[2];
    kernVector[0] = kernVector[1] = 0;
    
    boolean crossStream = false;
    
	int nTables = interpretter.getNumTables(data);
	int offset = interpretter.getOffsetOfFirstSubtable();
	for (int t = 0; t < nTables; t++) {
	  crossStream = applyKernSubtable (kernVector, offset, crossStream,
	                                   leftGid, rightGid);
	  offset = offset + interpretter.getLength(data, offset); }
	
    
    return kernVector;
  }
  
  // The Apple and Microsoft specs conflict in a couple of places. This
  // interface hides the differences.
  private interface KernInterpretter {
	  // Returns the number of subtables in the kern table
	  int getNumTables(OTByteArray data) throws InvalidFontException, UnsupportedFontException;
	  
	  // Returns the location of the first subtable in the kern table.
	  int getOffsetOfFirstSubtable();
	  
	  // Returns the size of the subtable header
	  int getSubtableHeaderSize();
	  
	  // Fetch the coverage 
	  int getCoverage(OTByteArray data, int subtableOffset) throws InvalidFontException;
	  
	  // Fetch the subtable's length
	  int getLength(OTByteArray data, int subtableOffset) throws InvalidFontException, UnsupportedFontException;
	  
	  // Returns true iff the subtable's coverage indicates that the table contains vertical kerning
	  boolean verticalSubtable(int coverage);
	  
	  // Returns true iff the subtable's coverage indicates that the table contains minimum kerning
	  boolean minimumSubtable(int coverage);
	  
	  // Returns true iff the subtable's coverage indicates that the table contains override kerning
	  boolean overrideSubtable(int coverage);
	  
	  boolean crossStreamSubtable(int coverage);
	  
	  // Returns the format of the subtable's data. 
	  int subtableFormat(int coverage);
	  
	  void writeSubtableHeader(OTByteArray origData, OTByteArrayBuilder newData, 
				int origOffset, int newOffset, int newSize, int format) throws InvalidFontException;
	  
	  void writeHeader(OTByteArrayBuilder newData, int newNTables);
  }
  
  private static class MSKernTable implements KernInterpretter {
	private static final int COVERAGE_HORIZONTAL   = 0x0001;
	private static final int COVERAGE_MINIMUM      = 0x0002;
	private static final int CROSS_STREAM			= 0x0004;
	private static final int COVERAGE_OVERRIDES    = 0x0008;

	public int getNumTables(OTByteArray data) throws InvalidFontException {
		if (data.getSize() < 4) return 0;
		
		return data.getuint16(2);
	}
	
	public int getSubtableHeaderSize() {
		return 6;
	}
	

	public int getOffsetOfFirstSubtable() {
		return 4;
	}
	
	public int getCoverage(OTByteArray data, int subtableOffset) throws InvalidFontException {
		  return data.getuint16(subtableOffset + 4);
	}
	  
	public int getLength(OTByteArray data, int subtableOffset) throws InvalidFontException {
		return data.getuint16(subtableOffset + 2);
	}
	  
	public boolean minimumSubtable(int coverage) {
		return (coverage & COVERAGE_MINIMUM) != 0;
	}
	  
	public boolean overrideSubtable(int coverage) {
		return (coverage & COVERAGE_OVERRIDES) != 0;
	} 
	  
	public int subtableFormat(int coverage) {
		return coverage >> 8;
	}
	 
	public boolean verticalSubtable(int coverage) {
		return (coverage & COVERAGE_HORIZONTAL) == 0;
	}
	  
	public boolean crossStreamSubtable(int coverage) {
		  return (coverage & CROSS_STREAM) != 0;
	}
	  
	
	public void writeSubtableHeader(OTByteArray origData, OTByteArrayBuilder newData, 
			int origOffset, int newOffset, int newSize, int format) 
	throws InvalidFontException {
		int coverage = getCoverage(origData, origOffset);
		coverage = (coverage & 0xff) | ((format << 8) & 0xff00);
		newData.setuint16(newOffset, origData.getuint16 (origOffset));
        newData.setuint16(newOffset + 2, newSize);
  	    newData.setuint16(newOffset + 4, coverage);
	}
	
	public void writeHeader(OTByteArrayBuilder newData, int newNTables)
	{
		newData.setuint16(0, 0);
		newData.setuint16(2, newNTables);
	}
  }
  
  private static class AppleKernTable implements KernInterpretter {
	  private static final int COVERAGE_VERTICAL = 0x8000;
	  private static final int CROSS_STREAM = 0x4000;
	  
	  public int getOffsetOfFirstSubtable() {
		return 8;
	  }
	  
	  public int getSubtableHeaderSize() {
		return 8;
	}
	  
	  public int getNumTables(OTByteArray data) throws InvalidFontException, UnsupportedFontException {
		  if (data.getSize() < 8) return 0;
		  
		  return data.getuint32asint(4, "Too many subtables in kern table");
	  }

	  public int getCoverage(OTByteArray data, int subtableOffset) throws InvalidFontException {
			  return data.getuint16(subtableOffset + 4);
	  }
		  
	  public int getLength(OTByteArray data, int subtableOffset) throws InvalidFontException, UnsupportedFontException {
			return data.getuint32asint(subtableOffset, "Kern subtable too large");
	  }
		
	  public boolean minimumSubtable(int coverage) {
		return false;
	  }
	  
	  public boolean overrideSubtable(int coverage) {
		return false;
	  }
	  
	  public int subtableFormat(int coverage) {
		return coverage & 0xff;
	  }
	  
	  public boolean verticalSubtable(int coverage) {
		  return (coverage & COVERAGE_VERTICAL) != 0;
	  }

	  public boolean crossStreamSubtable(int coverage) {
		  return (coverage & CROSS_STREAM) != 0;
	  }
	  
	  public void writeHeader(OTByteArrayBuilder newData, int newNTables) {
		  newData.setuint32(0, 0x00010000);
		  newData.setuint32(4, newNTables);
	  }
	  
	  public void writeSubtableHeader(OTByteArray origData, OTByteArrayBuilder newData, int origOffset, int newOffset, int newSize, int format) throws InvalidFontException {
		    int coverage = getCoverage(origData, origOffset);
		    coverage = (coverage & 0xff00) | (format & 0xff);
	        newData.setuint32(newOffset, newSize);
	        newData.setuint16(newOffset + 4, coverage);
	  	    newData.setuint16(newOffset + 6, origData.getuint16 (origOffset + 6));
	}
  }
  
  
  
  private boolean applyKernSubtable (int[] kernVector, int offset,
                                     boolean crossStream,
                                     int leftGid, int rightGid)
  throws InvalidFontException, UnsupportedFontException {
    int coverage = interpretter.getCoverage(data, offset);
    int format = interpretter.subtableFormat(coverage);
    
    // we don't support vertical text yet, so we can ignore this table
    if (interpretter.verticalSubtable(coverage)) {
      return crossStream; }
    
    switch (format) {
      case 0: { 
        crossStream = applyKernSubtableFormat0 (kernVector, offset, crossStream,
                                                leftGid, rightGid);
        break; }
      
      case 2: { 
        crossStream = applyKernSubtableFormat2(kernVector, offset, crossStream, leftGid, rightGid);
        break; }
      
      default: {
        throw new InvalidFontException ("invalid kern subtable format ("
                                        + format + ")"); }}
    
    return crossStream;
  }
    
  private boolean applyKernSubtableFormat2 (int[] kernVector, int offset,
            boolean crossStream,
            int leftGid, int rightGid) 
  throws InvalidFontException
  {
      int leftClassOffset = this.data.getOffset(offset, interpretter.getSubtableHeaderSize() + 2);
      int rightClassOffset = this.data.getOffset(offset, interpretter.getSubtableHeaderSize() + 4);
      
      int firstLeftGlyph = this.data.getuint16(leftClassOffset);
      int numLeftGlyph = this.data.getuint16(leftClassOffset + 2);
      if (leftGid < firstLeftGlyph || leftGid >= firstLeftGlyph + numLeftGlyph)
    	  return crossStream;

      int firstRightGlyph = this.data.getuint16(rightClassOffset);
      int numRightGlyph = this.data.getuint16(rightClassOffset + 2);
      if (rightGid < firstRightGlyph || rightGid >= firstRightGlyph + numRightGlyph)
    	  return crossStream;

      int leftClass = this.data.getuint16(leftClassOffset + (leftGid - firstLeftGlyph)*2 + 4);
      int rightClass = this.data.getuint16(rightClassOffset + (rightGid - firstRightGlyph)*2 + 4);
      int value = this.data.getint16(offset + leftClass + rightClass);
      
      if ((value & 0xffff) == 0x8000) {
          crossStream = false; }
      else  {
	      int vectorIndex = crossStream ? 1 : 0;
	      int coverage = interpretter.getCoverage(data, offset);
	      if (interpretter.minimumSubtable(coverage)) {
	          kernVector [vectorIndex] = Math.max (kernVector [vectorIndex], value); }
	      else if (interpretter.overrideSubtable(coverage)) {
	          kernVector [vectorIndex] = value; }
	      else {
	    	  kernVector [vectorIndex] += value; } }
    	  
	  return crossStream;
  }
    
  private boolean applyKernSubtableFormat0 (int[] kernVector, int offset,
                                            boolean crossStream,
                                            int leftGid, int rightGid)
  throws InvalidFontException {
    int nPairs = this.data.getuint16 (offset + interpretter.getSubtableHeaderSize());
    int min = 0;
    int max = nPairs;
    
    while (min < max) {
      int mid = (min + max) / 2;
      int l = this.data.getuint16 (offset + interpretter.getSubtableHeaderSize() + 8 + 6*mid);
      int r = this.data.getuint16 (offset + interpretter.getSubtableHeaderSize() + 8 + 6*mid + 2);
      if (   l < leftGid 
          || (l == leftGid && r < rightGid)) {
        min = mid + 1; }
      else if (   leftGid < l 
               || (leftGid == l && rightGid < r)) {
        max = mid; }
      else {
        int value = this.data.getint16 (offset + interpretter.getSubtableHeaderSize() + 8+ 6*mid + 4);
        
        if ((value & 0xffff) == 0x8000) {
          crossStream = false; }
        else {
          int coverage = interpretter.getCoverage(data, offset);
          int vectorIndex = crossStream ? 1 : 0;
          
          if (interpretter.minimumSubtable(coverage)) {
            kernVector [vectorIndex] = Math.max (kernVector [vectorIndex], value); }
          else if (interpretter.overrideSubtable(coverage)) {
            kernVector [vectorIndex] = value; }
          else {
            kernVector [vectorIndex] += value; }}
      
        return crossStream; }}
    
    return crossStream;    
  }  

  public void stream(Map tables) {
	  OTByteArrayBuilder newData = this.getDataAsByteArray();
	  tables.put (new Integer (Tag.table_kern), newData);
  }
  
  private class KernEntry implements Comparable {
	  final int left;
	  final int right;
	  final int kern;
	  
	  KernEntry(int left, int right, int kern)
	  {
		  this.left = left;
		  this.right = right;
		  this.kern = kern;
	  }
	  
	  public int compareTo(Object o) {
		KernEntry right = (KernEntry) o;
		if (this.left < right.left)
			return -1;
		else if (this.left > right.left)
			return 1;
		else if (this.right < right.right)
			return -1;
		else if (this.right > right.right)
			return 1;
		
		return 0;
	}
	  
	public String toString() {
		return "[" + this.left + ", " + this.right + ", " + this.kern + "]";
	}
  }
  
  private int writeTableDataFormat0(OTByteArrayBuilder builder, int newOffset, List /*<KernEntry>*/ pairs)
  {
	  if (pairs.size() == 0)
		  return 0;

	  Collections.sort(pairs);
	  int size = 8 + 6*pairs.size();
	  builder.ensureCapacity(newOffset + size);
	  
	  int entrySelector = (int)Math.floor(log2(pairs.size()));
	  int searchRange = (int)(Math.pow(2, entrySelector));
	  builder.setuint16(newOffset, pairs.size());
	  builder.setuint16(newOffset + 2, searchRange * 6);
	  builder.setuint16(newOffset + 4, entrySelector);
	  builder.setuint16(newOffset + 6, (pairs.size() - searchRange) * 6);
	  
	  for (int i = 0; i < pairs.size(); i++)
	  {
		  KernEntry entry = (KernEntry)pairs.get(i);
		  builder.setuint16(newOffset + 8 + 6*i, entry.left);
		  builder.setuint16(newOffset + 8 + 6*i + 2, entry.right);
		  builder.setuint16(newOffset + 8 + 6*i + 4, entry.kern);
	  }
  
	  return size;
  }
  
  private int writeTableDataFormat0(OTByteArrayBuilder builder, int origOffset, int newOffset, Subset subset) 
  throws InvalidFontException, UnsupportedFontException
  {
	  int origNPairs = this.data.getuint16(origOffset);
	  List /* <KernEntry> */ pairs = new ArrayList();
	  for (int i = 0; i < origNPairs; i++)
	  {
		  int leftGid;
		  
		  try {
			  leftGid = this.data.getuint16(origOffset + 8 + 6*i);
		  } catch (InvalidFontException e) {
			  break;
		  }

		  if (subset.getExistingSubsetGid(leftGid) == -1)
			  continue;
		  
		  int rightGid = this.data.getuint16(origOffset + 8 + 6*i + 2);

		  if (subset.getExistingSubsetGid(rightGid) == -1)
			  continue;

		  pairs.add(new KernEntry(subset.getExistingSubsetGid(leftGid), subset.getExistingSubsetGid(rightGid), this.data.getint16(origOffset + 8 + 6*i + 4)));
	  }
	  
	  return writeTableDataFormat0(builder, newOffset, pairs);
  }
  
  private int  writeTableDataFormat2As0(OTByteArrayBuilder builder, int origSTDataOffset, int newSTDataOffset, Subset subset) 
  throws InvalidFontException
  {
	  int leftOffset = origSTDataOffset - interpretter.getSubtableHeaderSize() + data.getuint16(origSTDataOffset + 2);
	  int rightOffset = origSTDataOffset - interpretter.getSubtableHeaderSize() + data.getuint16(origSTDataOffset + 4);
	  int leftFirstGlyph = data.getuint16(leftOffset);
	  int leftNGlyphs = data.getuint16(leftOffset + 2);
	  int rightFirstGlyph = data.getuint16(rightOffset);
	  int rightNGlyphs = data.getuint16(rightOffset + 2);
	  List /* <KernEntry> */ pairs = new ArrayList();
	 
	  for (int i = 0; i < subset.getNumGlyphs(); i++)
	  {
		  int fullLeft = subset.getFullGid(i);
		  if (fullLeft < leftFirstGlyph || fullLeft >= leftFirstGlyph + leftNGlyphs)
			  continue;
		  
		  // We found a possible left...
		  
		  for (int j = 0; j < subset.getNumGlyphs(); j++)
		  {
			  int fullRight = subset.getFullGid(j);
			  if (fullRight < rightFirstGlyph || fullRight >= rightFirstGlyph + rightNGlyphs)
				  continue;
			  
			  // We found a possible pair...see if it has non-0 kerning.
			  
			  int leftClass = this.data.getuint16(leftOffset + (fullLeft - leftFirstGlyph)*2 + 4);
		      int rightClass = this.data.getuint16(rightOffset + (fullRight - rightFirstGlyph)*2 + 4);
		      int value = this.data.getint16(origSTDataOffset - interpretter.getSubtableHeaderSize() + leftClass + rightClass);
		      if (value == 0)
		    	  continue;
		      
		      // We found a pair.
		      pairs.add(new KernEntry(i, j, value));
			  
		  }
	  }
	  return writeTableDataFormat0(builder, newSTDataOffset, pairs);
  }
  
  private int writeFormat0TableData(OTByteArrayBuilder builder, int origSTDataOffset, int newSTDataOffset, Subset subset, int format) throws InvalidFontException, UnsupportedFontException
  {
	  switch (format)
	  {
	  case 0:
		  return writeTableDataFormat0(builder, origSTDataOffset, newSTDataOffset, subset);

	  case 2:
		  return writeTableDataFormat2As0(builder, origSTDataOffset, newSTDataOffset, subset);
	  }
	  return 0;
  }

  private static double log2(double d) {
      return Math.log(d)/Math.log(2.0);
  }
  
  public void subsetAndStreamForSWF(Subset subset, Map tables) throws InvalidFontException, UnsupportedFontException
  {
	  OTByteArrayBuilder newData = OTByteArray.getOTByteArrayBuilderInstance();
	  int origNTables = interpretter.getNumTables(this.data);
	    
	  if (origNTables > 0)
	  {
	      int newNTables = 0;
		  int origOffset = interpretter.getOffsetOfFirstSubtable();
		  int newOffset = interpretter.getOffsetOfFirstSubtable();
		  for (int t = 0; t < origNTables; t++) {
			  int coverage = interpretter.getCoverage(this.data, origOffset);
		      
			  // Skip these subtables. CTS doesn't support them.
			  if (interpretter.verticalSubtable(coverage) || 
					  interpretter.minimumSubtable(coverage) ||
					  interpretter.crossStreamSubtable(coverage))
				  continue;
			  
			  int format = interpretter.subtableFormat(coverage);
			  
		      int newSize = writeFormat0TableData(newData, origOffset + interpretter.getSubtableHeaderSize(), 
		    		  newOffset + interpretter.getSubtableHeaderSize(), subset, format);
		          
	          if (newSize > 0)
	          {
	        	  interpretter.writeSubtableHeader(this.data, newData, origOffset, newOffset, newSize, 0);
	        	  newNTables++;
	          }
	          
	          origOffset = origOffset + interpretter.getLength(this.data, origOffset); 
	          newOffset += newSize;
	        }
		  
		  if (newNTables != 0) {
			  interpretter.writeHeader(newData, newNTables);
			  tables.put (new Integer (Tag.table_kern), newData);
		  }
	  }
  }
  
}

