/*
*
*	File: LayoutTableSubsetter.java
*
*
*	ADOBE CONFIDENTIAL
*	___________________
*
*	Copyright 2008 Adobe Systems Incorporated
*	All Rights Reserved.
*
*	NOTICE: All information contained herein is, and remains the property of
*	Adobe Systems Incorporated and its suppliers, if any. The intellectual
*	and technical concepts contained herein are proprietary to Adobe Systems
*	Incorporated and its suppliers and may be covered by U.S. and Foreign
*	Patents, patents in process, and are protected by trade secret or
*	copyright law. Dissemination of this information or reproduction of this
*	material is strictly forbidden unless prior written permission is obtained
*	from Adobe Systems Incorporated.
*
*/

package com.adobe.fontengine.font.opentype;

import java.util.Arrays;

import com.adobe.fontengine.font.InvalidFontException;
import com.adobe.fontengine.font.Subset;
import com.adobe.fontengine.font.UnsupportedFontException;
import com.adobe.fontengine.font.opentype.LayoutTable.ClassConsumer;
import com.adobe.fontengine.font.opentype.LayoutTable.CoverageConsumer;
import com.adobe.fontengine.font.opentype.OTByteArray.OTByteArrayBuilder;

class LayoutTableSubsetter {
	protected final LayoutTable origTable;
	protected final OTByteArrayBuilder builder;
	
	// the biggest offset permitted when dealing with non-extension lookups.
	static int maxOffset = 0xffff;
	
	LayoutTableSubsetter(LayoutTable origTable, OTByteArrayBuilder builder)
	{
		this.origTable = origTable;
		this.builder = builder;
	}
	
	protected int writeByteArrayAtOffset(int newOffset, OTByteArray byteArrayToWrite) throws InvalidFontException
	{
		builder.replace(newOffset, byteArrayToWrite, 0, byteArrayToWrite.getSize());
		return byteArrayToWrite.getSize();
	}
	

	protected int writeClassDef(int origSTOffset, int stOffsetDelta, int newSTOffset, int newSubtableSize, Subset subset, int origNumGlyphs) throws InvalidFontException, UnsupportedFontException
	{
		ClassDefGenerator generator = ClassDefGenerator.newInstance(origTable, origTable.data.getOffset(origSTOffset, stOffsetDelta), subset, origNumGlyphs);
		builder.setuint16(newSTOffset + stOffsetDelta, newSubtableSize);
		OTByteArray array = generator.generateClass().toOTByteArray();
		return writeByteArrayAtOffset(newSTOffset + newSubtableSize, array);
	}
	

	private static class CoverageFormatDecider implements CoverageConsumer
	{
		private final Subset subset;
		boolean[] subsetGlyphInCoverage;
		int numGids = 0;
		
		CoverageFormatDecider(Subset subset) {
			this.subset = subset;
			subsetGlyphInCoverage = new boolean[subset.getNumGlyphs()];
			Arrays.fill(subsetGlyphInCoverage, false);
		}
		
		public boolean glyphInfo(int gid, int coverageIndex) throws InvalidFontException, UnsupportedFontException {
			int subsetGid = subset.getExistingSubsetGid(gid);
			if (subsetGid != -1)
			{
				subsetGlyphInCoverage[subsetGid] = true;
				numGids++;
			}

			return true;
		}
		
		int countRanges() {
			int nRanges;
			if (subsetGlyphInCoverage[0] == true)
				nRanges = 1;
			else
				nRanges = 0;
			
			for (int i = 1; i < subsetGlyphInCoverage.length; i++)
			{
				if (!subsetGlyphInCoverage[i-1] && subsetGlyphInCoverage[i])
					nRanges++;
			}
			
			return nRanges;
		}
		
	}
	
	private static class ClassFormatDecider implements ClassConsumer
	{
		private final Subset subset;
		int[] subsetClasses;
		
		ClassFormatDecider(Subset subset)
		{
			this.subset = subset;	
			subsetClasses = new int[subset.getNumGlyphs()];
		}
		
		public boolean glyph(int glyphID, int classID) throws UnsupportedFontException, InvalidFontException {
			int subsetGid = subset.getExistingSubsetGid(glyphID);
			if (subsetGid != -1)
			{
				subsetClasses[subsetGid] = classID;			
			}
			
			return true;
		}
		
		int getFormat1Size()
		{
			int start, end;
			for (start = 0; start < subsetClasses.length; start++)
			{
				if (subsetClasses[start] != 0)
					break;
			}
			
			for ( end = subsetClasses.length - 1; end > start ; end--)
			{
				if (subsetClasses[end] != 0)
					break;
			}
			
			return 6 + 2*(end-start+1);
			
		}
		
		int getFormat2Size()
		{
			int i;
			int nRanges = 0;
			
			for (i = 0; i < subsetClasses.length; i++)
			{
				if (subsetClasses[i] != 0)
					break;
			}
			
			if (i < subsetClasses.length)
				nRanges++; // we found the start of the first range.
			
			i++;
			
			while (i < subsetClasses.length)
			{
				for (; i < subsetClasses.length && subsetClasses[i] == subsetClasses[i-1]; i++)
				{
					// we found the end of the current range.
				}
				
				for (; i < subsetClasses.length && subsetClasses[i] == 0; i++)
				{
					// we found a class 0 range. Don't count it in nRanges.
				}

				if (i < subsetClasses.length)
					nRanges++; // we found the start of the first range.
				
				i++;
				
			}
			
			return 4 + 6*nRanges;
		}
	}
	
	static abstract class ClassDefGenerator 
	{
		protected final int[] subsetClasses;
		
		ClassDefGenerator(int[] subsetClasses)
		{
			this.subsetClasses = subsetClasses;
		}
		
		static ClassDefGenerator newInstance(LayoutTable origTable, int origClassDefOffset, Subset subset, int numGlyphs) 
		throws InvalidFontException, UnsupportedFontException
		{
			ClassFormatDecider consumer = new ClassFormatDecider(subset);
			origTable.iterateClass(origClassDefOffset, numGlyphs, consumer, -1);
			int format1Size = consumer.getFormat1Size();
			int format2Size = consumer.getFormat2Size();
			
			if (format1Size < format2Size)
			{
				return new ClassDef1Generator(consumer.subsetClasses, format1Size);
			} else
			{
				return new ClassDef2Generator(consumer.subsetClasses, format2Size);
			}
		}
		
		abstract OTByteArrayBuilder generateClass() throws InvalidFontException, UnsupportedFontException;
	}
	
	static class ClassDef1Generator extends ClassDefGenerator
	{
		final private OTByteArrayBuilder builder;
		
		ClassDef1Generator(int[] subsetClasses, int size)
		{
			super(subsetClasses);
			builder = OTByteArray.getOTByteArrayBuilderInstance(size);
		}
		
		OTByteArrayBuilder generateClass() throws InvalidFontException, UnsupportedFontException {
			int startGID;
			for (startGID = 0; startGID < subsetClasses.length; startGID++)
			{
				if (subsetClasses[startGID] != 0)
					break;
			}
			
			int endGID;
			for (endGID = subsetClasses.length-1; endGID >= 0; endGID--)
			{
				if (subsetClasses[endGID] != 0)
					break;
			}
			
			builder.setuint16(0, 1);
			if (startGID <= endGID)
			{
				builder.setuint16(2, startGID);
				builder.setuint16(4, endGID - startGID + 1);
			} else
			{
				builder.setuint16(2, 0);
				builder.setuint16(4, 0);
			}
			
			for (int i = 0; i <= endGID-startGID; i++)
			{
				builder.setuint16(6 + 2*i, subsetClasses[startGID+i]);
			}

			return builder;
		}
	}
	
	static class ClassDef2Generator extends ClassDefGenerator
	{
		private final OTByteArrayBuilder builder;
		private final int nRanges;
		
		ClassDef2Generator(int[] subsetClasses, int size)
		{
			super(subsetClasses);
			builder = OTByteArray.getOTByteArrayBuilderInstance(size);
			nRanges = (size - 4)/6;
		}
		
		OTByteArrayBuilder generateClass() throws InvalidFontException, UnsupportedFontException {
			builder.setuint16(0, 2);
			builder.setuint16(2, nRanges);
			int i;
			int currentRange = 0;
			
			for (i = 0; i < subsetClasses.length; i++)
			{
				if (subsetClasses[i] != 0)
					break;
			}
			
			if (i < subsetClasses.length)
			{
				builder.setuint16(4, i);
				builder.setuint16(8, subsetClasses[i]);
				currentRange++; // we found the start of the first range.
			}
			
			i++;
			
			while (i < subsetClasses.length)
			{
				for (; i < subsetClasses.length && subsetClasses[i] == subsetClasses[i-1]; i++)
				{
					// find the end of this range.
				}

				builder.setuint16(4 + 6*(currentRange-1) + 2, i-1); // set the end gid.
				
				for (; i < subsetClasses.length && subsetClasses[i] == 0; i++)
				{
					// we found a class 0 range. Skip it.
				}

				if (i < subsetClasses.length)
				{
					builder.setuint16(4 + 6*currentRange, i);
					builder.setuint16(4 + 6*currentRange + 4, subsetClasses[i]);
					currentRange++; // we found the start of the next range.
				}
				
				i++;
				
			}

			return builder;
		}
	}
	
	static abstract class CoverageGenerator 
	{
		OTByteArrayBuilder newCoverage;
		protected final LayoutTable origTable;
		protected final int origCoverageOffset;
		protected final Subset subset;
		protected int numGlyphsFound = 0;
		
		CoverageGenerator(LayoutTable origTable, int origCoverageOffset, Subset subset)
		{
			this.origTable = origTable;
			this.origCoverageOffset = origCoverageOffset;
			this.subset = subset;
		}
		
		static CoverageGenerator newInstance(LayoutTable origTable, int origCoverageOffset, Subset subset) 
		throws InvalidFontException, UnsupportedFontException
		{
			CoverageFormatDecider consumer = new CoverageFormatDecider(subset);
			origTable.iterateCoverage(origCoverageOffset, null, consumer);
			
			int nRanges = consumer.countRanges();
			int format1Size = 2*consumer.numGids;
			int format2Size = 6*nRanges;
			
			if  (format1Size < format2Size)
			{
				return new CoverageFormat1Generator(origTable, origCoverageOffset, subset, consumer.numGids, consumer.subsetGlyphInCoverage);
			} else
			{
				return new CoverageFormat2Generator(origTable, origCoverageOffset, subset, nRanges, consumer.subsetGlyphInCoverage);
			}
		}
		
		
		abstract OTByteArrayBuilder generateCoverage() throws InvalidFontException, UnsupportedFontException;
	}
	
	static private class CoverageFormat1Generator extends CoverageGenerator
	{
		private final boolean[] subsetGlyphInCoverage;
		
		CoverageFormat1Generator(LayoutTable origTable, int origCoverageOffset, Subset subset, int numGids, boolean[] subsetGlyphInCoverage)
		{
			super(origTable, origCoverageOffset, subset);
			this.newCoverage = OTByteArray.getOTByteArrayBuilderInstance(4 + 2*numGids);
			this.newCoverage.setuint16(0, 1);  // emit the format;
			this.newCoverage.setuint16(2, numGids);
			this.subsetGlyphInCoverage = subsetGlyphInCoverage;
			
		}
		
		OTByteArrayBuilder generateCoverage() throws InvalidFontException, UnsupportedFontException
		{
			int j = 0;
			for (int i = 0; i < subsetGlyphInCoverage.length; i++)
			{
				if (subsetGlyphInCoverage[i])
				{
					this.newCoverage.setuint16(4 + 2*j, i);
					j++;
				}
			}
			
			numGlyphsFound = j;
			return newCoverage;
		}
	}
	
	static private class CoverageFormat2Generator extends CoverageGenerator 
	{
		private final int numRanges;
		final private boolean[] subsetGlyphInCoverage;
		
		public CoverageFormat2Generator(LayoutTable origTable, int origCoverageOffset, Subset subset, int numRanges, boolean[] subsetGlyphInCoverage) {
			super(origTable, origCoverageOffset, subset);
			this.numRanges = numRanges;
			this.newCoverage = OTByteArray.getOTByteArrayBuilderInstance(4 + 6*numRanges);
			this.newCoverage.setuint16(0, 2);  // emit the format;
			this.newCoverage.setuint16(2, numRanges);
			this.subsetGlyphInCoverage = subsetGlyphInCoverage;
		}
		
		private void writeRanges()
		{
			int currentRange = 0;
			if (subsetGlyphInCoverage[0])
			{
				this.newCoverage.setuint16(4, 0);
				numGlyphsFound++;
			}
			else
				currentRange = -1;
			
			for (int i = 1; i < subsetGlyphInCoverage.length; i++)
			{
				if (subsetGlyphInCoverage[i])
					numGlyphsFound++;
				
				if (!subsetGlyphInCoverage[i-1] && subsetGlyphInCoverage[i])
				{
					// start of range
					currentRange++;
					this.newCoverage.setuint16(4 + 6*currentRange, i);
				}
				else if (subsetGlyphInCoverage[i-1] && !subsetGlyphInCoverage[i])
				{
					//end of range
					this.newCoverage.setuint16(4 + 6*currentRange + 2, i-1);
				}
			}
			
			if (subsetGlyphInCoverage[subsetGlyphInCoverage.length-1])
			{
				// If the last glyph is in the last range, we need to close down the range.
				this.newCoverage.setuint16(4 + 6*currentRange + 2, subsetGlyphInCoverage.length-1);
			}
		}
		
		OTByteArrayBuilder generateCoverage() throws InvalidFontException, UnsupportedFontException
		{

			writeRanges();
			if (numRanges != 0)
			{				
				int currentCoverageIndex = 0;
				for (int i = 0; i < numRanges; i++)
				{
					int start = this.newCoverage.getuint16(4 + 6*i);
					int end = this.newCoverage.getuint16(4 + 6*i + 2);
					this.newCoverage.setuint16(4 + 6*i + 4, currentCoverageIndex);
					currentCoverageIndex += 1 + end - start;
				}
			}
			
			return newCoverage;
		}
	}	
	
	static class ClassCoveredBySubset implements ClassConsumer
	{
		private final Subset subset;
		private boolean glyphFound;
		
		ClassCoveredBySubset(Subset subset)
		{
			this.subset = subset;
		}
		
		public boolean glyph(int glyphID, int classID) throws UnsupportedFontException, InvalidFontException {
			if (subset.getExistingSubsetGid(glyphID) != -1)
			{
				glyphFound = true;
				return false;
			}
			return true;
		}
		
		boolean classCoveredBySubset(LayoutTable origTable, int origClassDefOffset, int numGlyphs, int theClass) throws UnsupportedFontException, InvalidFontException
		{
			glyphFound = false;
			origTable.iterateClass(origClassDefOffset, numGlyphs, this, theClass);
			return glyphFound;
		}
		
	}
	
}
