/*
*
*	File: Maxp.java
*
*
*	ADOBE CONFIDENTIAL
*	___________________
*
*	Copyright 2004-2005 Adobe Systems Incorporated
*	All Rights Reserved.
*
*	NOTICE: All information contained herein is, and remains the property of
*	Adobe Systems Incorporated and its suppliers, if any. The intellectual
*	and technical concepts contained herein are proprietary to Adobe Systems
*	Incorporated and its suppliers and may be covered by U.S. and Foreign
*	Patents, patents in process, and are protected by trade secret or
*	copyright law. Dissemination of this information or reproduction of this
*	material is strictly forbidden unless prior written permission is obtained
*	from Adobe Systems Incorporated.
*
*/

package com.adobe.fontengine.font.opentype;

import java.util.Map;

import com.adobe.fontengine.font.FontByteArray;
import com.adobe.fontengine.font.InvalidFontException;
import com.adobe.fontengine.font.Subset;
import com.adobe.fontengine.font.UnsupportedFontException;
import com.adobe.fontengine.font.opentype.OTByteArray.OTByteArrayBuilder;

/** Gives access to the 'maxp' table.
 * 
 * <h4>Version handling</h4>
 * 
 * <p>'maxp' tables have a major/minor version number. This 
 * implementation:
 * <ul> 
 * <li>fully supports version 0.5 and 1.0 tables,</li>
 * <li>interprets 0.x tables with x > 5 as 0.5 tables</li>
 * <li>interprets 1.x tables as 1.0 tables,</li>
 * <li>rejects other versions with an <code>UnsupportedFontException</code> 
 * at construction time.</li>
 * </ul>
 *
 * <p>When some data is not present in the actual version of the table,
 * the accessors return identified values.</p>
 * 
 * <h4>Synchronization</h4>
 * 
 * <p>Like all tables, these objects are immutable.</p> 
 */

final public class Maxp extends Table {
    
  protected Maxp (FontByteArray buffer) 
  throws UnsupportedFontException, InvalidFontException, java.io.IOException {
    
    super (buffer);
    
    if (buffer.getSize() < 6) {
      throw new InvalidFontException ("'maxp' table must be at least 6 bytes (actual = " + buffer.getSize() + ")"); }
    
    int majorVersion = getTableMajorVersion ();
    
    switch (majorVersion) {
      case 0: {
        int minorVersion = getTableMinorVersion ();
        if (minorVersion < 0x5000) {
          throw new UnsupportedFontException ("'maxp' table version " 
              + majorVersion + "." + minorVersion + " are not supported"); }
        break; }
      
      case 1: {
        break; }
      
      default: {
        throw new UnsupportedFontException 
                   ("'maxp' table major version " 
                        + majorVersion + " are not supported"); }}
  }
  
  /** Get the major version of the table */
  public int getTableMajorVersion () throws InvalidFontException {
    return this.data.getuint16 (0);
  }
                                                                            
  /** Get the minor version of the table */
  public int getTableMinorVersion () throws InvalidFontException {
    return this.data.getuint16 (2);
  }
                                                                            
  /** Get the number of glyphs */
  public int getNumGlyphs () throws InvalidFontException {
    return this.data.getuint16 (4);
  }
   
  
  
  public int getMaxTwilightPoints () throws InvalidFontException {
    if (getTableMajorVersion () >= 1) {
      return this.data.getuint16 (16); }
    else {
      throw new InvalidFontException ("no maxTwilightPoins for fonts with maxp.major = 0"); }
  }
  
  public int getMaxStorage () throws InvalidFontException {
    if (getTableMajorVersion () >= 1) {
      return this.data.getuint16 (18); }
    else {
      throw new InvalidFontException ("no maxStorage for fonts with maxp.major = 0"); }
  }
  
  public int getMaxFunctionDefs () throws InvalidFontException {
    if (getTableMajorVersion () >= 1) {
      return this.data.getuint16 (20); }
    else {
      throw new InvalidFontException ("no maxFunctionDefs for fonts with maxp.major = 0"); }
  }
  
  public int getMaxStackElements () throws InvalidFontException {
    if (getTableMajorVersion () >= 1) {
      return this.data.getuint16 (24); }
    else {
      throw new InvalidFontException ("no maxStackElements for fonts with maxp.major = 0"); }
  }
  
  public void subsetAndStreamForCFF(int numGlyphs, Map tables)
  {
	  OTByteArrayBuilder newData = OTByteArray.getOTByteArrayBuilderInstance(6);
	  newData.setuint32(0, 0x00005000);
	  newData.setuint16(4, numGlyphs);   
	  tables.put (new Integer (Tag.table_maxp), newData); 
  }

  public void subsetAndStream (Subset subset, Map tables) {
    OTByteArrayBuilder newData = OTByteArray.getOTByteArrayBuilderInstance(this.data);
    newData.setuint16 (4, subset.getNumGlyphs ());      
    tables.put (new Integer (Tag.table_maxp), newData); 
  }  
  
  public void stream(Map tables) {
	  OTByteArrayBuilder newData = this.getDataAsByteArray();
	  tables.put (new Integer (Tag.table_maxp), newData);
  }
}
