/*
 *
 *	File: Name.java
 *
 *
 *	ADOBE CONFIDENTIAL
 *	___________________
 *
 *	Copyright 2004-2005 Adobe Systems Incorporated
 *	All Rights Reserved.
 *
 *	NOTICE: All information contained herein is, and remains the property of
 *	Adobe Systems Incorporated and its suppliers, if any. The intellectual
 *	and technical concepts contained herein are proprietary to Adobe Systems
 *	Incorporated and its suppliers and may be covered by U.S. and Foreign
 *	Patents, patents in process, and are protected by trade secret or
 *	copyright law. Dissemination of this information or reproduction of this
 *	material is strictly forbidden unless prior written permission is obtained
 *	from Adobe Systems Incorporated.
 *
 */

package com.adobe.fontengine.font.opentype;

import java.io.UnsupportedEncodingException;
import java.nio.ByteBuffer;
import java.nio.CharBuffer;
import java.nio.charset.Charset;
import java.nio.charset.CharsetDecoder;
import java.nio.charset.CoderMalfunctionError;
import java.nio.charset.CoderResult;
import java.nio.charset.MalformedInputException;
import java.nio.charset.UnmappableCharacterException;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import com.adobe.agl.util.ULocale;
import com.adobe.fontengine.CharsetUtil;
import com.adobe.fontengine.font.FontByteArray;
import com.adobe.fontengine.font.InvalidFontException;
import com.adobe.fontengine.font.Subset;
import com.adobe.fontengine.font.UnsupportedFontException;
import com.adobe.fontengine.font.opentype.Cmap.PlatformID;
import com.adobe.fontengine.font.opentype.OTByteArray.OTByteArrayBuilder;
import com.adobe.fontengine.fontmanagement.Platform;

/** Gives access to the 'name' table.
 *
 * <h4>Version handling</h4>
 * 
 * <p>'name' tables have no version.</p>
 *
 * <h4>Synchronization</h4>
 * 
 * <p>Like all tables, these objects are immutable.</p> 
 */

final public class Name extends Table 
{
	// undefined entry used for any of the identifiers
	private static final int UNDEFINED = -1;

	/** This class defines constants for the predefined entries
	 * in a 'name' table.
	 */
	public static class PredefinedNames {
		private PredefinedNames () {
			// This class is only a namespace for constants
			// and should not be instantiated
		}
		public static final int COPYRIGHT_NOTICE = 0;
		public static final int FONT_FAMILY = 1;
		public static final int FONT_SUBFAMILY = 2;
		public static final int UNIQUE_FONT_IDENTIFIER = 3;
		public static final int FULL_FONT_NAME = 4;
		public static final int VERSION_STRING = 5;
		public static final int POSTSCRIPT_NAME = 6;
		public static final int TRADEMARK = 7;
		public static final int MANUFACTURER = 8;
		public static final int DESIGNER = 9;
		public static final int DESCRIPTION = 10;
		public static final int VENDOR_URL = 11;
		public static final int DESIGNER_URL = 12;
		public static final int LICENSE_DESCRIPTION = 13;
		public static final int LICENSE_URL = 14;
		public static final int PREFERRED_FAMILY = 16;
		public static final int PREFERRED_SUBFAMILY = 17;
		public static final int COMPATIBLE_FULL_NAME = 18;
		public static final int SAMPLE_TEXT = 19;
		public static final int POSTSCRIPT_CID_NAME = 20;
	}

	/** This class defines constants for the platform ids.
	 * 
	 * This class only defines constants for the platform ids which
	 * can be used in a name tables.
	 * In OT 1.4, the page describing the 'name' table also serves
	 * to document the 'cmap' table, and lists more platform ids than
	 * are actually allowed in 'name' tables.
	 */
	final public static class PlatformId 
	{
		private PlatformId () 
		{
			// This class is only a namespace for constants
			// and should not be instantiated
		}

		public static final int UNICODE = 0;
		public static final int MACINTOSH = 1;
		public static final int MICROSOFT = 3;

		public final static int getPlatformId(Platform platform)
		{
			if (platform == Platform.MAC_OSX)
			{
				return PlatformId .MACINTOSH;
			}
			else if (platform == Platform.WINDOWS)
			{
				return PlatformId.MICROSOFT;
			}
			return Name.UNDEFINED;
		}

		public final static Platform getPlatform(int platformId)
		{
			switch (platformId)
			{
			case MACINTOSH:
				return Platform.MAC_OSX;

			case MICROSOFT:
				return Platform.WINDOWS;

			default:
				return Platform.UNKNOWN;
			}
		}
	}

	/** This class defines constants for the Microsoft encoding ids.
	 */  
	final public static class MicrosoftEncodingId 
	{
		private MicrosoftEncodingId () 
		{
			// This class is only a namespace for constants
			// and should not be instantiated
		}
		public static final int SYMBOL = 0;
		public static final int UTF16_BE_BMP_ONLY = 1;
		public static final int SHIFT_JIS = 2;
		public static final int PRC = 3;
		public static final int BIG5 = 4;
		public static final int WANSUNG = 5;
		public static final int JOHAB = 6;
		public static final int UTF16_BE_FULL = 10;
	}

	final public static class LanguageCodeEntry
	{
		private final int languageCode;
		private final String language;
		private final String script;
		private final String region;
		private final String variant;

		LanguageCodeEntry(int languageCode, String language, String script, String region) 
		{
			this(languageCode, language, script, region, null);
		}

		public LanguageCodeEntry(int languageCode, String language, String script, String region, String variant) 
		{
			this.languageCode = languageCode;
			this.language = language;
			this.script = script;
			this.region = region;
			this.variant = variant;
		}

		public int getLanguageCode() 
		{
			return languageCode;
		}

		public String getLocaleString()
		{
			return new String(
					this.language 
					+ ((this.script == null) ? "" : ("_" + this.script)) 
					+ ((this.region == null) ? "" : ("_" + this.region))
					+ ((this.variant == null) ? "" : ("_" + this.variant)));
		}

		public ULocale getLocale() 
		{
			return new ULocale(getLocaleString());
		}	
	}

	/** This class defines constants for the Microsoft language ids. */
	final public static class MicrosoftLCID 
	{
		private MicrosoftLCID() 
		{
			// This class is only a namespace for constants
			// and should not be instantiated
		}

		public static final LanguageCodeEntry getCodeEntry(int code)
		{
			return (LanguageCodeEntry) MicrosoftLCID.LCID_MAP.get(new Integer(code));
		}

		private static final Map LCID_MAP = new HashMap(250);

		public static final LanguageCodeEntry AFRIKAANS_SOUTH_AFRICA = new LanguageCodeEntry(0x0436, "af", null, "ZA");
		public static final LanguageCodeEntry ALBANIAN_ALBANIA = new LanguageCodeEntry(0x041C, "sq", null, "AL");
		public static final LanguageCodeEntry ALSATIAN_FRANCE = new LanguageCodeEntry(0x0484, "gsw", null, "FR");
		public static final LanguageCodeEntry AMHARIC_ETHIOPIA = new LanguageCodeEntry(0x045E, "am", null, "ET");
		public static final LanguageCodeEntry ARABIC_ALGERIA = new LanguageCodeEntry(0x1401, "ar", null, "DZ");
		public static final LanguageCodeEntry ARABIC_BAHRAIN = new LanguageCodeEntry(0x3C01, "ar", null, "BH");
		public static final LanguageCodeEntry ARABIC_EGYPT = new LanguageCodeEntry(0x0C01, "ar", null, "EG");
		public static final LanguageCodeEntry ARABIC_IRAQ = new LanguageCodeEntry(0x0801, "ar", null, "IQ");
		public static final LanguageCodeEntry ARABIC_JORDAN = new LanguageCodeEntry(0x2C01, "ar", null, "JO");
		public static final LanguageCodeEntry ARABIC_KUWAIT = new LanguageCodeEntry(0x3401, "ar", null, "KW");
		public static final LanguageCodeEntry ARABIC_LEBANON = new LanguageCodeEntry(0x3001, "ar", null, "LB");
		public static final LanguageCodeEntry ARABIC_LIBYA = new LanguageCodeEntry(0x1001, "ar", null, "LY");
		public static final LanguageCodeEntry ARABIC_MOROCCO = new LanguageCodeEntry(0x1801, "ar", null, "MA");
		public static final LanguageCodeEntry ARABIC_OMAN = new LanguageCodeEntry(0x2001, "ar", null, "OM");
		public static final LanguageCodeEntry ARABIC_QATAR = new LanguageCodeEntry(0x4001, "ar", null, "QA");
		public static final LanguageCodeEntry ARABIC_SAUDI_ARABIA = new LanguageCodeEntry(0x0401, "ar", null, "SA");
		public static final LanguageCodeEntry ARABIC_SYRIA = new LanguageCodeEntry(0x2801, "ar", null, "SY");
		public static final LanguageCodeEntry ARABIC_TUNISIA = new LanguageCodeEntry(0x1C01, "ar", null, "TN");
		public static final LanguageCodeEntry ARABIC_UAE = new LanguageCodeEntry(0x3801, "ar", null, "AE");
		public static final LanguageCodeEntry ARABIC_YEMEN = new LanguageCodeEntry(0x2401, "ar", null, "YE");
		public static final LanguageCodeEntry ARMENIAN_ARMENIA = new LanguageCodeEntry(0x042B, "hy", null, "AM");
		public static final LanguageCodeEntry ASSAMESE_INDIA = new LanguageCodeEntry(0x044D, "as", null, "IN");
		public static final LanguageCodeEntry AZERI_CYRILLIC_AZERBAIJAN = new LanguageCodeEntry(0x082C, "az", "Cyrl", "AZ");
		public static final LanguageCodeEntry AZERI_LATIN_AZERBAIJAN = new LanguageCodeEntry(0x042C, "az", "Latn", "AZ");
		public static final LanguageCodeEntry BASHKIR_RUSSIA = new LanguageCodeEntry(0x046D, "ba", null, "RU");
		public static final LanguageCodeEntry BASQUE_BASQUE = new LanguageCodeEntry(0x042D, "eu", null, "ES");
		public static final LanguageCodeEntry BELARUSIAN_BELARUS = new LanguageCodeEntry(0x0423, "be", null, "BY");
		public static final LanguageCodeEntry BENGALI_BANGLADESH = new LanguageCodeEntry(0x0845, "bn", null, "BD");
		public static final LanguageCodeEntry BENGALI_INDIA = new LanguageCodeEntry(0x0445, "bn", null, "IN");
		public static final LanguageCodeEntry BOSNIAN_CYRILLIC_BOSNIA_AND_HERZEGOVINA = new LanguageCodeEntry(0x201A, "bs", "Cyrl", "BA");
		public static final LanguageCodeEntry BOSNIAN_LATIN_BOSNIA_AND_HERZEGOVINA = new LanguageCodeEntry(0x141A, "bs", "Latn", "BA");
		public static final LanguageCodeEntry BRETON_FRANCE = new LanguageCodeEntry(0x047E, "br", null, "FR");
		public static final LanguageCodeEntry BULGARIAN_BULGARIA = new LanguageCodeEntry(0x0402, "bg", null, "BG");
		public static final LanguageCodeEntry CATALAN_CATALAN = new LanguageCodeEntry(0x0403, "ca", null, "ES");
		public static final LanguageCodeEntry CHINESE_HONG_KONG = new LanguageCodeEntry(0x0C04, "zh", null, "HK");
		public static final LanguageCodeEntry CHINESE_MACAO = new LanguageCodeEntry(0x1404, "zh", null, "MO");
		public static final LanguageCodeEntry CHINESE_PRC = new LanguageCodeEntry(0x0804, "zh", null, "CN");
		public static final LanguageCodeEntry CHINESE_SINGAPORE = new LanguageCodeEntry(0x1004, "zh", null, "SG");
		public static final LanguageCodeEntry CHINESE_TAIWAN = new LanguageCodeEntry(0x0404, "zh", null, "TW");
		public static final LanguageCodeEntry CORSICAN_FRANCE = new LanguageCodeEntry(0x0483, "co", null, "FR");
		public static final LanguageCodeEntry CROATIAN_CROATIA = new LanguageCodeEntry(0x041A, "hr", null, "HR");
		public static final LanguageCodeEntry CROATIAN_LATIN_BOSNIA_AND_HERZEGOVINA = new LanguageCodeEntry(0x101A, "hr", null, "BA");
		public static final LanguageCodeEntry CZECH_CZECH_REPUBLIC = new LanguageCodeEntry(0x0405, "cs", null, "CZ");
		public static final LanguageCodeEntry DANISH_DENMARK = new LanguageCodeEntry(0x0406, "da", null, "DK");
		public static final LanguageCodeEntry DARI_AFGHANISTAN = new LanguageCodeEntry(0x048C, "fa", null, "AF");
		public static final LanguageCodeEntry DIVEHI_MALDIVES = new LanguageCodeEntry(0x0465, "dv", null, "MV");
		public static final LanguageCodeEntry DUTCH_BELGIUM = new LanguageCodeEntry(0x0813, "nl", null, "BE");
		public static final LanguageCodeEntry DUTCH_NETHERLANDS = new LanguageCodeEntry(0x0413, "nl", null, "NL");
		public static final LanguageCodeEntry ENGLISH_AUSTRALIA = new LanguageCodeEntry(0x0C09, "en", null, "AU");
		public static final LanguageCodeEntry ENGLISH_BELIZE = new LanguageCodeEntry(0x2809, "en", null, "BZ");
		public static final LanguageCodeEntry ENGLISH_CANADA = new LanguageCodeEntry(0x1009, "en", null, "CA");
		public static final LanguageCodeEntry ENGLISH_CARIBBEAN = new LanguageCodeEntry(0x2409, "en", null, "029");
		public static final LanguageCodeEntry ENGLISH_INDIA = new LanguageCodeEntry(0x4009, "en", null, "IN");
		public static final LanguageCodeEntry ENGLISH_IRELAND = new LanguageCodeEntry(0x1809, "en", null, "IE");
		public static final LanguageCodeEntry ENGLISH_JAMAICA = new LanguageCodeEntry(0x2009, "en", null, "JM");
		public static final LanguageCodeEntry ENGLISH_MALAYSIA = new LanguageCodeEntry(0x4409, "en", null, "MY");
		public static final LanguageCodeEntry ENGLISH_NEW_ZEALAND = new LanguageCodeEntry(0x1409, "en", null, "NZ");
		public static final LanguageCodeEntry ENGLISH_PHILIPPINES = new LanguageCodeEntry(0x3409, "en", null, "PH");
		public static final LanguageCodeEntry ENGLISH_SINGAPORE = new LanguageCodeEntry(0x4809, "en", null, "SG");
		public static final LanguageCodeEntry ENGLISH_SOUTH_AFRICA = new LanguageCodeEntry(0x1C09, "en", null, "ZA");
		public static final LanguageCodeEntry ENGLISH_TRINIDAD_AND_TOBAGO = new LanguageCodeEntry(0x2C09, "en", null, "TT");
		public static final LanguageCodeEntry ENGLISH_UNITED_KINGDOM = new LanguageCodeEntry(0x0809, "en", null, "GB");
		public static final LanguageCodeEntry ENGLISH_UNITED_STATES = new LanguageCodeEntry(0x0409, "en", null, "US");
		public static final LanguageCodeEntry ENGLISH_ZIMBABWE = new LanguageCodeEntry(0x3009, "en", null, "ZW");
		public static final LanguageCodeEntry ESTONIAN_ESTONIA = new LanguageCodeEntry(0x0425, "et", null, "EE");
		public static final LanguageCodeEntry FAROESE_FAROE_ISLANDS = new LanguageCodeEntry(0x0438, "fo", null, "FO");
		public static final LanguageCodeEntry FILIPINO_PHILIPPINES = new LanguageCodeEntry(0x0464, "fil", null, "PH");
		public static final LanguageCodeEntry FINNISH_FINLAND = new LanguageCodeEntry(0x040B, "fil", null, "FI");
		public static final LanguageCodeEntry FRENCH_BELGIUM = new LanguageCodeEntry(0x080C, "fr", null, "BE");
		public static final LanguageCodeEntry FRENCH_CANADA = new LanguageCodeEntry(0x0C0C, "fr", null, "CA");
		public static final LanguageCodeEntry FRENCH_FRANCE = new LanguageCodeEntry(0x040C, "fr", null, "FR");
		public static final LanguageCodeEntry FRENCH_LUXEMBOURG = new LanguageCodeEntry(0x140c, "fr", null, "LU");
		public static final LanguageCodeEntry FRENCH_MONOCO = new LanguageCodeEntry(0x180C, "fr", null, "MC");
		public static final LanguageCodeEntry FRENCH_SWITZERLAND = new LanguageCodeEntry(0x100C, "fr", null, "CH");
		public static final LanguageCodeEntry FRISIAN_NETHERLANDS = new LanguageCodeEntry(0x0462, "fy", null, "NL");
		public static final LanguageCodeEntry GALICIAN_GALICIAN = new LanguageCodeEntry(0x0456, "gl", null, "ES");
		public static final LanguageCodeEntry GEORGIAN_GEORGIA = new LanguageCodeEntry(0x0437, "ka", null, "GE");
		public static final LanguageCodeEntry GERMAN_AUSTRIA = new LanguageCodeEntry(0x0C07, "de", null, "AT");
		public static final LanguageCodeEntry GERMAN_GERMANY = new LanguageCodeEntry(0x0407, "de", null, "DE");
		public static final LanguageCodeEntry GERMAN_LIECHTENSTEIN = new LanguageCodeEntry(0x1407, "de", null, "LI");
		public static final LanguageCodeEntry GERMAN_LUXEMBOURG = new LanguageCodeEntry(0x1007, "de", null, "LU");
		public static final LanguageCodeEntry GERMAN_SWITZERLAND = new LanguageCodeEntry(0x0807, "de", null, "CH");
		public static final LanguageCodeEntry GREEK_GREECE = new LanguageCodeEntry(0x0408, "el", null, "GR");
		public static final LanguageCodeEntry GREENLANDIC_GREENLAND = new LanguageCodeEntry(0x046F, "kl", null, "GL");
		public static final LanguageCodeEntry GUJARATI_INDIA = new LanguageCodeEntry(0x0447, "gu", null, "IN");
		public static final LanguageCodeEntry HAUSA_LATIN_NIGERIA = new LanguageCodeEntry(0x0468, "ha", null, "NG");
		public static final LanguageCodeEntry HEBREW_ISRAEL = new LanguageCodeEntry(0x040D, "he", null, "IL");
		public static final LanguageCodeEntry HINDI_INDIA = new LanguageCodeEntry(0x0439, "hi", null, "IN");
		public static final LanguageCodeEntry HUNGARIAN_HUNGARY = new LanguageCodeEntry(0x040E, "hu", null, "HU");
		public static final LanguageCodeEntry ICELANDIC_ICELAND = new LanguageCodeEntry(0x040F, "is", null, "IS");
		public static final LanguageCodeEntry IGBO_NIGERIA = new LanguageCodeEntry(0x0470, "ig", null, "NG");
		public static final LanguageCodeEntry INDONESIAN_INDONESIA = new LanguageCodeEntry(0x0421, "id", null, "ID");
		public static final LanguageCodeEntry INUKTITUT_CANADA = new LanguageCodeEntry(0x045D, "iu", null, "CA");
		public static final LanguageCodeEntry INUKTITUT_LATIN_CANADA = new LanguageCodeEntry(0x085D, "iu", "Latn", "CA");
		public static final LanguageCodeEntry IRISH_IRELAND = new LanguageCodeEntry(0x083C, "ga", null, "IE");
		public static final LanguageCodeEntry ISIXHOSA_SOUTH_AFRICA = new LanguageCodeEntry(0x0434, "xh", null, "ZA");
		public static final LanguageCodeEntry ISIZULU_SOUTH_AFRICA = new LanguageCodeEntry(0x0435, "zu", null, "ZA");
		public static final LanguageCodeEntry ITALIAN_ITALY = new LanguageCodeEntry(0x0410, "it", null, "IT");
		public static final LanguageCodeEntry ITALIAN_SWITZERLAND = new LanguageCodeEntry(0x0810, "it", null, "CH");
		public static final LanguageCodeEntry JAPANESE_JAPAN = new LanguageCodeEntry(0x0411, "ja", null, "JP");
		public static final LanguageCodeEntry KICHE_GUATEMALA = new LanguageCodeEntry(0x0486, "qut", null, "GT");
		public static final LanguageCodeEntry KANNADA_INDIA = new LanguageCodeEntry(0x044B, "kn", null, "IN");
		public static final LanguageCodeEntry KAZAKH_KAZAKHSTAN = new LanguageCodeEntry(0x043F, "kk", null, "KZ");
		public static final LanguageCodeEntry KHMER_CAMBODIA = new LanguageCodeEntry(0x0453, "km", null, "KH");
		public static final LanguageCodeEntry KINYARWANDA_RWANDA = new LanguageCodeEntry(0x0487, "rw", null, "RW");
		public static final LanguageCodeEntry KISWAHILI_KENYA = new LanguageCodeEntry(0x0441, "sw", null, "KE");
		public static final LanguageCodeEntry KONKANI_INDIA = new LanguageCodeEntry(0x0457, "kok", null, "IN");
		public static final LanguageCodeEntry KOREAN_KOREA = new LanguageCodeEntry(0x0412, "ko", null, "KR");
		public static final LanguageCodeEntry KYRGYZ_KYRGYZSTAN = new LanguageCodeEntry(0x0440, "ky", null, "KG");
		public static final LanguageCodeEntry LAO_LAOS = new LanguageCodeEntry(0x0454, "lo", null, "LA");
		public static final LanguageCodeEntry LATVIAN_LATVIA = new LanguageCodeEntry(0x0426, "lv", null, "LV");
		public static final LanguageCodeEntry LITHUANIAN_LITHUANIA = new LanguageCodeEntry(0x0427, "lt", null, "LT");
		public static final LanguageCodeEntry LOWER_SORBIAN_GERMANY = new LanguageCodeEntry(0x082E, "dsb", null, "DE");
		public static final LanguageCodeEntry LUXEMBOURGISH_LUXEMBOURG = new LanguageCodeEntry(0x046E, "lb", null, "LU");
		public static final LanguageCodeEntry MACEDONIAN_MACEDONIA = new LanguageCodeEntry(0x042F, "mk", null, "MK");
		public static final LanguageCodeEntry MALAY_BRUNEI = new LanguageCodeEntry(0x083E, "ms", null, "BN");
		public static final LanguageCodeEntry MALAY_MALAYSIA = new LanguageCodeEntry(0x043E, "ms", null, "MY");
		public static final LanguageCodeEntry MALAYALAM_INDIA = new LanguageCodeEntry(0x044C, "ml", null, "IN");
		public static final LanguageCodeEntry MALTESE_MALTA = new LanguageCodeEntry(0x043A, "mt", null, "MT");
		public static final LanguageCodeEntry MAORI_NEW_ZEALAND = new LanguageCodeEntry(0x0481, "mi", null, "NZ");
		public static final LanguageCodeEntry MAPUDUNGUN_CHILE = new LanguageCodeEntry(0x047A, "arn", null, "CL");
		public static final LanguageCodeEntry MARATHI_INDIA = new LanguageCodeEntry(0x044E, "mr", null, "IN");
		public static final LanguageCodeEntry MOHAWK_MOHAWK = new LanguageCodeEntry(0x047C, "moh", null, "CA");
		public static final LanguageCodeEntry MONGOLIAN_CYRILLIC_MONGOLIA = new LanguageCodeEntry(0x0450, "mn", null, "MN");
		public static final LanguageCodeEntry MONGOLIAN_TRADITIONAL_PRC = new LanguageCodeEntry(0x0850, "mn", null, "CN");
		public static final LanguageCodeEntry NEPALI_NEPAL = new LanguageCodeEntry(0x0461, "ne", null, "NP");
		public static final LanguageCodeEntry NORWEGIAN_BOKMAL_NORWAY = new LanguageCodeEntry(0x0414, "nb", null, "NO");
		public static final LanguageCodeEntry NORWEGIAN_NYNORSK_NORWAY = new LanguageCodeEntry(0x0814, "nn", null, "NO");
		public static final LanguageCodeEntry OCCITAN_FRANCE = new LanguageCodeEntry(0x0482, "oc", null, "FR");
		public static final LanguageCodeEntry ORIYA_INDIA = new LanguageCodeEntry(0x0448, "or", null, "IN");
		public static final LanguageCodeEntry PASHTO_AFGHANISTAN = new LanguageCodeEntry(0x0463, "ps", null, "AF");
		public static final LanguageCodeEntry POLISH_POLAND = new LanguageCodeEntry(0x0415, "pl", null, "PL");
		public static final LanguageCodeEntry PORTUGUESE_BRAZIL = new LanguageCodeEntry(0x0416, "pt", null, "BR");
		public static final LanguageCodeEntry PORTUGUESE_PORTUGAL = new LanguageCodeEntry(0x0816, "pt", null, "PT");
		public static final LanguageCodeEntry PUNJABI_INDIA = new LanguageCodeEntry(0x0446, "pa", null, "IN");
		public static final LanguageCodeEntry QUECHUA_BOLIVIA = new LanguageCodeEntry(0x046B, "qu", null, "BO");
		public static final LanguageCodeEntry QUECHUA_ECUADOR = new LanguageCodeEntry(0x086B, "qu", null, "EC");
		public static final LanguageCodeEntry QUECHUA_PERU = new LanguageCodeEntry(0x0C6B, "qu", null, "PE");
		public static final LanguageCodeEntry ROMANIAN_ROMANIA = new LanguageCodeEntry(0x0418, "ro", null, "RO");
		public static final LanguageCodeEntry ROMANSH_SWITZERLAND = new LanguageCodeEntry(0x0417, "rm", null, "CH");
		public static final LanguageCodeEntry RUSSIAN_RUSSIA = new LanguageCodeEntry(0x0419, "ru", null, "RU");
		public static final LanguageCodeEntry SAMI_INARI_FINLAND = new LanguageCodeEntry(0x243B, "smn", null, "FI");
		public static final LanguageCodeEntry SAMI_LULE_NORWAY = new LanguageCodeEntry(0x103B, "smj", null, "NO");
		public static final LanguageCodeEntry SAMI_LULE_SWEDEN = new LanguageCodeEntry(0x143B, "smj", null, "SE");
		public static final LanguageCodeEntry SAMI_NORTHERN_FINLAND = new LanguageCodeEntry(0x0C3B, "se", null, "FI");
		public static final LanguageCodeEntry SAMI_NORTHERN_NORWAY = new LanguageCodeEntry(0x043B, "se", null, "NO");
		public static final LanguageCodeEntry SAMI_NORTHERN_SWEDEN = new LanguageCodeEntry(0x083B, "se", null, "SE");
		public static final LanguageCodeEntry SAMI_SKOLT_FINLAND = new LanguageCodeEntry(0x203B, "sms", null, "FI");
		public static final LanguageCodeEntry SAMI_SOUTHERN_NORWAY = new LanguageCodeEntry(0x183B, "sma", null, "NO");
		public static final LanguageCodeEntry SAMI_SOUTHERN_SWEDEN = new LanguageCodeEntry(0x1C3B, "sma", null, "SE");
		public static final LanguageCodeEntry SANSKRIT_INDIA = new LanguageCodeEntry(0x044F, "sa", null, "IN");
		public static final LanguageCodeEntry SERBIAN_CYRILLIC_BOSNIA_AND_HERZEGOVINA = new LanguageCodeEntry(0x1C1A, "sr", "Cyrl", "BA");
		public static final LanguageCodeEntry SERBIAN_CYRILLIC_SERBIA = new LanguageCodeEntry(0x0C1A, "sr", "Cyrl", "RS");
		public static final LanguageCodeEntry SERBIAN_LATIN_BOSNIA_AND_HERZEGOVINA = new LanguageCodeEntry(0x181A, "sr", "Latn", "BA");
		public static final LanguageCodeEntry SERBIAN_LATIN_SERBIA = new LanguageCodeEntry(0x081A, "sr", "Latn", "RS");
		public static final LanguageCodeEntry SESOTHO_SA_LEBOA_SOUTH_AFRICA = new LanguageCodeEntry(0x046C, "nso", null, "ZA");
		public static final LanguageCodeEntry SETSWANA_SOUTH_AFRICA = new LanguageCodeEntry(0x0432, "tn", null, "ZA");
		public static final LanguageCodeEntry SINHALA_SRI_LANKA = new LanguageCodeEntry(0x045B, "si", null, "LK");
		public static final LanguageCodeEntry SLOVAK_SLOVAKIA = new LanguageCodeEntry(0x041B, "sk", null, "SK");
		public static final LanguageCodeEntry SLOVENIAN_SLOVENIA = new LanguageCodeEntry(0x0424, "sl", null, "SI");
		public static final LanguageCodeEntry SPANISH_ARGENTINA = new LanguageCodeEntry(0x2C0A, "es", null, "AR");
		public static final LanguageCodeEntry SPANISH_BOLIVIA = new LanguageCodeEntry(0x400A, "es", null, "BO");
		public static final LanguageCodeEntry SPANISH_CHILE = new LanguageCodeEntry(0x340A, "es", null, "CL");
		public static final LanguageCodeEntry SPANISH_COLOMBIA = new LanguageCodeEntry(0x240A, "es", null, "CO");
		public static final LanguageCodeEntry SPANISH_COSTA_RICA = new LanguageCodeEntry(0x140A, "es", null, "CR");
		public static final LanguageCodeEntry SPANISH_DOMINICAN_REPUBLIC = new LanguageCodeEntry(0x1C0A, "es", null, "DO");
		public static final LanguageCodeEntry SPANISH_ECUADOR = new LanguageCodeEntry(0x300A, "es", null, "EC");
		public static final LanguageCodeEntry SPANISH_EL_SALVADOR = new LanguageCodeEntry(0x440A, "es", null, "SV");
		public static final LanguageCodeEntry SPANISH_GUATEMALA = new LanguageCodeEntry(0x100A, "es", null, "GT");
		public static final LanguageCodeEntry SPANISH_HONDURAS = new LanguageCodeEntry(0x480A, "es", null, "HN");
		public static final LanguageCodeEntry SPANISH_MEXICO = new LanguageCodeEntry(0x080A, "es", null, "MX");
		public static final LanguageCodeEntry SPANISH_NICARAGUA = new LanguageCodeEntry(0x4C0A, "es", null, "NI");
		public static final LanguageCodeEntry SPANISH_PANAMA = new LanguageCodeEntry(0x180A, "es", null, "PA");
		public static final LanguageCodeEntry SPANISH_PARAGUAY = new LanguageCodeEntry(0x3C0A, "es", null, "PY");
		public static final LanguageCodeEntry SPANISH_PERU = new LanguageCodeEntry(0x280A, "es", null, "PE");
		public static final LanguageCodeEntry SPANISH_PUERTO_RICO = new LanguageCodeEntry(0x500A, "es", null, "PR");
		public static final LanguageCodeEntry SPANISH_SPAIN = new LanguageCodeEntry(0x0c0a, "es", null, "ES");
		public static final LanguageCodeEntry SPANISH_SPAIN_TRADITIONAL = new LanguageCodeEntry(0x040a, "es", null, "ES", "TRADITIONAL");
		public static final LanguageCodeEntry SPANISH_UNITED_STATES = new LanguageCodeEntry(0x540A, "es", null, "US");
		public static final LanguageCodeEntry SPANISH_URUGUAY = new LanguageCodeEntry(0x380A, "es", null, "UY");
		public static final LanguageCodeEntry SPANISH_VENEZUELA = new LanguageCodeEntry(0x200A, "es", null, "VE");
		public static final LanguageCodeEntry SWEDISH_FINLAND = new LanguageCodeEntry(0x081D, "sv", null, "FI");
		public static final LanguageCodeEntry SWEDISH_SWEDEN = new LanguageCodeEntry(0x041D, "sv", null, "SE");
		public static final LanguageCodeEntry SYRIAC_SYRIA = new LanguageCodeEntry(0x045A, "syr", null, "SY");
		public static final LanguageCodeEntry TAJIK_CYRILLIC_TAJIKISTAN = new LanguageCodeEntry(0x0428, "tg", "Cyrl", "TJ");
		public static final LanguageCodeEntry TAMAZIGHT_LATIN_ALGERIA = new LanguageCodeEntry(0x085F, "ber", "Latn", "DZ");
		public static final LanguageCodeEntry TAMIL_INDIA = new LanguageCodeEntry(0x0449, "ta", null, "IN");
		public static final LanguageCodeEntry TATAR_RUSSIA = new LanguageCodeEntry(0x0444, "tt", null, "RU");
		public static final LanguageCodeEntry TELUGU_INDIA = new LanguageCodeEntry(0x044A, "te", null, "IN");
		public static final LanguageCodeEntry THAI_THAILAND = new LanguageCodeEntry(0x041E, "th", null, "TH");
		public static final LanguageCodeEntry TIBETAN_PRC = new LanguageCodeEntry(0x0451, "bo", null, "CN");
		public static final LanguageCodeEntry TURKISH_TURKEY = new LanguageCodeEntry(0x041F, "tr", null, "TR");
		public static final LanguageCodeEntry TURKMEN_TURKMENISTAN = new LanguageCodeEntry(0x0442, "tk", null, "TM");
		public static final LanguageCodeEntry UIGHUR_PRC = new LanguageCodeEntry(0x0480, "ug", null, "CN");
		public static final LanguageCodeEntry UKRAINIAN_UKRAINE = new LanguageCodeEntry(0x0422, "uk", null, "UA");
		public static final LanguageCodeEntry UPPER_SORBIAN_GERMANY = new LanguageCodeEntry(0x042E, "wen", null, "DE");
		public static final LanguageCodeEntry URDU_PAKISTAN = new LanguageCodeEntry(0x0420, "ur", null, "PK");
		public static final LanguageCodeEntry UZBEK_CYRILLIC_UZBEKISTAN = new LanguageCodeEntry(0x0843, "uz", "Cyrl", "UZ");
		public static final LanguageCodeEntry UZBEK_LATIN_UZBEKISTAN = new LanguageCodeEntry(0x0443, "uz", "Latn", "UZ");
		public static final LanguageCodeEntry VIETNAMESE_VIETNAM = new LanguageCodeEntry(0x042A, "vi", null, "VN");
		public static final LanguageCodeEntry WELSH_UNITED_KINGDOM = new LanguageCodeEntry(0x0452, "cy", null, "GB");
		public static final LanguageCodeEntry WOLOF_SENEGAL = new LanguageCodeEntry(0x0448, "wo", null, "SN");
		public static final LanguageCodeEntry YAKUT_RUSSIA = new LanguageCodeEntry(0x0485, "sah", null, "RU");

		static 
		{
			LCID_MAP.put(new Integer(MicrosoftLCID.AFRIKAANS_SOUTH_AFRICA.getLanguageCode()), MicrosoftLCID.AFRIKAANS_SOUTH_AFRICA);
			LCID_MAP.put(new Integer(MicrosoftLCID.ALBANIAN_ALBANIA.getLanguageCode()), MicrosoftLCID.ALBANIAN_ALBANIA);
			LCID_MAP.put(new Integer(MicrosoftLCID.ALSATIAN_FRANCE.getLanguageCode()), MicrosoftLCID.ALSATIAN_FRANCE);
			LCID_MAP.put(new Integer(MicrosoftLCID.AMHARIC_ETHIOPIA.getLanguageCode()), MicrosoftLCID.AMHARIC_ETHIOPIA);
			LCID_MAP.put(new Integer(MicrosoftLCID.ARABIC_ALGERIA.getLanguageCode()), MicrosoftLCID.ARABIC_ALGERIA);
			LCID_MAP.put(new Integer(MicrosoftLCID.ARABIC_BAHRAIN.getLanguageCode()), MicrosoftLCID.ARABIC_BAHRAIN);
			LCID_MAP.put(new Integer(MicrosoftLCID.ARABIC_EGYPT.getLanguageCode()), MicrosoftLCID.ARABIC_EGYPT);
			LCID_MAP.put(new Integer(MicrosoftLCID.ARABIC_IRAQ.getLanguageCode()), MicrosoftLCID.ARABIC_IRAQ);
			LCID_MAP.put(new Integer(MicrosoftLCID.ARABIC_JORDAN.getLanguageCode()), MicrosoftLCID.ARABIC_JORDAN);
			LCID_MAP.put(new Integer(MicrosoftLCID.ARABIC_KUWAIT.getLanguageCode()), MicrosoftLCID.ARABIC_KUWAIT);
			LCID_MAP.put(new Integer(MicrosoftLCID.ARABIC_LEBANON.getLanguageCode()), MicrosoftLCID.ARABIC_LEBANON);
			LCID_MAP.put(new Integer(MicrosoftLCID.ARABIC_LIBYA.getLanguageCode()), MicrosoftLCID.ARABIC_LIBYA);
			LCID_MAP.put(new Integer(MicrosoftLCID.ARABIC_MOROCCO.getLanguageCode()), MicrosoftLCID.ARABIC_MOROCCO);
			LCID_MAP.put(new Integer(MicrosoftLCID.ARABIC_OMAN.getLanguageCode()), MicrosoftLCID.ARABIC_OMAN);
			LCID_MAP.put(new Integer(MicrosoftLCID.ARABIC_QATAR.getLanguageCode()), MicrosoftLCID.ARABIC_QATAR);
			LCID_MAP.put(new Integer(MicrosoftLCID.ARABIC_SAUDI_ARABIA.getLanguageCode()), MicrosoftLCID.ARABIC_SAUDI_ARABIA);
			LCID_MAP.put(new Integer(MicrosoftLCID.ARABIC_SYRIA.getLanguageCode()), MicrosoftLCID.ARABIC_SYRIA);
			LCID_MAP.put(new Integer(MicrosoftLCID.ARABIC_TUNISIA.getLanguageCode()), MicrosoftLCID.ARABIC_TUNISIA);
			LCID_MAP.put(new Integer(MicrosoftLCID.ARABIC_UAE.getLanguageCode()), MicrosoftLCID.ARABIC_UAE);
			LCID_MAP.put(new Integer(MicrosoftLCID.ARABIC_YEMEN.getLanguageCode()), MicrosoftLCID.ARABIC_YEMEN);
			LCID_MAP.put(new Integer(MicrosoftLCID.ARMENIAN_ARMENIA.getLanguageCode()), MicrosoftLCID.ARMENIAN_ARMENIA);
			LCID_MAP.put(new Integer(MicrosoftLCID.ASSAMESE_INDIA.getLanguageCode()), MicrosoftLCID.ASSAMESE_INDIA);
			LCID_MAP.put(new Integer(MicrosoftLCID.AZERI_CYRILLIC_AZERBAIJAN.getLanguageCode()), MicrosoftLCID.AZERI_CYRILLIC_AZERBAIJAN);
			LCID_MAP.put(new Integer(MicrosoftLCID.AZERI_LATIN_AZERBAIJAN.getLanguageCode()), MicrosoftLCID.AZERI_LATIN_AZERBAIJAN);
			LCID_MAP.put(new Integer(MicrosoftLCID.BASHKIR_RUSSIA.getLanguageCode()), MicrosoftLCID.BASHKIR_RUSSIA);
			LCID_MAP.put(new Integer(MicrosoftLCID.BASQUE_BASQUE.getLanguageCode()), MicrosoftLCID.BASQUE_BASQUE);
			LCID_MAP.put(new Integer(MicrosoftLCID.BELARUSIAN_BELARUS.getLanguageCode()), MicrosoftLCID.BELARUSIAN_BELARUS);
			LCID_MAP.put(new Integer(MicrosoftLCID.BENGALI_BANGLADESH.getLanguageCode()), MicrosoftLCID.BENGALI_BANGLADESH);
			LCID_MAP.put(new Integer(MicrosoftLCID.BENGALI_INDIA.getLanguageCode()), MicrosoftLCID.BENGALI_INDIA);
			LCID_MAP.put(new Integer(MicrosoftLCID.BOSNIAN_CYRILLIC_BOSNIA_AND_HERZEGOVINA.getLanguageCode()), MicrosoftLCID.BOSNIAN_CYRILLIC_BOSNIA_AND_HERZEGOVINA);
			LCID_MAP.put(new Integer(MicrosoftLCID.BOSNIAN_LATIN_BOSNIA_AND_HERZEGOVINA.getLanguageCode()), MicrosoftLCID.BOSNIAN_LATIN_BOSNIA_AND_HERZEGOVINA);
			LCID_MAP.put(new Integer(MicrosoftLCID.BRETON_FRANCE.getLanguageCode()), MicrosoftLCID.BRETON_FRANCE);
			LCID_MAP.put(new Integer(MicrosoftLCID.BULGARIAN_BULGARIA.getLanguageCode()), MicrosoftLCID.BULGARIAN_BULGARIA);
			LCID_MAP.put(new Integer(MicrosoftLCID.CATALAN_CATALAN.getLanguageCode()), MicrosoftLCID.CATALAN_CATALAN);
			LCID_MAP.put(new Integer(MicrosoftLCID.CHINESE_HONG_KONG.getLanguageCode()), MicrosoftLCID.CHINESE_HONG_KONG);
			LCID_MAP.put(new Integer(MicrosoftLCID.CHINESE_MACAO.getLanguageCode()), MicrosoftLCID.CHINESE_MACAO);
			LCID_MAP.put(new Integer(MicrosoftLCID.CHINESE_PRC.getLanguageCode()), MicrosoftLCID.CHINESE_PRC);
			LCID_MAP.put(new Integer(MicrosoftLCID.CHINESE_SINGAPORE.getLanguageCode()), MicrosoftLCID.CHINESE_SINGAPORE);
			LCID_MAP.put(new Integer(MicrosoftLCID.CHINESE_TAIWAN.getLanguageCode()), MicrosoftLCID.CHINESE_TAIWAN);
			LCID_MAP.put(new Integer(MicrosoftLCID.CORSICAN_FRANCE.getLanguageCode()), MicrosoftLCID.CORSICAN_FRANCE);
			LCID_MAP.put(new Integer(MicrosoftLCID.CROATIAN_CROATIA.getLanguageCode()), MicrosoftLCID.CROATIAN_CROATIA);
			LCID_MAP.put(new Integer(MicrosoftLCID.CROATIAN_LATIN_BOSNIA_AND_HERZEGOVINA.getLanguageCode()), MicrosoftLCID.CROATIAN_LATIN_BOSNIA_AND_HERZEGOVINA);
			LCID_MAP.put(new Integer(MicrosoftLCID.CZECH_CZECH_REPUBLIC.getLanguageCode()), MicrosoftLCID.CZECH_CZECH_REPUBLIC);
			LCID_MAP.put(new Integer(MicrosoftLCID.DANISH_DENMARK.getLanguageCode()), MicrosoftLCID.DANISH_DENMARK);
			LCID_MAP.put(new Integer(MicrosoftLCID.DARI_AFGHANISTAN.getLanguageCode()), MicrosoftLCID.DARI_AFGHANISTAN);
			LCID_MAP.put(new Integer(MicrosoftLCID.DIVEHI_MALDIVES.getLanguageCode()), MicrosoftLCID.DIVEHI_MALDIVES);
			LCID_MAP.put(new Integer(MicrosoftLCID.DUTCH_BELGIUM.getLanguageCode()), MicrosoftLCID.DUTCH_BELGIUM);
			LCID_MAP.put(new Integer(MicrosoftLCID.DUTCH_NETHERLANDS.getLanguageCode()), MicrosoftLCID.DUTCH_NETHERLANDS);
			LCID_MAP.put(new Integer(MicrosoftLCID.ENGLISH_AUSTRALIA.getLanguageCode()), MicrosoftLCID.ENGLISH_AUSTRALIA);
			LCID_MAP.put(new Integer(MicrosoftLCID.ENGLISH_BELIZE.getLanguageCode()), MicrosoftLCID.ENGLISH_BELIZE);
			LCID_MAP.put(new Integer(MicrosoftLCID.ENGLISH_CANADA.getLanguageCode()), MicrosoftLCID.ENGLISH_CANADA);
			LCID_MAP.put(new Integer(MicrosoftLCID.ENGLISH_CARIBBEAN.getLanguageCode()), MicrosoftLCID.ENGLISH_CARIBBEAN);
			LCID_MAP.put(new Integer(MicrosoftLCID.ENGLISH_INDIA.getLanguageCode()), MicrosoftLCID.ENGLISH_INDIA);
			LCID_MAP.put(new Integer(MicrosoftLCID.ENGLISH_IRELAND.getLanguageCode()), MicrosoftLCID.ENGLISH_IRELAND);
			LCID_MAP.put(new Integer(MicrosoftLCID.ENGLISH_JAMAICA.getLanguageCode()), MicrosoftLCID.ENGLISH_JAMAICA);
			LCID_MAP.put(new Integer(MicrosoftLCID.ENGLISH_MALAYSIA.getLanguageCode()), MicrosoftLCID.ENGLISH_MALAYSIA);
			LCID_MAP.put(new Integer(MicrosoftLCID.ENGLISH_NEW_ZEALAND.getLanguageCode()), MicrosoftLCID.ENGLISH_NEW_ZEALAND);
			LCID_MAP.put(new Integer(MicrosoftLCID.ENGLISH_PHILIPPINES.getLanguageCode()), MicrosoftLCID.ENGLISH_PHILIPPINES);
			LCID_MAP.put(new Integer(MicrosoftLCID.ENGLISH_SINGAPORE.getLanguageCode()), MicrosoftLCID.ENGLISH_SINGAPORE);
			LCID_MAP.put(new Integer(MicrosoftLCID.ENGLISH_SOUTH_AFRICA.getLanguageCode()), MicrosoftLCID.ENGLISH_SOUTH_AFRICA);
			LCID_MAP.put(new Integer(MicrosoftLCID.ENGLISH_TRINIDAD_AND_TOBAGO.getLanguageCode()), MicrosoftLCID.ENGLISH_TRINIDAD_AND_TOBAGO);
			LCID_MAP.put(new Integer(MicrosoftLCID.ENGLISH_UNITED_KINGDOM.getLanguageCode()), MicrosoftLCID.ENGLISH_UNITED_KINGDOM);
			LCID_MAP.put(new Integer(MicrosoftLCID.ENGLISH_UNITED_STATES.getLanguageCode()), MicrosoftLCID.ENGLISH_UNITED_STATES);
			LCID_MAP.put(new Integer(MicrosoftLCID.ENGLISH_ZIMBABWE.getLanguageCode()), MicrosoftLCID.ENGLISH_ZIMBABWE);
			LCID_MAP.put(new Integer(MicrosoftLCID.ESTONIAN_ESTONIA.getLanguageCode()), MicrosoftLCID.ESTONIAN_ESTONIA);
			LCID_MAP.put(new Integer(MicrosoftLCID.FAROESE_FAROE_ISLANDS.getLanguageCode()), MicrosoftLCID.FAROESE_FAROE_ISLANDS);
			LCID_MAP.put(new Integer(MicrosoftLCID.FILIPINO_PHILIPPINES.getLanguageCode()), MicrosoftLCID.FILIPINO_PHILIPPINES);
			LCID_MAP.put(new Integer(MicrosoftLCID.FINNISH_FINLAND.getLanguageCode()), MicrosoftLCID.FINNISH_FINLAND);
			LCID_MAP.put(new Integer(MicrosoftLCID.FRENCH_BELGIUM.getLanguageCode()), MicrosoftLCID.FRENCH_BELGIUM);
			LCID_MAP.put(new Integer(MicrosoftLCID.FRENCH_CANADA.getLanguageCode()), MicrosoftLCID.FRENCH_CANADA);
			LCID_MAP.put(new Integer(MicrosoftLCID.FRENCH_FRANCE.getLanguageCode()), MicrosoftLCID.FRENCH_FRANCE);
			LCID_MAP.put(new Integer(MicrosoftLCID.FRENCH_LUXEMBOURG.getLanguageCode()), MicrosoftLCID.FRENCH_LUXEMBOURG);
			LCID_MAP.put(new Integer(MicrosoftLCID.FRENCH_MONOCO.getLanguageCode()), MicrosoftLCID.FRENCH_MONOCO);
			LCID_MAP.put(new Integer(MicrosoftLCID.FRENCH_SWITZERLAND.getLanguageCode()), MicrosoftLCID.FRENCH_SWITZERLAND);
			LCID_MAP.put(new Integer(MicrosoftLCID.FRISIAN_NETHERLANDS.getLanguageCode()), MicrosoftLCID.FRISIAN_NETHERLANDS);
			LCID_MAP.put(new Integer(MicrosoftLCID.GALICIAN_GALICIAN.getLanguageCode()), MicrosoftLCID.GALICIAN_GALICIAN);
			LCID_MAP.put(new Integer(MicrosoftLCID.GEORGIAN_GEORGIA.getLanguageCode()), MicrosoftLCID.GEORGIAN_GEORGIA);
			LCID_MAP.put(new Integer(MicrosoftLCID.GERMAN_AUSTRIA.getLanguageCode()), MicrosoftLCID.GERMAN_AUSTRIA);
			LCID_MAP.put(new Integer(MicrosoftLCID.GERMAN_GERMANY.getLanguageCode()), MicrosoftLCID.GERMAN_GERMANY);
			LCID_MAP.put(new Integer(MicrosoftLCID.GERMAN_LIECHTENSTEIN.getLanguageCode()), MicrosoftLCID.GERMAN_LIECHTENSTEIN);
			LCID_MAP.put(new Integer(MicrosoftLCID.GERMAN_LUXEMBOURG.getLanguageCode()), MicrosoftLCID.GERMAN_LUXEMBOURG);
			LCID_MAP.put(new Integer(MicrosoftLCID.GERMAN_SWITZERLAND.getLanguageCode()), MicrosoftLCID.GERMAN_SWITZERLAND);
			LCID_MAP.put(new Integer(MicrosoftLCID.GREEK_GREECE.getLanguageCode()), MicrosoftLCID.GREEK_GREECE);
			LCID_MAP.put(new Integer(MicrosoftLCID.GREENLANDIC_GREENLAND.getLanguageCode()), MicrosoftLCID.GREENLANDIC_GREENLAND);
			LCID_MAP.put(new Integer(MicrosoftLCID.GUJARATI_INDIA.getLanguageCode()), MicrosoftLCID.GUJARATI_INDIA);
			LCID_MAP.put(new Integer(MicrosoftLCID.HAUSA_LATIN_NIGERIA.getLanguageCode()), MicrosoftLCID.HAUSA_LATIN_NIGERIA);
			LCID_MAP.put(new Integer(MicrosoftLCID.HEBREW_ISRAEL.getLanguageCode()), MicrosoftLCID.HEBREW_ISRAEL);
			LCID_MAP.put(new Integer(MicrosoftLCID.HINDI_INDIA.getLanguageCode()), MicrosoftLCID.HINDI_INDIA);
			LCID_MAP.put(new Integer(MicrosoftLCID.HUNGARIAN_HUNGARY.getLanguageCode()), MicrosoftLCID.HUNGARIAN_HUNGARY);
			LCID_MAP.put(new Integer(MicrosoftLCID.ICELANDIC_ICELAND.getLanguageCode()), MicrosoftLCID.ICELANDIC_ICELAND);
			LCID_MAP.put(new Integer(MicrosoftLCID.IGBO_NIGERIA.getLanguageCode()), MicrosoftLCID.IGBO_NIGERIA);
			LCID_MAP.put(new Integer(MicrosoftLCID.INDONESIAN_INDONESIA.getLanguageCode()), MicrosoftLCID.INDONESIAN_INDONESIA);
			LCID_MAP.put(new Integer(MicrosoftLCID.INUKTITUT_CANADA.getLanguageCode()), MicrosoftLCID.INUKTITUT_CANADA);
			LCID_MAP.put(new Integer(MicrosoftLCID.INUKTITUT_LATIN_CANADA.getLanguageCode()), MicrosoftLCID.INUKTITUT_LATIN_CANADA);
			LCID_MAP.put(new Integer(MicrosoftLCID.IRISH_IRELAND.getLanguageCode()), MicrosoftLCID.IRISH_IRELAND);
			LCID_MAP.put(new Integer(MicrosoftLCID.ISIXHOSA_SOUTH_AFRICA.getLanguageCode()), MicrosoftLCID.ISIXHOSA_SOUTH_AFRICA);
			LCID_MAP.put(new Integer(MicrosoftLCID.ISIZULU_SOUTH_AFRICA.getLanguageCode()), MicrosoftLCID.ISIZULU_SOUTH_AFRICA);
			LCID_MAP.put(new Integer(MicrosoftLCID.ITALIAN_ITALY.getLanguageCode()), MicrosoftLCID.ITALIAN_ITALY);
			LCID_MAP.put(new Integer(MicrosoftLCID.ITALIAN_SWITZERLAND.getLanguageCode()), MicrosoftLCID.ITALIAN_SWITZERLAND);
			LCID_MAP.put(new Integer(MicrosoftLCID.JAPANESE_JAPAN.getLanguageCode()), MicrosoftLCID.JAPANESE_JAPAN);
			LCID_MAP.put(new Integer(MicrosoftLCID.KICHE_GUATEMALA.getLanguageCode()), MicrosoftLCID.KICHE_GUATEMALA);
			LCID_MAP.put(new Integer(MicrosoftLCID.KANNADA_INDIA.getLanguageCode()), MicrosoftLCID.KANNADA_INDIA);
			LCID_MAP.put(new Integer(MicrosoftLCID.KAZAKH_KAZAKHSTAN.getLanguageCode()), MicrosoftLCID.KAZAKH_KAZAKHSTAN);
			LCID_MAP.put(new Integer(MicrosoftLCID.KHMER_CAMBODIA.getLanguageCode()), MicrosoftLCID.KHMER_CAMBODIA);
			LCID_MAP.put(new Integer(MicrosoftLCID.KINYARWANDA_RWANDA.getLanguageCode()), MicrosoftLCID.KINYARWANDA_RWANDA);
			LCID_MAP.put(new Integer(MicrosoftLCID.KISWAHILI_KENYA.getLanguageCode()), MicrosoftLCID.KISWAHILI_KENYA);
			LCID_MAP.put(new Integer(MicrosoftLCID.KONKANI_INDIA.getLanguageCode()), MicrosoftLCID.KONKANI_INDIA);
			LCID_MAP.put(new Integer(MicrosoftLCID.KOREAN_KOREA.getLanguageCode()), MicrosoftLCID.KOREAN_KOREA);
			LCID_MAP.put(new Integer(MicrosoftLCID.KYRGYZ_KYRGYZSTAN.getLanguageCode()), MicrosoftLCID.KYRGYZ_KYRGYZSTAN);
			LCID_MAP.put(new Integer(MicrosoftLCID.LAO_LAOS.getLanguageCode()), MicrosoftLCID.LAO_LAOS);
			LCID_MAP.put(new Integer(MicrosoftLCID.LATVIAN_LATVIA.getLanguageCode()), MicrosoftLCID.LATVIAN_LATVIA);
			LCID_MAP.put(new Integer(MicrosoftLCID.LITHUANIAN_LITHUANIA.getLanguageCode()), MicrosoftLCID.LITHUANIAN_LITHUANIA);
			LCID_MAP.put(new Integer(MicrosoftLCID.LOWER_SORBIAN_GERMANY.getLanguageCode()), MicrosoftLCID.LOWER_SORBIAN_GERMANY);
			LCID_MAP.put(new Integer(MicrosoftLCID.LUXEMBOURGISH_LUXEMBOURG.getLanguageCode()), MicrosoftLCID.LUXEMBOURGISH_LUXEMBOURG);
			LCID_MAP.put(new Integer(MicrosoftLCID.MACEDONIAN_MACEDONIA.getLanguageCode()), MicrosoftLCID.MACEDONIAN_MACEDONIA);
			LCID_MAP.put(new Integer(MicrosoftLCID.MALAY_BRUNEI.getLanguageCode()), MicrosoftLCID.MALAY_BRUNEI);
			LCID_MAP.put(new Integer(MicrosoftLCID.MALAY_MALAYSIA.getLanguageCode()), MicrosoftLCID.MALAY_MALAYSIA);
			LCID_MAP.put(new Integer(MicrosoftLCID.MALAYALAM_INDIA.getLanguageCode()), MicrosoftLCID.MALAYALAM_INDIA);
			LCID_MAP.put(new Integer(MicrosoftLCID.MALTESE_MALTA.getLanguageCode()), MicrosoftLCID.MALTESE_MALTA);
			LCID_MAP.put(new Integer(MicrosoftLCID.MAORI_NEW_ZEALAND.getLanguageCode()), MicrosoftLCID.MAORI_NEW_ZEALAND);
			LCID_MAP.put(new Integer(MicrosoftLCID.MAPUDUNGUN_CHILE.getLanguageCode()), MicrosoftLCID.MAPUDUNGUN_CHILE);
			LCID_MAP.put(new Integer(MicrosoftLCID.MARATHI_INDIA.getLanguageCode()), MicrosoftLCID.MARATHI_INDIA);
			LCID_MAP.put(new Integer(MicrosoftLCID.MOHAWK_MOHAWK.getLanguageCode()), MicrosoftLCID.MOHAWK_MOHAWK);
			LCID_MAP.put(new Integer(MicrosoftLCID.MONGOLIAN_CYRILLIC_MONGOLIA.getLanguageCode()), MicrosoftLCID.MONGOLIAN_CYRILLIC_MONGOLIA);
			LCID_MAP.put(new Integer(MicrosoftLCID.MONGOLIAN_TRADITIONAL_PRC.getLanguageCode()), MicrosoftLCID.MONGOLIAN_TRADITIONAL_PRC);
			LCID_MAP.put(new Integer(MicrosoftLCID.NEPALI_NEPAL.getLanguageCode()), MicrosoftLCID.NEPALI_NEPAL);
			LCID_MAP.put(new Integer(MicrosoftLCID.NORWEGIAN_BOKMAL_NORWAY.getLanguageCode()), MicrosoftLCID.NORWEGIAN_BOKMAL_NORWAY);
			LCID_MAP.put(new Integer(MicrosoftLCID.NORWEGIAN_NYNORSK_NORWAY.getLanguageCode()), MicrosoftLCID.NORWEGIAN_NYNORSK_NORWAY);
			LCID_MAP.put(new Integer(MicrosoftLCID.OCCITAN_FRANCE.getLanguageCode()), MicrosoftLCID.OCCITAN_FRANCE);
			LCID_MAP.put(new Integer(MicrosoftLCID.ORIYA_INDIA.getLanguageCode()), MicrosoftLCID.ORIYA_INDIA);
			LCID_MAP.put(new Integer(MicrosoftLCID.PASHTO_AFGHANISTAN.getLanguageCode()), MicrosoftLCID.PASHTO_AFGHANISTAN);
			LCID_MAP.put(new Integer(MicrosoftLCID.POLISH_POLAND.getLanguageCode()), MicrosoftLCID.POLISH_POLAND);
			LCID_MAP.put(new Integer(MicrosoftLCID.PORTUGUESE_BRAZIL.getLanguageCode()), MicrosoftLCID.PORTUGUESE_BRAZIL);
			LCID_MAP.put(new Integer(MicrosoftLCID.PORTUGUESE_PORTUGAL.getLanguageCode()), MicrosoftLCID.PORTUGUESE_PORTUGAL);
			LCID_MAP.put(new Integer(MicrosoftLCID.PUNJABI_INDIA.getLanguageCode()), MicrosoftLCID.PUNJABI_INDIA);
			LCID_MAP.put(new Integer(MicrosoftLCID.QUECHUA_BOLIVIA.getLanguageCode()), MicrosoftLCID.QUECHUA_BOLIVIA);
			LCID_MAP.put(new Integer(MicrosoftLCID.QUECHUA_ECUADOR.getLanguageCode()), MicrosoftLCID.QUECHUA_ECUADOR);
			LCID_MAP.put(new Integer(MicrosoftLCID.QUECHUA_PERU.getLanguageCode()), MicrosoftLCID.QUECHUA_PERU);
			LCID_MAP.put(new Integer(MicrosoftLCID.ROMANIAN_ROMANIA.getLanguageCode()), MicrosoftLCID.ROMANIAN_ROMANIA);
			LCID_MAP.put(new Integer(MicrosoftLCID.ROMANSH_SWITZERLAND.getLanguageCode()), MicrosoftLCID.ROMANSH_SWITZERLAND);
			LCID_MAP.put(new Integer(MicrosoftLCID.RUSSIAN_RUSSIA.getLanguageCode()), MicrosoftLCID.RUSSIAN_RUSSIA);
			LCID_MAP.put(new Integer(MicrosoftLCID.SAMI_INARI_FINLAND.getLanguageCode()), MicrosoftLCID.SAMI_INARI_FINLAND);
			LCID_MAP.put(new Integer(MicrosoftLCID.SAMI_LULE_NORWAY.getLanguageCode()), MicrosoftLCID.SAMI_LULE_NORWAY);
			LCID_MAP.put(new Integer(MicrosoftLCID.SAMI_LULE_SWEDEN.getLanguageCode()), MicrosoftLCID.SAMI_LULE_SWEDEN);
			LCID_MAP.put(new Integer(MicrosoftLCID.SAMI_NORTHERN_FINLAND.getLanguageCode()), MicrosoftLCID.SAMI_NORTHERN_FINLAND);
			LCID_MAP.put(new Integer(MicrosoftLCID.SAMI_NORTHERN_NORWAY.getLanguageCode()), MicrosoftLCID.SAMI_NORTHERN_NORWAY);
			LCID_MAP.put(new Integer(MicrosoftLCID.SAMI_NORTHERN_SWEDEN.getLanguageCode()), MicrosoftLCID.SAMI_NORTHERN_SWEDEN);
			LCID_MAP.put(new Integer(MicrosoftLCID.SAMI_SKOLT_FINLAND.getLanguageCode()), MicrosoftLCID.SAMI_SKOLT_FINLAND);
			LCID_MAP.put(new Integer(MicrosoftLCID.SAMI_SOUTHERN_NORWAY.getLanguageCode()), MicrosoftLCID.SAMI_SOUTHERN_NORWAY);
			LCID_MAP.put(new Integer(MicrosoftLCID.SAMI_SOUTHERN_SWEDEN.getLanguageCode()), MicrosoftLCID.SAMI_SOUTHERN_SWEDEN);
			LCID_MAP.put(new Integer(MicrosoftLCID.SANSKRIT_INDIA.getLanguageCode()), MicrosoftLCID.SANSKRIT_INDIA);
			LCID_MAP.put(new Integer(MicrosoftLCID.SERBIAN_CYRILLIC_BOSNIA_AND_HERZEGOVINA.getLanguageCode()), MicrosoftLCID.SERBIAN_CYRILLIC_BOSNIA_AND_HERZEGOVINA);
			LCID_MAP.put(new Integer(MicrosoftLCID.SERBIAN_CYRILLIC_SERBIA.getLanguageCode()), MicrosoftLCID.SERBIAN_CYRILLIC_SERBIA);
			LCID_MAP.put(new Integer(MicrosoftLCID.SERBIAN_LATIN_BOSNIA_AND_HERZEGOVINA.getLanguageCode()), MicrosoftLCID.SERBIAN_LATIN_BOSNIA_AND_HERZEGOVINA);
			LCID_MAP.put(new Integer(MicrosoftLCID.SERBIAN_LATIN_SERBIA.getLanguageCode()), MicrosoftLCID.SERBIAN_LATIN_SERBIA);
			LCID_MAP.put(new Integer(MicrosoftLCID.SESOTHO_SA_LEBOA_SOUTH_AFRICA.getLanguageCode()), MicrosoftLCID.SESOTHO_SA_LEBOA_SOUTH_AFRICA);
			LCID_MAP.put(new Integer(MicrosoftLCID.SETSWANA_SOUTH_AFRICA.getLanguageCode()), MicrosoftLCID.SETSWANA_SOUTH_AFRICA);
			LCID_MAP.put(new Integer(MicrosoftLCID.SINHALA_SRI_LANKA.getLanguageCode()), MicrosoftLCID.SINHALA_SRI_LANKA);
			LCID_MAP.put(new Integer(MicrosoftLCID.SLOVAK_SLOVAKIA.getLanguageCode()), MicrosoftLCID.SLOVAK_SLOVAKIA);
			LCID_MAP.put(new Integer(MicrosoftLCID.SLOVENIAN_SLOVENIA.getLanguageCode()), MicrosoftLCID.SLOVENIAN_SLOVENIA);
			LCID_MAP.put(new Integer(MicrosoftLCID.SPANISH_ARGENTINA.getLanguageCode()), MicrosoftLCID.SPANISH_ARGENTINA);
			LCID_MAP.put(new Integer(MicrosoftLCID.SPANISH_BOLIVIA.getLanguageCode()), MicrosoftLCID.SPANISH_BOLIVIA);
			LCID_MAP.put(new Integer(MicrosoftLCID.SPANISH_CHILE.getLanguageCode()), MicrosoftLCID.SPANISH_CHILE);
			LCID_MAP.put(new Integer(MicrosoftLCID.SPANISH_COLOMBIA.getLanguageCode()), MicrosoftLCID.SPANISH_COLOMBIA);
			LCID_MAP.put(new Integer(MicrosoftLCID.SPANISH_COSTA_RICA.getLanguageCode()), MicrosoftLCID.SPANISH_COSTA_RICA);
			LCID_MAP.put(new Integer(MicrosoftLCID.SPANISH_DOMINICAN_REPUBLIC.getLanguageCode()), MicrosoftLCID.SPANISH_DOMINICAN_REPUBLIC);
			LCID_MAP.put(new Integer(MicrosoftLCID.SPANISH_ECUADOR.getLanguageCode()), MicrosoftLCID.SPANISH_ECUADOR);
			LCID_MAP.put(new Integer(MicrosoftLCID.SPANISH_EL_SALVADOR.getLanguageCode()), MicrosoftLCID.SPANISH_EL_SALVADOR);
			LCID_MAP.put(new Integer(MicrosoftLCID.SPANISH_GUATEMALA.getLanguageCode()), MicrosoftLCID.SPANISH_GUATEMALA);
			LCID_MAP.put(new Integer(MicrosoftLCID.SPANISH_HONDURAS.getLanguageCode()), MicrosoftLCID.SPANISH_HONDURAS);
			LCID_MAP.put(new Integer(MicrosoftLCID.SPANISH_MEXICO.getLanguageCode()), MicrosoftLCID.SPANISH_MEXICO);
			LCID_MAP.put(new Integer(MicrosoftLCID.SPANISH_NICARAGUA.getLanguageCode()), MicrosoftLCID.SPANISH_NICARAGUA);
			LCID_MAP.put(new Integer(MicrosoftLCID.SPANISH_PANAMA.getLanguageCode()), MicrosoftLCID.SPANISH_PANAMA);
			LCID_MAP.put(new Integer(MicrosoftLCID.SPANISH_PARAGUAY.getLanguageCode()), MicrosoftLCID.SPANISH_PARAGUAY);
			LCID_MAP.put(new Integer(MicrosoftLCID.SPANISH_PERU.getLanguageCode()), MicrosoftLCID.SPANISH_PERU);
			LCID_MAP.put(new Integer(MicrosoftLCID.SPANISH_PUERTO_RICO.getLanguageCode()), MicrosoftLCID.SPANISH_PUERTO_RICO);
			LCID_MAP.put(new Integer(MicrosoftLCID.SPANISH_SPAIN.getLanguageCode()), MicrosoftLCID.SPANISH_SPAIN);
			LCID_MAP.put(new Integer(MicrosoftLCID.SPANISH_SPAIN_TRADITIONAL.getLanguageCode()), MicrosoftLCID.SPANISH_SPAIN_TRADITIONAL);
			LCID_MAP.put(new Integer(MicrosoftLCID.SPANISH_UNITED_STATES.getLanguageCode()), MicrosoftLCID.SPANISH_UNITED_STATES);
			LCID_MAP.put(new Integer(MicrosoftLCID.SPANISH_URUGUAY.getLanguageCode()), MicrosoftLCID.SPANISH_URUGUAY);
			LCID_MAP.put(new Integer(MicrosoftLCID.SPANISH_VENEZUELA.getLanguageCode()), MicrosoftLCID.SPANISH_VENEZUELA);
			LCID_MAP.put(new Integer(MicrosoftLCID.SWEDISH_FINLAND.getLanguageCode()), MicrosoftLCID.SWEDISH_FINLAND);
			LCID_MAP.put(new Integer(MicrosoftLCID.SWEDISH_SWEDEN.getLanguageCode()), MicrosoftLCID.SWEDISH_SWEDEN);
			LCID_MAP.put(new Integer(MicrosoftLCID.SYRIAC_SYRIA.getLanguageCode()), MicrosoftLCID.SYRIAC_SYRIA);
			LCID_MAP.put(new Integer(MicrosoftLCID.TAJIK_CYRILLIC_TAJIKISTAN.getLanguageCode()), MicrosoftLCID.TAJIK_CYRILLIC_TAJIKISTAN);
			LCID_MAP.put(new Integer(MicrosoftLCID.TAMAZIGHT_LATIN_ALGERIA.getLanguageCode()), MicrosoftLCID.TAMAZIGHT_LATIN_ALGERIA);
			LCID_MAP.put(new Integer(MicrosoftLCID.TAMIL_INDIA.getLanguageCode()), MicrosoftLCID.TAMIL_INDIA);
			LCID_MAP.put(new Integer(MicrosoftLCID.TATAR_RUSSIA.getLanguageCode()), MicrosoftLCID.TATAR_RUSSIA);
			LCID_MAP.put(new Integer(MicrosoftLCID.TELUGU_INDIA.getLanguageCode()), MicrosoftLCID.TELUGU_INDIA);
			LCID_MAP.put(new Integer(MicrosoftLCID.THAI_THAILAND.getLanguageCode()), MicrosoftLCID.THAI_THAILAND);
			LCID_MAP.put(new Integer(MicrosoftLCID.TIBETAN_PRC.getLanguageCode()), MicrosoftLCID.TIBETAN_PRC);
			LCID_MAP.put(new Integer(MicrosoftLCID.TURKISH_TURKEY.getLanguageCode()), MicrosoftLCID.TURKISH_TURKEY);
			LCID_MAP.put(new Integer(MicrosoftLCID.TURKMEN_TURKMENISTAN.getLanguageCode()), MicrosoftLCID.TURKMEN_TURKMENISTAN);
			LCID_MAP.put(new Integer(MicrosoftLCID.UIGHUR_PRC.getLanguageCode()), MicrosoftLCID.UIGHUR_PRC);
			LCID_MAP.put(new Integer(MicrosoftLCID.UKRAINIAN_UKRAINE.getLanguageCode()), MicrosoftLCID.UKRAINIAN_UKRAINE);
			LCID_MAP.put(new Integer(MicrosoftLCID.UPPER_SORBIAN_GERMANY.getLanguageCode()), MicrosoftLCID.UPPER_SORBIAN_GERMANY);
			LCID_MAP.put(new Integer(MicrosoftLCID.URDU_PAKISTAN.getLanguageCode()), MicrosoftLCID.URDU_PAKISTAN);
			LCID_MAP.put(new Integer(MicrosoftLCID.UZBEK_CYRILLIC_UZBEKISTAN.getLanguageCode()), MicrosoftLCID.UZBEK_CYRILLIC_UZBEKISTAN);
			LCID_MAP.put(new Integer(MicrosoftLCID.UZBEK_LATIN_UZBEKISTAN.getLanguageCode()), MicrosoftLCID.UZBEK_LATIN_UZBEKISTAN);
			LCID_MAP.put(new Integer(MicrosoftLCID.VIETNAMESE_VIETNAM.getLanguageCode()), MicrosoftLCID.VIETNAMESE_VIETNAM);
			LCID_MAP.put(new Integer(MicrosoftLCID.WELSH_UNITED_KINGDOM.getLanguageCode()), MicrosoftLCID.WELSH_UNITED_KINGDOM);
			LCID_MAP.put(new Integer(MicrosoftLCID.WOLOF_SENEGAL.getLanguageCode()), MicrosoftLCID.WOLOF_SENEGAL);
			LCID_MAP.put(new Integer(MicrosoftLCID.YAKUT_RUSSIA.getLanguageCode()), MicrosoftLCID.YAKUT_RUSSIA);		}
	}

	/** This class defines constants for the Macintosh encoding ids.
	 */
	final public static class MacintoshEncodingId 
	{
		private MacintoshEncodingId() 
		{
			// This class is only a namespace for constants
			// and should not be instantiatet
		}
		public static final int ROMAN = 0;
		public static final int JAPANESE = 1;
		public static final int CHINESE_TRADITIONAL = 2;
		public static final int KOREAN = 3;
		public static final int ARABIC = 4;
		public static final int HEBREW = 5;
		public static final int GREEK = 6;
		public static final int RUSSIAN = 7;
		public static final int RSYMBOL = 8;
		public static final int DEVANAGARI = 9;
		public static final int GURMUKHI = 10;
		public static final int GUJARATI = 11;

		/* From Peter Edberg at Apple:
	    The encoding constants [kTextEncodingMacOriya = 12, kTextEncodingMacBengali = 13,
		kTextEncodingMacTamil = 14,
		kTextEncodingMacTelugu = 15,
		kTextEncodingMacKannada = 16,
		kTextEncodingMacMalayalam = 17,
		kTextEncodingMacSinhalese = 18,
		kTextEncodingMacBurmese = 19,
		kTextEncodingMacKhmer = 20,
		kTextEncodingMacLaotian = 22,
		kTextEncodingMacGeorgian = 23,
		kTextEncodingMacArmenian = 24,
		kTextEncodingMacMongolian = 27,
		kTextEncodingMacEthiopic = 28,
		kTextEncodingMacVietnamese = 30,
		kTextEncodingMacExtArabic = 31] 
	    correspond to old Script Manager script codes which were reserved to
	    designate encodings for the corresponding scripts, but for which Mac OS
	    encodings were never defined - and at this point likely never will be.
		 */
		public static final int ORIYA = 12;
		public static final int BENGALI = 13;
		public static final int TAMIL = 14;
		public static final int TELUGU = 15;
		public static final int KANNADA = 16;
		public static final int MALAYALAM = 17;
		public static final int SINHALESE = 18;
		public static final int BURMESE = 19;
		public static final int KHMER = 20;
		public static final int THAI = 21;
		public static final int LAOTIAN = 22;
		public static final int GEORGIAN = 23;
		public static final int ARMENIAN = 24;
		public static final int CHINESE_SIMPLIFIED = 25;
		public static final int TIBETAN = 26;
		public static final int MONGOLIAN = 27;
		public static final int GEEZ = 28;
		public static final int SLAVIC = 29;
		public static final int VIETNAMESE = 30;
		public static final int SINDHI = 31; 
	}

	final public static class MacintoshLanguageCode
	{
		private MacintoshLanguageCode() 
		{
			// This class is only a namespace for constants
			// and should not be instantiatet
		}

		public static final LanguageCodeEntry getCodeEntry(int code)
		{
			return (LanguageCodeEntry) MacintoshLanguageCode.LANGUAGE_MAP.get(new Integer(code));
		}

		private static final Map LANGUAGE_MAP = new HashMap(150);

		public static final LanguageCodeEntry AFRIKAANS = new LanguageCodeEntry(141, "af", null, null);
		public static final LanguageCodeEntry ALBANIAN = new LanguageCodeEntry(36, "sq", null, null);
		public static final LanguageCodeEntry AMHARIC = new LanguageCodeEntry(85, "am", null, null);
		public static final LanguageCodeEntry ARABIC = new LanguageCodeEntry(12, "ar", null, null);
		public static final LanguageCodeEntry ARMENIAN = new LanguageCodeEntry(51, "hy", null, null);
		public static final LanguageCodeEntry ASSAMESE = new LanguageCodeEntry(68, "as", null, null);
		public static final LanguageCodeEntry AYMARA = new LanguageCodeEntry(134, "ay", null, null);
		public static final LanguageCodeEntry AZERBAIJANI_ARABIC_SCRIPT = new LanguageCodeEntry(50, "az", "Arab", null);
		public static final LanguageCodeEntry AZERBAIJANI_CYRILLIC_SCRIPT = new LanguageCodeEntry(49, "az", "Cyrl", null);
		public static final LanguageCodeEntry AZERBAIJANI_ROMAN_SCRIPT = new LanguageCodeEntry(150, "az", "Latn", null);
		public static final LanguageCodeEntry BASQUE = new LanguageCodeEntry(129, "eu", null, null);
		public static final LanguageCodeEntry BENGALI = new LanguageCodeEntry(67, "bn", null, null);
		public static final LanguageCodeEntry BRETON = new LanguageCodeEntry(142, "br", null, null);
		public static final LanguageCodeEntry BULGARIAN = new LanguageCodeEntry(44, "bg", null, null);
		public static final LanguageCodeEntry BURMESE = new LanguageCodeEntry(77, "my", null, null);
		public static final LanguageCodeEntry BYELORUSSIAN = new LanguageCodeEntry(46, "be", null, null);
		public static final LanguageCodeEntry CATALAN = new LanguageCodeEntry(130, "ca", null, null);
		public static final LanguageCodeEntry CHINESE_SIMPLIFIED = new LanguageCodeEntry(33, "zh", "Hans", null);
		public static final LanguageCodeEntry CHINESE_TRADITIONAL = new LanguageCodeEntry(19, "zh", "Hant", null);
		public static final LanguageCodeEntry CROATIAN = new LanguageCodeEntry(18, "hr", null, null);
		public static final LanguageCodeEntry CZECH = new LanguageCodeEntry(38, "cs", null, null);
		public static final LanguageCodeEntry DANISH = new LanguageCodeEntry(7, "da", null, null);
		public static final LanguageCodeEntry DUTCH = new LanguageCodeEntry(4, "nl", null, null);
		public static final LanguageCodeEntry DZONGKHA = new LanguageCodeEntry(137, "dz", null, null);
		public static final LanguageCodeEntry ENGLISH = new LanguageCodeEntry(0, "en", null, null);
		public static final LanguageCodeEntry ESPERANTO = new LanguageCodeEntry(94, "eo", null, null);
		public static final LanguageCodeEntry ESTONIAN = new LanguageCodeEntry(27, "et", null, null);
		public static final LanguageCodeEntry FAROESE = new LanguageCodeEntry(30, "fo", null, null);
		public static final LanguageCodeEntry FARSI_PERSIAN = new LanguageCodeEntry(31, "fa", null, null);
		public static final LanguageCodeEntry FINNISH = new LanguageCodeEntry(13, "fi", null, null);
		public static final LanguageCodeEntry FLEMISH = new LanguageCodeEntry(34, "nl", null, null);
		public static final LanguageCodeEntry FRENCH = new LanguageCodeEntry(1, "fr", null, null);
		public static final LanguageCodeEntry GALICIAN = new LanguageCodeEntry(140, "gl", null, null);
		public static final LanguageCodeEntry GALLA = new LanguageCodeEntry(87, "om", null, null);
		public static final LanguageCodeEntry GEORGIAN = new LanguageCodeEntry(52, "ka", null, null);
		public static final LanguageCodeEntry GERMAN = new LanguageCodeEntry(2, "de", null, null);
		public static final LanguageCodeEntry GREEK = new LanguageCodeEntry(14, "el", null, null);
		public static final LanguageCodeEntry GREEK_POLYTONIC = new LanguageCodeEntry(148, "grc", null, null);
		public static final LanguageCodeEntry GREENLANDIC = new LanguageCodeEntry(149, "kl", null, null);
		public static final LanguageCodeEntry GUARANI = new LanguageCodeEntry(133, "gn", null, null);
		public static final LanguageCodeEntry GUJARATI = new LanguageCodeEntry(69, "gu", null, null);
		public static final LanguageCodeEntry HEBREW = new LanguageCodeEntry(10, "he", null, null);
		public static final LanguageCodeEntry HINDI = new LanguageCodeEntry(21, "hi", null, null);
		public static final LanguageCodeEntry HUNGARIAN = new LanguageCodeEntry(26, "hu", null, null);
		public static final LanguageCodeEntry ICELANDIC = new LanguageCodeEntry(15, "is", null, null);
		public static final LanguageCodeEntry INDONESIAN = new LanguageCodeEntry(81, "id", null, null);
		public static final LanguageCodeEntry INUKTITUT = new LanguageCodeEntry(14, "iu", null, null);
		public static final LanguageCodeEntry IRISH_GAELIC = new LanguageCodeEntry(35, "ga", null, null);
		public static final LanguageCodeEntry IRISH_GAELIC_WITH_DOT_ABOVE = new LanguageCodeEntry(146, "ga", "Latg", null);
		public static final LanguageCodeEntry ITALIAN = new LanguageCodeEntry(3, "it", null, null);
		public static final LanguageCodeEntry JAPANESE = new LanguageCodeEntry(11, "ja", null, null);
		public static final LanguageCodeEntry JAVANESE_ROMAN_SCRIPT = new LanguageCodeEntry(138, "jv", "Latn", null);
		public static final LanguageCodeEntry KANNADA = new LanguageCodeEntry(73, "kn", null, null);
		public static final LanguageCodeEntry KASHMIRI = new LanguageCodeEntry(61, "ks", null, null);
		public static final LanguageCodeEntry KAZAKH = new LanguageCodeEntry(48, "kk", null, null);
		public static final LanguageCodeEntry KHMER = new LanguageCodeEntry(78, "km", null, null);
		public static final LanguageCodeEntry KINYARWANDA_RUANDA = new LanguageCodeEntry(90, "rw", null, null);
		public static final LanguageCodeEntry KIRGHIZ = new LanguageCodeEntry(54, "ky", null, null);
		public static final LanguageCodeEntry KOREAN = new LanguageCodeEntry(23, "ko", null, null);
		public static final LanguageCodeEntry KURDISH = new LanguageCodeEntry(60, "ku", null, null);
		public static final LanguageCodeEntry LAO = new LanguageCodeEntry(79, "lo", null, null);
		public static final LanguageCodeEntry LATIN = new LanguageCodeEntry(131, "la", null, null);
		public static final LanguageCodeEntry LATVIAN = new LanguageCodeEntry(28, "lv", null, null);
		public static final LanguageCodeEntry LITHUANIAN = new LanguageCodeEntry(24, "lt", null, null);
		public static final LanguageCodeEntry MACEDONIAN = new LanguageCodeEntry(43, "mk", null, null);
		public static final LanguageCodeEntry MALAGASY = new LanguageCodeEntry(93, "mg", null, null);
		public static final LanguageCodeEntry MALAY_ARABIC_SCRIPT = new LanguageCodeEntry(84, "ms", "Arab", null);
		public static final LanguageCodeEntry MALAY_ROMAN_SCRIPT = new LanguageCodeEntry(83, "ms", "Latn", null);
		public static final LanguageCodeEntry MALAYALAM = new LanguageCodeEntry(72, "ml", null, null);
		public static final LanguageCodeEntry MALTESE = new LanguageCodeEntry(16, "mt", null, null);
		public static final LanguageCodeEntry MANX_GAELIC = new LanguageCodeEntry(145, "gv", null, null);
		public static final LanguageCodeEntry MARATHI = new LanguageCodeEntry(66, "mr", null, null);
		public static final LanguageCodeEntry MOLDAVIAN = new LanguageCodeEntry(53, "mo", null, null);
		public static final LanguageCodeEntry MONGOLIAN_CYRILLIC_SCRIPT = new LanguageCodeEntry(58, "mn", "Cyrl", null);
		public static final LanguageCodeEntry MONGOLIAN_MONGOLIAN_SCRIPT = new LanguageCodeEntry(57, "mn", "Mong", null);
		public static final LanguageCodeEntry NEPALI = new LanguageCodeEntry(64, "ne", null, null);
		public static final LanguageCodeEntry NORWEGIAN = new LanguageCodeEntry(9, "no", null, null);
		public static final LanguageCodeEntry NYANJA_CHEWA = new LanguageCodeEntry(92, "ny", null, null);
		public static final LanguageCodeEntry ORIYA = new LanguageCodeEntry(71, "or", null, null);
		public static final LanguageCodeEntry PASHTO = new LanguageCodeEntry(59, "ps", null, null);
		public static final LanguageCodeEntry POLISH = new LanguageCodeEntry(25, "pl", null, null);
		public static final LanguageCodeEntry PORTUGUESE = new LanguageCodeEntry(8, "pt", null, null);
		public static final LanguageCodeEntry PUNJABI = new LanguageCodeEntry(70, "pa", null, null);
		public static final LanguageCodeEntry QUENCHUA = new LanguageCodeEntry(132, "qu", null, null);
		public static final LanguageCodeEntry ROMANIAN = new LanguageCodeEntry(37, "ro", null, null);
		public static final LanguageCodeEntry RUNDI = new LanguageCodeEntry(91, "rn", null, null);
		public static final LanguageCodeEntry RUSSIAN = new LanguageCodeEntry(32, "ru", null, null);
		public static final LanguageCodeEntry SAMI = new LanguageCodeEntry(29, "se", null, null);
		public static final LanguageCodeEntry SANSKRIT = new LanguageCodeEntry(65, "sa", null, null);
		public static final LanguageCodeEntry SCOTTISH_GAELIC = new LanguageCodeEntry(144, "gd", null, null);
		public static final LanguageCodeEntry SERBIAN = new LanguageCodeEntry(42, "sr", null, null);
		public static final LanguageCodeEntry SINDHI = new LanguageCodeEntry(62, "sd", null, null);
		public static final LanguageCodeEntry SINHALESE = new LanguageCodeEntry(76, "si", null, null);
		public static final LanguageCodeEntry SLOVAK = new LanguageCodeEntry(39, "sk", null, null);
		public static final LanguageCodeEntry SLOVENIAN = new LanguageCodeEntry(40, "sl", null, null);
		public static final LanguageCodeEntry SOMALI = new LanguageCodeEntry(88, "so", null, null);
		public static final LanguageCodeEntry SPANISH = new LanguageCodeEntry(6, "es", null, null);
		public static final LanguageCodeEntry SUNDANESE_ROMAN_SCRIPT = new LanguageCodeEntry(139, "su", "Latn", null);
		public static final LanguageCodeEntry SWAHILI = new LanguageCodeEntry(89, "sw", null, null);
		public static final LanguageCodeEntry SWEDISH = new LanguageCodeEntry(5, "sv", null, null);
		public static final LanguageCodeEntry TAGALONG = new LanguageCodeEntry(82, "tl", null, null);
		public static final LanguageCodeEntry TAJIKI = new LanguageCodeEntry(55, "tg", null, null);
		public static final LanguageCodeEntry TAMIL = new LanguageCodeEntry(74, "ta", null, null);
		public static final LanguageCodeEntry TATAR = new LanguageCodeEntry(135, "tt", null, null);
		public static final LanguageCodeEntry TELUGU = new LanguageCodeEntry(75, "te", null, null);
		public static final LanguageCodeEntry THAI = new LanguageCodeEntry(22, "th", null, null);
		public static final LanguageCodeEntry TIBETAN = new LanguageCodeEntry(63, "bo", null, null);
		public static final LanguageCodeEntry TIGRINYA = new LanguageCodeEntry(86, "ti", null, null);
		public static final LanguageCodeEntry TONGAN = new LanguageCodeEntry(147, "to", null, null);
		public static final LanguageCodeEntry TURKISH = new LanguageCodeEntry(17, "tr", null, null);
		public static final LanguageCodeEntry TURKMEN = new LanguageCodeEntry(56, "tk", null, null);
		public static final LanguageCodeEntry UIGHUR = new LanguageCodeEntry(136, "ug", null, null);
		public static final LanguageCodeEntry UKRAINIAN = new LanguageCodeEntry(45, "uk", null, null);
		public static final LanguageCodeEntry URDU = new LanguageCodeEntry(20, "ur", null, null);
		public static final LanguageCodeEntry UZBEK = new LanguageCodeEntry(47, "uz", null, null);
		public static final LanguageCodeEntry VIETNAMESE = new LanguageCodeEntry(80, "vi", null, null);
		public static final LanguageCodeEntry WELSH = new LanguageCodeEntry(128, "cy", null, null);
		public static final LanguageCodeEntry YIDDISH = new LanguageCodeEntry(41, "yi", null, null);

		static
		{
			LANGUAGE_MAP.put(new Integer(MacintoshLanguageCode.AFRIKAANS.getLanguageCode()), MacintoshLanguageCode.AFRIKAANS);
			LANGUAGE_MAP.put(new Integer(MacintoshLanguageCode.ALBANIAN.getLanguageCode()), MacintoshLanguageCode.ALBANIAN);
			LANGUAGE_MAP.put(new Integer(MacintoshLanguageCode.AMHARIC.getLanguageCode()), MacintoshLanguageCode.AMHARIC);
			LANGUAGE_MAP.put(new Integer(MacintoshLanguageCode.ARABIC.getLanguageCode()), MacintoshLanguageCode.ARABIC);
			LANGUAGE_MAP.put(new Integer(MacintoshLanguageCode.ARMENIAN.getLanguageCode()), MacintoshLanguageCode.ARMENIAN);
			LANGUAGE_MAP.put(new Integer(MacintoshLanguageCode.ASSAMESE.getLanguageCode()), MacintoshLanguageCode.ASSAMESE);
			LANGUAGE_MAP.put(new Integer(MacintoshLanguageCode.AYMARA.getLanguageCode()), MacintoshLanguageCode.AYMARA);
			LANGUAGE_MAP.put(new Integer(MacintoshLanguageCode.AZERBAIJANI_ARABIC_SCRIPT.getLanguageCode()), MacintoshLanguageCode.AZERBAIJANI_ARABIC_SCRIPT);
			LANGUAGE_MAP.put(new Integer(MacintoshLanguageCode.AZERBAIJANI_CYRILLIC_SCRIPT.getLanguageCode()), MacintoshLanguageCode.AZERBAIJANI_CYRILLIC_SCRIPT);
			LANGUAGE_MAP.put(new Integer(MacintoshLanguageCode.AZERBAIJANI_ROMAN_SCRIPT.getLanguageCode()), MacintoshLanguageCode.AZERBAIJANI_ROMAN_SCRIPT);
			LANGUAGE_MAP.put(new Integer(MacintoshLanguageCode.BASQUE.getLanguageCode()), MacintoshLanguageCode.BASQUE);
			LANGUAGE_MAP.put(new Integer(MacintoshLanguageCode.BENGALI.getLanguageCode()), MacintoshLanguageCode.BENGALI);
			LANGUAGE_MAP.put(new Integer(MacintoshLanguageCode.BRETON.getLanguageCode()), MacintoshLanguageCode.BRETON);
			LANGUAGE_MAP.put(new Integer(MacintoshLanguageCode.BULGARIAN.getLanguageCode()), MacintoshLanguageCode.BULGARIAN);
			LANGUAGE_MAP.put(new Integer(MacintoshLanguageCode.BURMESE.getLanguageCode()), MacintoshLanguageCode.BURMESE);
			LANGUAGE_MAP.put(new Integer(MacintoshLanguageCode.BYELORUSSIAN.getLanguageCode()), MacintoshLanguageCode.BYELORUSSIAN);
			LANGUAGE_MAP.put(new Integer(MacintoshLanguageCode.CATALAN.getLanguageCode()), MacintoshLanguageCode.CATALAN);
			LANGUAGE_MAP.put(new Integer(MacintoshLanguageCode.CHINESE_SIMPLIFIED.getLanguageCode()), MacintoshLanguageCode.CHINESE_SIMPLIFIED);
			LANGUAGE_MAP.put(new Integer(MacintoshLanguageCode.CHINESE_TRADITIONAL.getLanguageCode()), MacintoshLanguageCode.CHINESE_TRADITIONAL);
			LANGUAGE_MAP.put(new Integer(MacintoshLanguageCode.CROATIAN.getLanguageCode()), MacintoshLanguageCode.CROATIAN);
			LANGUAGE_MAP.put(new Integer(MacintoshLanguageCode.CZECH.getLanguageCode()), MacintoshLanguageCode.CZECH);
			LANGUAGE_MAP.put(new Integer(MacintoshLanguageCode.DANISH.getLanguageCode()), MacintoshLanguageCode.DANISH);
			LANGUAGE_MAP.put(new Integer(MacintoshLanguageCode.DUTCH.getLanguageCode()), MacintoshLanguageCode.DUTCH);
			LANGUAGE_MAP.put(new Integer(MacintoshLanguageCode.DZONGKHA.getLanguageCode()), MacintoshLanguageCode.DZONGKHA);
			LANGUAGE_MAP.put(new Integer(MacintoshLanguageCode.ENGLISH.getLanguageCode()), MacintoshLanguageCode.ENGLISH);
			LANGUAGE_MAP.put(new Integer(MacintoshLanguageCode.ESPERANTO.getLanguageCode()), MacintoshLanguageCode.ESPERANTO);
			LANGUAGE_MAP.put(new Integer(MacintoshLanguageCode.ESTONIAN.getLanguageCode()), MacintoshLanguageCode.ESTONIAN);
			LANGUAGE_MAP.put(new Integer(MacintoshLanguageCode.FAROESE.getLanguageCode()), MacintoshLanguageCode.FAROESE);
			LANGUAGE_MAP.put(new Integer(MacintoshLanguageCode.FARSI_PERSIAN.getLanguageCode()), MacintoshLanguageCode.FARSI_PERSIAN);
			LANGUAGE_MAP.put(new Integer(MacintoshLanguageCode.FINNISH.getLanguageCode()), MacintoshLanguageCode.FINNISH);
			LANGUAGE_MAP.put(new Integer(MacintoshLanguageCode.FLEMISH.getLanguageCode()), MacintoshLanguageCode.FLEMISH);
			LANGUAGE_MAP.put(new Integer(MacintoshLanguageCode.FRENCH.getLanguageCode()), MacintoshLanguageCode.FRENCH);
			LANGUAGE_MAP.put(new Integer(MacintoshLanguageCode.GALICIAN.getLanguageCode()), MacintoshLanguageCode.GALICIAN);
			LANGUAGE_MAP.put(new Integer(MacintoshLanguageCode.GALLA.getLanguageCode()), MacintoshLanguageCode.GALLA);
			LANGUAGE_MAP.put(new Integer(MacintoshLanguageCode.GEORGIAN.getLanguageCode()), MacintoshLanguageCode.GEORGIAN);
			LANGUAGE_MAP.put(new Integer(MacintoshLanguageCode.GERMAN.getLanguageCode()), MacintoshLanguageCode.GERMAN);
			LANGUAGE_MAP.put(new Integer(MacintoshLanguageCode.GREEK.getLanguageCode()), MacintoshLanguageCode.GREEK);
			LANGUAGE_MAP.put(new Integer(MacintoshLanguageCode.GREEK_POLYTONIC.getLanguageCode()), MacintoshLanguageCode.GREEK_POLYTONIC);
			LANGUAGE_MAP.put(new Integer(MacintoshLanguageCode.GREENLANDIC.getLanguageCode()), MacintoshLanguageCode.GREENLANDIC);
			LANGUAGE_MAP.put(new Integer(MacintoshLanguageCode.GUARANI.getLanguageCode()), MacintoshLanguageCode.GUARANI);
			LANGUAGE_MAP.put(new Integer(MacintoshLanguageCode.GUJARATI.getLanguageCode()), MacintoshLanguageCode.GUJARATI);
			LANGUAGE_MAP.put(new Integer(MacintoshLanguageCode.HEBREW.getLanguageCode()), MacintoshLanguageCode.HEBREW);
			LANGUAGE_MAP.put(new Integer(MacintoshLanguageCode.HINDI.getLanguageCode()), MacintoshLanguageCode.HINDI);
			LANGUAGE_MAP.put(new Integer(MacintoshLanguageCode.HUNGARIAN.getLanguageCode()), MacintoshLanguageCode.HUNGARIAN);
			LANGUAGE_MAP.put(new Integer(MacintoshLanguageCode.ICELANDIC.getLanguageCode()), MacintoshLanguageCode.ICELANDIC);
			LANGUAGE_MAP.put(new Integer(MacintoshLanguageCode.INDONESIAN.getLanguageCode()), MacintoshLanguageCode.INDONESIAN);
			LANGUAGE_MAP.put(new Integer(MacintoshLanguageCode.INUKTITUT.getLanguageCode()), MacintoshLanguageCode.INUKTITUT);
			LANGUAGE_MAP.put(new Integer(MacintoshLanguageCode.IRISH_GAELIC.getLanguageCode()), MacintoshLanguageCode.IRISH_GAELIC);
			LANGUAGE_MAP.put(new Integer(MacintoshLanguageCode.IRISH_GAELIC_WITH_DOT_ABOVE.getLanguageCode()), MacintoshLanguageCode.IRISH_GAELIC_WITH_DOT_ABOVE);
			LANGUAGE_MAP.put(new Integer(MacintoshLanguageCode.ITALIAN.getLanguageCode()), MacintoshLanguageCode.ITALIAN);
			LANGUAGE_MAP.put(new Integer(MacintoshLanguageCode.JAPANESE.getLanguageCode()), MacintoshLanguageCode.JAPANESE);
			LANGUAGE_MAP.put(new Integer(MacintoshLanguageCode.JAVANESE_ROMAN_SCRIPT.getLanguageCode()), MacintoshLanguageCode.JAVANESE_ROMAN_SCRIPT);
			LANGUAGE_MAP.put(new Integer(MacintoshLanguageCode.KANNADA.getLanguageCode()), MacintoshLanguageCode.KANNADA);
			LANGUAGE_MAP.put(new Integer(MacintoshLanguageCode.KASHMIRI.getLanguageCode()), MacintoshLanguageCode.KASHMIRI);
			LANGUAGE_MAP.put(new Integer(MacintoshLanguageCode.KAZAKH.getLanguageCode()), MacintoshLanguageCode.KAZAKH);
			LANGUAGE_MAP.put(new Integer(MacintoshLanguageCode.KHMER.getLanguageCode()), MacintoshLanguageCode.KHMER);
			LANGUAGE_MAP.put(new Integer(MacintoshLanguageCode.KINYARWANDA_RUANDA.getLanguageCode()), MacintoshLanguageCode.KINYARWANDA_RUANDA);
			LANGUAGE_MAP.put(new Integer(MacintoshLanguageCode.KIRGHIZ.getLanguageCode()), MacintoshLanguageCode.KIRGHIZ);
			LANGUAGE_MAP.put(new Integer(MacintoshLanguageCode.KOREAN.getLanguageCode()), MacintoshLanguageCode.KOREAN);
			LANGUAGE_MAP.put(new Integer(MacintoshLanguageCode.KURDISH.getLanguageCode()), MacintoshLanguageCode.KURDISH);
			LANGUAGE_MAP.put(new Integer(MacintoshLanguageCode.LAO.getLanguageCode()), MacintoshLanguageCode.LAO);
			LANGUAGE_MAP.put(new Integer(MacintoshLanguageCode.LATIN.getLanguageCode()), MacintoshLanguageCode.LATIN);
			LANGUAGE_MAP.put(new Integer(MacintoshLanguageCode.LATVIAN.getLanguageCode()), MacintoshLanguageCode.LATVIAN);
			LANGUAGE_MAP.put(new Integer(MacintoshLanguageCode.LITHUANIAN.getLanguageCode()), MacintoshLanguageCode.LITHUANIAN);
			LANGUAGE_MAP.put(new Integer(MacintoshLanguageCode.MACEDONIAN.getLanguageCode()), MacintoshLanguageCode.MACEDONIAN);
			LANGUAGE_MAP.put(new Integer(MacintoshLanguageCode.MALAGASY.getLanguageCode()), MacintoshLanguageCode.MALAGASY);
			LANGUAGE_MAP.put(new Integer(MacintoshLanguageCode.MALAY_ARABIC_SCRIPT.getLanguageCode()), MacintoshLanguageCode.MALAY_ARABIC_SCRIPT);
			LANGUAGE_MAP.put(new Integer(MacintoshLanguageCode.MALAY_ROMAN_SCRIPT.getLanguageCode()), MacintoshLanguageCode.MALAY_ROMAN_SCRIPT);
			LANGUAGE_MAP.put(new Integer(MacintoshLanguageCode.MALAYALAM.getLanguageCode()), MacintoshLanguageCode.MALAYALAM);
			LANGUAGE_MAP.put(new Integer(MacintoshLanguageCode.MALTESE.getLanguageCode()), MacintoshLanguageCode.MALTESE);
			LANGUAGE_MAP.put(new Integer(MacintoshLanguageCode.MANX_GAELIC.getLanguageCode()), MacintoshLanguageCode.MANX_GAELIC);
			LANGUAGE_MAP.put(new Integer(MacintoshLanguageCode.MARATHI.getLanguageCode()), MacintoshLanguageCode.MARATHI);
			LANGUAGE_MAP.put(new Integer(MacintoshLanguageCode.MOLDAVIAN.getLanguageCode()), MacintoshLanguageCode.MOLDAVIAN);
			LANGUAGE_MAP.put(new Integer(MacintoshLanguageCode.MONGOLIAN_CYRILLIC_SCRIPT.getLanguageCode()), MacintoshLanguageCode.MONGOLIAN_CYRILLIC_SCRIPT);
			LANGUAGE_MAP.put(new Integer(MacintoshLanguageCode.MONGOLIAN_MONGOLIAN_SCRIPT.getLanguageCode()), MacintoshLanguageCode.MONGOLIAN_MONGOLIAN_SCRIPT);
			LANGUAGE_MAP.put(new Integer(MacintoshLanguageCode.NEPALI.getLanguageCode()), MacintoshLanguageCode.NEPALI);
			LANGUAGE_MAP.put(new Integer(MacintoshLanguageCode.NORWEGIAN.getLanguageCode()), MacintoshLanguageCode.NORWEGIAN);
			LANGUAGE_MAP.put(new Integer(MacintoshLanguageCode.NYANJA_CHEWA.getLanguageCode()), MacintoshLanguageCode.NYANJA_CHEWA);
			LANGUAGE_MAP.put(new Integer(MacintoshLanguageCode.ORIYA.getLanguageCode()), MacintoshLanguageCode.ORIYA);
			LANGUAGE_MAP.put(new Integer(MacintoshLanguageCode.PASHTO.getLanguageCode()), MacintoshLanguageCode.PASHTO);
			LANGUAGE_MAP.put(new Integer(MacintoshLanguageCode.POLISH.getLanguageCode()), MacintoshLanguageCode.POLISH);
			LANGUAGE_MAP.put(new Integer(MacintoshLanguageCode.PORTUGUESE.getLanguageCode()), MacintoshLanguageCode.PORTUGUESE);
			LANGUAGE_MAP.put(new Integer(MacintoshLanguageCode.PUNJABI.getLanguageCode()), MacintoshLanguageCode.PUNJABI);
			LANGUAGE_MAP.put(new Integer(MacintoshLanguageCode.QUENCHUA.getLanguageCode()), MacintoshLanguageCode.QUENCHUA);
			LANGUAGE_MAP.put(new Integer(MacintoshLanguageCode.ROMANIAN.getLanguageCode()), MacintoshLanguageCode.ROMANIAN);
			LANGUAGE_MAP.put(new Integer(MacintoshLanguageCode.RUNDI.getLanguageCode()), MacintoshLanguageCode.RUNDI);
			LANGUAGE_MAP.put(new Integer(MacintoshLanguageCode.RUSSIAN.getLanguageCode()), MacintoshLanguageCode.RUSSIAN);
			LANGUAGE_MAP.put(new Integer(MacintoshLanguageCode.SAMI.getLanguageCode()), MacintoshLanguageCode.SAMI);
			LANGUAGE_MAP.put(new Integer(MacintoshLanguageCode.SANSKRIT.getLanguageCode()), MacintoshLanguageCode.SANSKRIT);
			LANGUAGE_MAP.put(new Integer(MacintoshLanguageCode.SCOTTISH_GAELIC.getLanguageCode()), MacintoshLanguageCode.SCOTTISH_GAELIC);
			LANGUAGE_MAP.put(new Integer(MacintoshLanguageCode.SERBIAN.getLanguageCode()), MacintoshLanguageCode.SERBIAN);
			LANGUAGE_MAP.put(new Integer(MacintoshLanguageCode.SINDHI.getLanguageCode()), MacintoshLanguageCode.SINDHI);
			LANGUAGE_MAP.put(new Integer(MacintoshLanguageCode.SINHALESE.getLanguageCode()), MacintoshLanguageCode.SINHALESE);
			LANGUAGE_MAP.put(new Integer(MacintoshLanguageCode.SLOVAK.getLanguageCode()), MacintoshLanguageCode.SLOVAK);
			LANGUAGE_MAP.put(new Integer(MacintoshLanguageCode.SLOVENIAN.getLanguageCode()), MacintoshLanguageCode.SLOVENIAN);
			LANGUAGE_MAP.put(new Integer(MacintoshLanguageCode.SOMALI.getLanguageCode()), MacintoshLanguageCode.SOMALI);
			LANGUAGE_MAP.put(new Integer(MacintoshLanguageCode.SPANISH.getLanguageCode()), MacintoshLanguageCode.SPANISH);
			LANGUAGE_MAP.put(new Integer(MacintoshLanguageCode.SUNDANESE_ROMAN_SCRIPT.getLanguageCode()), MacintoshLanguageCode.SUNDANESE_ROMAN_SCRIPT);
			LANGUAGE_MAP.put(new Integer(MacintoshLanguageCode.SWAHILI.getLanguageCode()), MacintoshLanguageCode.SWAHILI);
			LANGUAGE_MAP.put(new Integer(MacintoshLanguageCode.SWEDISH.getLanguageCode()), MacintoshLanguageCode.SWEDISH);
			LANGUAGE_MAP.put(new Integer(MacintoshLanguageCode.TAGALONG.getLanguageCode()), MacintoshLanguageCode.TAGALONG);
			LANGUAGE_MAP.put(new Integer(MacintoshLanguageCode.TAJIKI.getLanguageCode()), MacintoshLanguageCode.TAJIKI);
			LANGUAGE_MAP.put(new Integer(MacintoshLanguageCode.TAMIL.getLanguageCode()), MacintoshLanguageCode.TAMIL);
			LANGUAGE_MAP.put(new Integer(MacintoshLanguageCode.TATAR.getLanguageCode()), MacintoshLanguageCode.TATAR);
			LANGUAGE_MAP.put(new Integer(MacintoshLanguageCode.TELUGU.getLanguageCode()), MacintoshLanguageCode.TELUGU);
			LANGUAGE_MAP.put(new Integer(MacintoshLanguageCode.THAI.getLanguageCode()), MacintoshLanguageCode.THAI);
			LANGUAGE_MAP.put(new Integer(MacintoshLanguageCode.TIBETAN.getLanguageCode()), MacintoshLanguageCode.TIBETAN);
			LANGUAGE_MAP.put(new Integer(MacintoshLanguageCode.TIGRINYA.getLanguageCode()), MacintoshLanguageCode.TIGRINYA);
			LANGUAGE_MAP.put(new Integer(MacintoshLanguageCode.TONGAN.getLanguageCode()), MacintoshLanguageCode.TONGAN);
			LANGUAGE_MAP.put(new Integer(MacintoshLanguageCode.TURKISH.getLanguageCode()), MacintoshLanguageCode.TURKISH);
			LANGUAGE_MAP.put(new Integer(MacintoshLanguageCode.TURKMEN.getLanguageCode()), MacintoshLanguageCode.TURKMEN);
			LANGUAGE_MAP.put(new Integer(MacintoshLanguageCode.UIGHUR.getLanguageCode()), MacintoshLanguageCode.UIGHUR);
			LANGUAGE_MAP.put(new Integer(MacintoshLanguageCode.UKRAINIAN.getLanguageCode()), MacintoshLanguageCode.UKRAINIAN);
			LANGUAGE_MAP.put(new Integer(MacintoshLanguageCode.URDU.getLanguageCode()), MacintoshLanguageCode.URDU);
			LANGUAGE_MAP.put(new Integer(MacintoshLanguageCode.UZBEK.getLanguageCode()), MacintoshLanguageCode.UZBEK);
			LANGUAGE_MAP.put(new Integer(MacintoshLanguageCode.VIETNAMESE.getLanguageCode()), MacintoshLanguageCode.VIETNAMESE);
			LANGUAGE_MAP.put(new Integer(MacintoshLanguageCode.WELSH.getLanguageCode()), MacintoshLanguageCode.WELSH);
			LANGUAGE_MAP.put(new Integer(MacintoshLanguageCode.YIDDISH.getLanguageCode()), MacintoshLanguageCode.YIDDISH);
		}
	}

	/*package protected*/ static ULocale getLocaleForLocaleID(int platformID, int localeID)
	{
		LanguageCodeEntry entry = null;
		switch (platformID)
		{
		case PlatformId.MACINTOSH:
			entry = MacintoshLanguageCode.getCodeEntry(localeID);
			break;
		case PlatformId.MICROSOFT:
			entry = MicrosoftLCID.getCodeEntry(localeID);
			break;
		}
		if (entry == null)
		{
			return null;
		}
		return entry.getLocale();
	}

	/** An interface that receives names from this table */
	static interface NameSelector {
		boolean nameFound(String name, int nameID, int platformID, int encoding, int language)
		throws InvalidFontException, UnsupportedFontException;
	}

	private static class CSSNameSelector implements NameSelector {
		private String currentSelection;
		int currentRanking = selectors.length;

		private final static int[][] selectors= new int[][] { 
			{PlatformId.MICROSOFT, MicrosoftEncodingId.UTF16_BE_BMP_ONLY, MicrosoftLCID.ENGLISH_UNITED_STATES.getLanguageCode()},
			{PlatformId.MICROSOFT, MicrosoftEncodingId.UTF16_BE_BMP_ONLY, -1},
			{PlatformId.MACINTOSH, MacintoshEncodingId.ROMAN, 0}};

		public boolean nameFound(String name, int nameID, int platformID, int encoding, int language)
		throws InvalidFontException, UnsupportedFontException 
		{
			for (int i = 0; i < selectors.length; i++) 
			{
				if (   platformID == selectors[i][0] 
				                                  && encoding == selectors[i][1] 
				                                                              && (language == selectors[i][2] || selectors[i][2] == -1) 
				                                                              && currentRanking > i) 
				{
					currentSelection = name;
					currentRanking = i;
					return false; 
				}
			}
			return false;
		}

		public String getName () 
		{
			return currentSelection;
		}
	}

	static class NameEntry
	{
		private final String name;
		private final int nameID;
		private final int platformID;
		private final int language;
		private final boolean nameIdDifferentiates;


		public NameEntry(String name, int nameID, int platformID, int language, int encoding, boolean nameIdDifferentiates) 
		{
			this.name = name;
			this.nameID = nameID;
			this.platformID = platformID;
			this.language = language;
			this.nameIdDifferentiates = nameIdDifferentiates;
		}

		String getName() 
		{
			return name;
		}

		int getNameID()
		{
			return this.nameID;
		}

		int getPlatformID() 
		{
			return this.platformID;
		}

		int getLanguage() 
		{
			return this.language;
		}


		public int hashCode() 
		{
			final int prime = 31;
			int result = 1;
			result = prime * result + this.language;
			if (this.nameIdDifferentiates)
			{
				result = prime * result + this.nameID;
			}
			result = prime * result + (this.nameIdDifferentiates ? 1231 : 1237);
			result = prime * result + this.platformID;
			return result;
		}

		public boolean equals(Object obj) 
		{
			if (this == obj)
			{
				return true;
			}
			if (obj == null)
			{
				return false;
			}
			if (!(obj instanceof NameEntry))
			{
				return false;
			}
			final NameEntry other = (NameEntry) obj;
			if (this.language != other.language)
			{
				return false;
			}
			if (this.nameIdDifferentiates && this.nameID != other.nameID)
			{
				return false;
			}
			if (this.nameIdDifferentiates != other.nameIdDifferentiates)
			{
				return false;
			}
			if (this.platformID != other.platformID)
			{
				return false;
			}
			return true;
		}

		public String toString() 
		{
			return new String("[" + this.nameID + ", " + this.platformID + " ," + this.language + "] = " + this.name);
		}
	}

	/** All names given to nameFound are added to a Set which can then be fetched */
	static class NameEntrySet implements NameSelector
	{
		private Map set;
		private int platformID = UNDEFINED;
		private ULocale locale = null;
		private int[] nameIDs = null;

		NameEntrySet() 
		{
			set = new HashMap();
		}

		public boolean nameFound(String name, int nameID, int platformID, int encoding, int language) 
		{
			// check for platform id
			if ((this.platformID != UNDEFINED) && (platformID != this.platformID))
			{
				return false;
			}

			// check for locale
			if (this.locale != null)
			{
				ULocale nameLocale = Name.getLocaleForLocaleID(platformID, language);
				if (!isContainedWithin(this.locale, nameLocale))
				{
					return false;
				}
			}

			NameEntry entry = new NameEntry(name, nameID, platformID, language, encoding, false /*name id doesn't differentiate*/);

			// check for name ids
			if (this.nameIDs != null)
			{
				NameEntry oldEntry = (NameEntry) this.set.get(entry);
				if (oldEntry != null)
				{
					int entryIndex = Integer.MAX_VALUE;
					int oldEntryIndex = Integer.MAX_VALUE;
					for (int nameIDIndex = 0; nameIDIndex < this.nameIDs.length; nameIDIndex++)
					{
						if (this.nameIDs[nameIDIndex] == entry.getLanguage())
						{
							entryIndex = nameIDIndex;
						}
						if (this.nameIDs[nameIDIndex] == oldEntry.getLanguage())
						{
							oldEntryIndex = nameIDIndex;
						}
					}
					if (oldEntryIndex < entryIndex)
					{
						entry = oldEntry;
					} else {
						this.set.remove(entry);
					}
				}
			}
			this.set.put(entry, entry);
			return false;
		}

		Set getSet() 
		{
			if (this.set.isEmpty())
			{
				return null;
			}
			return this.set.keySet();
		}

		public int hashCode() 
		{
			final int prime = 31;
			int result = 1;
			result = prime * result + this.locale.hashCode();
			result = prime * result + this.platformID;
			result = prime * result + ((this.set == null) ? 0 : this.set.hashCode());
			return result;
		}

		public boolean equals(Object obj) 
		{
			if (this == obj)
			{
				return true;
			}
			if (obj == null)
			{
				return false;
			}
			if (!(obj instanceof NameEntrySet))
			{
				return false;
			}
			final NameEntrySet other = (NameEntrySet) obj;
			if (!this.locale.equals(other.locale))
			{
				return false;
			}
			if (platformID != other.platformID)
			{
				return false;
			}
			if (set == null) 
			{
				if (other.set != null)
				{
					return false;
				}
			} 
			else if (!set.equals(other.set))
			{
				return false;
			}
			return true;
		}

		public void setPlatform(int platformID) 
		{
			this.platformID = platformID;
		}

		public void setLocale(ULocale locale) 
		{
			this.locale = locale;
		}

		public void setNameIDs(int[] nameIDs)
		{
			this.nameIDs = nameIDs;
		}

		/**
		 * Check whether this <code>LanguageCodeEntry</code> is contained within the given <b>ULocale</b>.
		 * It is contained within the other locale if this entry is equal to the locale or if this entry
		 * is a "tighter" locale specification than the given local.  For example, "en_CA" is contained
		 * within "en" and "mn_MN_Cyrl" is contained within "mn_MN".  However, "zh_SG" is <b>not</b>
		 * contained within "zh_TW".
		 * 
		 * This is also used in the ResourceFontData class.  Should be moved off to Rideau
		 * but will require that Rideau start using AGL.
		 * 
		 * @param baseLocale the locale to check against for containment
		 * @param locale the locale to check if it is withing the baseLocale
		 * @return <code>true</code> if this entry defines a locale that is contained within the given locale; 
		 * <code>false</code> otherwise
		 */
		static boolean isContainedWithin(ULocale baseLocale, ULocale locale)
		{
			if (locale == null)
			{
				return true;
			}

			// language
			String baseLanguage = baseLocale.getLanguage();
			if ((baseLanguage != null && !baseLanguage.equals("")) && !baseLanguage.equals(locale.getLanguage()))
			{
				return false;
			}

			// country
			String baseCountry = baseLocale.getCountry();
			if ((baseCountry != null && !baseCountry.equals("")) && !baseCountry.equals(locale.getCountry()))
			{
				return false;
			}

			// script
			String baseScript = baseLocale.getScript();
			if ((baseScript != null && !baseScript.equals("")) && !baseScript.equals(locale.getScript()))
			{
				return false;
			}

			// variant
			String baseVariant = baseLocale.getVariant();
			if ((baseVariant != null && !baseVariant.equals("")) && !baseVariant.equals(locale.getVariant()))
			{
				return false;
			}

			return true;
		}
	}

	/** All names given to nameFound are added to a Set which can then be fetched */
	private static class NameSetSelector implements NameSelector
	{
		private Set set;
		private int platformID = UNDEFINED;
		private int language = UNDEFINED;

		NameSetSelector() 
		{
			set = new HashSet ();
		}

		NameSetSelector(int platformID, int language)
		{
			this();
			this.platformID = platformID;
			this.language = language;
		}

		public boolean nameFound(String name, int nameID, int platformID, int encoding, int language) 
		{
			// all strings go in the set.
			if (   ((this.platformID == UNDEFINED) || (platformID == this.platformID))
					&& ((this.language == UNDEFINED) || (language == this.language)))
			{
				set.add(name);
			}
			return false;
		}

		Set getSet() 
		{
			return set;
		}
	}

	protected Name (FontByteArray buffer) 
	throws java.io.IOException, InvalidFontException, UnsupportedFontException 
	{
		super (buffer);
	}

	/** Figure out the Java charset name for an OpenType encoding.
	 * @param platformId
	 * @param encodingId
	 * @return the name of the corresponding charset, null if not supported
	 */
	protected static String getCharsetName(int platformId, int encodingId) 
	{
		if (platformId == PlatformId.MICROSOFT) 
		{
			switch (encodingId) 
			{
			case MicrosoftEncodingId.SYMBOL: return "UTF-16BE";
			case MicrosoftEncodingId.UTF16_BE_BMP_ONLY: return "UTF-16BE";
			case MicrosoftEncodingId.SHIFT_JIS: return "windows-932";
			case MicrosoftEncodingId.PRC: return "windows-936";
			case MicrosoftEncodingId.BIG5: return "windows-950";
			case MicrosoftEncodingId.WANSUNG: return "windows-949";
			case MicrosoftEncodingId.JOHAB: return "ms1361";
			case MicrosoftEncodingId.UTF16_BE_FULL: return "UTF-16BE";
			default: return null; 
			}
		} else if (platformId == PlatformId.MACINTOSH) {
			switch (encodingId) 
			{
			case MacintoshEncodingId.ROMAN: return "MacRoman";
			case MacintoshEncodingId.JAPANESE: return "MacJapanese"; 
			case MacintoshEncodingId.CHINESE_TRADITIONAL: return "MacChineseTraditional"; 
			case MacintoshEncodingId.KOREAN: return "MacKorean";
			case MacintoshEncodingId.ARABIC: return "MacArabic";
			case MacintoshEncodingId.HEBREW: return "MacHebrew";
			case MacintoshEncodingId.GREEK: return "MacGreek";
			case MacintoshEncodingId.RUSSIAN: return "MacCyrillic";
			case MacintoshEncodingId.RSYMBOL: return "MacSymbol";
			case MacintoshEncodingId.DEVANAGARI: return null; // not supported
			case MacintoshEncodingId.GURMUKHI : return null; // not supported
			case MacintoshEncodingId.GUJARATI : return null; // not supported
			case MacintoshEncodingId.ORIYA : return null;  // not supported
			case MacintoshEncodingId.BENGALI : return null;  // not supported
			case MacintoshEncodingId.TAMIL : return null;  // not supported
			case MacintoshEncodingId.TELUGU : return null;  // not supported
			case MacintoshEncodingId.KANNADA : return null; // not supported
			case MacintoshEncodingId.MALAYALAM : return null;  // not supported
			case MacintoshEncodingId.SINHALESE : return null;  // not supported
			case MacintoshEncodingId.BURMESE : return null;  // not supported
			case MacintoshEncodingId.KHMER : return null;  // not supported
			case MacintoshEncodingId.THAI : return "MacThai";
			case MacintoshEncodingId.LAOTIAN : return null;  // not supported
			case MacintoshEncodingId.GEORGIAN : return null;  // not supported
			case MacintoshEncodingId.ARMENIAN : return null;  // not supported
			case MacintoshEncodingId.CHINESE_SIMPLIFIED : return "MacChineseSimplified"; 
			case MacintoshEncodingId.TIBETAN : return null;  // not supported
			case MacintoshEncodingId.MONGOLIAN : return null;  // not supported
			case MacintoshEncodingId.GEEZ : return null;  // not supported
			case MacintoshEncodingId.SLAVIC : return null;  // not supported
			case MacintoshEncodingId.VIETNAMESE : return null;  // not supported
			case MacintoshEncodingId.SINDHI : return null;  // not supported
			default: return null; 
			}
		} else if (platformId == PlatformId.UNICODE) {
			return "UTF-16BE"; 
		} else {
			return null; 
		}
	}

	static Charset getCharset(String charsetName)
	{
		return CharsetUtil.forNameICU(charsetName);
	}

	/**
	 * Use AGL explicitly to convert strings in the name table.
	 * @throws UnmappableCharacterException 
	 */
	static String bytesToUnicode(byte[] data, int offset, int length, String charsetName)
	throws InvalidFontException, java.io.UnsupportedEncodingException, 
	CoderMalfunctionError, IllegalStateException, MalformedInputException, UnmappableCharacterException 
	{
		Charset charset = CharsetUtil.forNameICU(charsetName);
		return bytesToUnicode(data, offset, length, charset);
	}

	/**
	 * Use AGL explicitly to convert strings in the name table.
	 * @throws UnmappableCharacterException 
	 */
	static String bytesToUnicode(byte[] data, int offset, int length, Charset charset)
	throws InvalidFontException, java.io.UnsupportedEncodingException, 
	CoderMalfunctionError, IllegalStateException, MalformedInputException, UnmappableCharacterException 
	{
		CharsetDecoder decoder = charset.newDecoder();

		ByteBuffer bb = ByteBuffer.wrap(data, offset, length);

		int charsPerByte = (int) Math.ceil(decoder.maxCharsPerByte());
		CharBuffer cb = CharBuffer.allocate(length * charsPerByte);

		while (true) 
		{
			CoderResult coderResult = CharsetUtil.decodeLoop(bb, cb, decoder, true);
			if (coderResult.isOverflow())
			{
				// this should never happen if getMaxCharsPerByte 
				// returns an accurate answer. If it does, we can restart
				// with a larger buffer.
				decoder.reset ();
				cb = CharBuffer.allocate(cb.capacity() * 2);
				continue; 
			}
			else if (coderResult.isMalformed())
			{
				throw new MalformedInputException(length);
			}

			return new String(cb.array(), 0, cb.position());
		}
	}

	/** Get a 3, 1 name table entry. 
	 * 
	 * If <code>languageID</code> is -1, then any actual language will do.
	 */
	public String getName (int languageID, int nameID)
	throws InvalidFontException {

		try {
			return getName (3, 1, languageID, nameID); }
		catch (java.io.UnsupportedEncodingException e) {
			// 3, 1 -> UTF-16BE, which must be supported
			return null; }
	}

	/** Get an arbitrary name table entry.
	 * 
	 * If <code>languageID</code> is -1, then any actual language will do.
	 * @throws InvalidFontException
	 * @throws UnsupportedEncodingException
	 */
	public String getName (int platformID, int encodingID, 
			int languageID, int nameID) 
	throws InvalidFontException, UnsupportedEncodingException {

		String charsetName = getCharsetName(platformID, encodingID);
		Charset charset = getCharset(charsetName);

		if (charset == null) 
		{ 
			throw new java.io.UnsupportedEncodingException (); 
		}

		int nameRecordsNumber = this.data.getuint16 (2);
		int stringStorageOffset = this.data.getuint16 (4);

		for (int i = 0; i < nameRecordsNumber; i++) 
		{
			try {
				if (   (this.data.getuint16 (6 + 12*i + 0) == platformID )
						&& (this.data.getuint16 (6 + 12*i + 2) == encodingID)
						&& (this.data.getuint16 (6 + 12*i + 4) == languageID || -1 == languageID)
						&& this.data.getuint16 (6 + 12*i + 6) == nameID) 
				{
					int stringLength = this.data.getuint16 (6 + 12*i + 8);
					int stringOffset = stringStorageOffset + this.data.getuint16 (6 + 12*i + 10);

					return bytesToUnicode(data.getBytes(stringOffset, stringLength), 0, stringLength, charset);
				}
			} catch (java.io.UnsupportedEncodingException e) {
				/* ignore string in unsupported encodings */ 
			} catch (IllegalStateException e) {
				/*  ignore invalid strings */ 
			} catch (CoderMalfunctionError e) {
				/* ignore invalid strings */
			} catch (MalformedInputException e) {
				/* ignore invalid strings */
			} catch (UnmappableCharacterException e) {
				/* ignore unmappable characters */
			} catch (ArrayIndexOutOfBoundsException e) {
				/* ignore bad string table indices */ 
			}
		}

		return null;
	} 

	public boolean hasName (int platformID, int encodingID, 
			int languageID, int nameID)
	throws InvalidFontException {

		int nameRecordsNumber = this.data.getuint16 (2);

		for (int i = 0; i < nameRecordsNumber; i++) {
			if (   (this.data.getuint16 (6 + 12*i + 0) == platformID || -1 == platformID)
					&& (this.data.getuint16 (6 + 12*i + 2) == encodingID || -1 == encodingID)
					&& (this.data.getuint16 (6 + 12*i + 4) == languageID || -1 == languageID)
					&& (this.data.getuint16 (6 + 12*i + 6) == nameID     || -1 == nameID)) {

				return true; }}

		return false;
	} 

	/** look for names that have selectedNameId as their name id and that
	 * are in a supported charset/encoding. call s.nameFound on each of them.
	 * If nameFound returns true, stop and return the current string. Else continue to the next string.
	 */
	private String lookForNames (NameSelector s, int selectedNameId, 
			int nameRecordsNumber, int stringStorageOffset)
	throws UnsupportedFontException, InvalidFontException 
	{
		return lookForNames(s, new int[] {selectedNameId}, nameRecordsNumber, stringStorageOffset);
	}

	private String lookForNames (NameSelector s, int[] selectedNameId, 
			int nameRecordsNumber, int stringStorageOffset)
	throws UnsupportedFontException, InvalidFontException 
	{
		for (int i = 0; i < nameRecordsNumber; i++) 
		{
			int nameId = this.data.getuint16 (6 + 12*i + 6);
			for (int selectedNameIdIndex = 0; selectedNameIdIndex < selectedNameId.length; selectedNameIdIndex++)
			{
				if (nameId == selectedNameId[selectedNameIdIndex]) 
				{
					String charset = getCharsetName(this.data.getuint16 (6 + 12*i + 0), 
							this.data.getuint16 (6 + 12*i + 2));
					if (charset != null) 
					{
						int stringLength = this.data.getuint16 (6 + 12*i + 8);
						int stringOffset = stringStorageOffset + this.data.getuint16 (6 + 12*i + 10);
						try {
							String theString = bytesToUnicode(data.getBytes(stringOffset, stringLength), 0, stringLength, charset);
							if (s.nameFound (theString, nameId,
									this.data.getuint16 (6 + 12*i + 0),
									this.data.getuint16 (6 + 12*i + 2),
									this.data.getuint16 (6 + 12*i + 4))) 
							{
								return theString;
							}
						} catch (java.io.UnsupportedEncodingException e) {
							/* ignore string in unsupported encodings */ 
						} catch (IllegalStateException e) {
							/*  ignore invalid strings */ 
						} catch (CoderMalfunctionError e) {
							/* ignore invalid strings */
						} catch (MalformedInputException e) {
							/* ignore invalid strings */
						} catch (UnmappableCharacterException e) {
							/* ignore unmappable characters */
						} catch (ArrayIndexOutOfBoundsException e) {
							/* ignore bad string table indices */ 
						}
					}
				}
			}
		}

		return null;
	}

	private int pickFamilyNameSelector(int nameRecordsNumber)
	throws InvalidFontException {

		// Preferred family names can be found under nameId 16. If we find one
		// such name, then we report all the entries under nameId 16. Otherwise,
		// we report all the entries under nameId 1.

		int selectedNameId = PredefinedNames.FONT_FAMILY;
		for (int i = 0; i < nameRecordsNumber; i++) {
			int nameId = this.data.getuint16 (6 + 12*i + 6);
			if (nameId == PredefinedNames.PREFERRED_FAMILY) {
				selectedNameId = PredefinedNames.PREFERRED_FAMILY; 
				break; }}

		return selectedNameId;
	}

	String getFirstName(int selector)
	throws InvalidFontException, UnsupportedFontException {
		int nameRecordsNumber = this.data.getuint16 (2);
		int stringStorageOffset = this.data.getuint16 (4);
		return lookForNames (new NameSelector(){ 
			public boolean nameFound(String name, int nameID, int platformID, int encoding, int language) { 
				return true; }},
				selector, nameRecordsNumber, stringStorageOffset);
	}

	/**
	 * Return all of the CSS family names of this font.
	 * 
	 * This method removes duplicate names (e.g. if the same name occurs for two
	 * languages, or two different encodings). This method silently removes names
	 * which are in an unsupported encoding.
	 */
	public Set /*<String>*/ getCSSFamilyNames ()
	throws InvalidFontException, UnsupportedFontException 
	{
		int nameRecordsNumber = this.data.getuint16 (2);
		int stringStorageOffset = this.data.getuint16 (4);
		NameSetSelector nameSet = new NameSetSelector ();
		lookForNames (nameSet, PredefinedNames.PREFERRED_FAMILY, nameRecordsNumber, stringStorageOffset);
		lookForNames (nameSet, PredefinedNames.FONT_FAMILY, nameRecordsNumber, stringStorageOffset);		
		return nameSet.getSet();
	}

	public Set /*<NameSetEntry>*/ getMacFXGFamilyNames(ULocale locale)
	throws InvalidFontException, UnsupportedFontException 
	{
		Set /*<NameSetEntry>*/ fxgNames = null;
		int[] names = new int[] {PredefinedNames.COMPATIBLE_FULL_NAME, PredefinedNames.FULL_FONT_NAME};

		NameEntrySet nameSet = new NameEntrySet();
		nameSet.setNameIDs(names);
		nameSet.setPlatform(PlatformId.MACINTOSH);
		nameSet.setLocale(locale);
		this.getNamesExtended(names, nameSet);
		fxgNames = nameSet.getSet();

		// no Mac names in the font - try Windows names
		if (fxgNames == null)
		{
			// but only if there are no Mac names at all
			if (locale != null)
			{
				nameSet.setLocale(null);
				this.getNamesExtended(names, nameSet);
				Set tempSet = nameSet.getSet();
				if (!tempSet.isEmpty())
				{
					// there are mac names for other locales so none for this locale
					return null;
				}
			}

			nameSet.setLocale(locale);
			nameSet.setPlatform(PlatformID.MICROSOFT);
			this.getNamesExtended(names, nameSet);
			fxgNames = nameSet.getSet();
		}
		return fxgNames;
	}

	public Set /*<NameSetEntry>*/ getWindowsFXGFamilyNames(ULocale locale)
	throws InvalidFontException, UnsupportedFontException 
	{
		int[] names = new int[] {PredefinedNames.FONT_FAMILY};

		NameEntrySet nameSet = new NameEntrySet();
		nameSet.setPlatform(PlatformId.MICROSOFT);
		nameSet.setLocale(locale);
		this.getNamesExtended(names, nameSet);
		return nameSet.getSet();
	}

	public Set /*<NameSetEntry>*/ getMacPlatformNames(ULocale locale)
	throws InvalidFontException, UnsupportedFontException 
	{
		Set /*<NameSetEntry>*/ platformNames = null;
		int[] names = {PredefinedNames.FULL_FONT_NAME};
		NameEntrySet nameSet = new NameEntrySet();
		nameSet.setPlatform(PlatformID.MACINTOSH);
		nameSet.setLocale(locale);
		this.getNamesExtended(names, nameSet);
		platformNames = nameSet.getSet();

		// no Mac names in the font - try Windows names
		if (platformNames == null)
		{
			// but only if there are no Mac names at all
			if (locale != null)
			{
				nameSet.setLocale(null);
				this.getNamesExtended(names, nameSet);
				Set tempSet = nameSet.getSet();
				if (!tempSet.isEmpty())
				{
					// there are mac names for other locales so none for this locale
					return null;
				}
			}

			nameSet.setLocale(locale);
			nameSet.setPlatform(PlatformID.MICROSOFT);
			this.getNamesExtended(names, nameSet);
			platformNames = nameSet.getSet();
		}
		return platformNames;
	}

	public Set /*<NameSetEntry>*/ getWindowsPlatformNames(ULocale locale)
	throws InvalidFontException, UnsupportedFontException 
	{
		int[] names = {PredefinedNames.FULL_FONT_NAME};
		NameEntrySet nameSet = new NameEntrySet();
		nameSet.setPlatform(PlatformID.MICROSOFT);
		nameSet.setLocale(locale);
		this.getNamesExtended(names, nameSet);
		return nameSet.getSet();
	}

	private void getNamesExtended(int[] nameids, NameEntrySet nameSet) 
	throws InvalidFontException, UnsupportedFontException
	{
		int nameRecordsNumber = this.data.getuint16 (2);
		int stringStorageOffset = this.data.getuint16 (4);

		//lookForNames (nameSet, PredefinedNames.PREFERRED_FAMILY, nameRecordsNumber, stringStorageOffset);
		lookForNames(nameSet, nameids, nameRecordsNumber, stringStorageOffset);		
	}

	public String getPreferredCSSFamilyName()
	throws InvalidFontException, UnsupportedFontException {

		int nameRecordsNumber = this.data.getuint16 (2);
		int stringStorageOffset = this.data.getuint16 (4);

		CSSNameSelector s = new CSSNameSelector();
		lookForNames(s, PredefinedNames.FONT_FAMILY, nameRecordsNumber, stringStorageOffset);
		return s.getName();
	}

	public Set /*<String>*/ getPostscriptNames ()
	throws InvalidFontException, UnsupportedFontException {

		int nameRecordsNumber = this.data.getuint16 (2);
		int stringStorageOffset = this.data.getuint16 (4);

		NameSetSelector s = new NameSetSelector();
		lookForNames(s, PredefinedNames.POSTSCRIPT_NAME, nameRecordsNumber, stringStorageOffset);
		return s.getSet();
	}

	/**
	 * Fetch the nameID that should be used to fetch preferred family names
	 * for this font.
	 */
	public int selectFamilyNameId () throws InvalidFontException {
		return pickFamilyNameSelector(this.data.getuint16 (2));
	}

	/**
	 * Look for all of the names that have selectedNameId as their
	 * nameId. Call s's nameFound method for each one.  
	 * This method silently removes names
	 * which are in an unsupported encoding.
	 */
	public void enumerateNames(NameSelector s, int selectedNameId)
	throws InvalidFontException, UnsupportedFontException {
		int nameRecordsNumber = this.data.getuint16 (2);
		int stringStorageOffset = this.data.getuint16 (4);

		lookForNames(s, selectedNameId, nameRecordsNumber, stringStorageOffset);
	}

	//------------------------------------------------------------- embedding ---

	private boolean keepNameId(int nameId, boolean preserveTrademark, int[] lookupNameIds)
	{
		if (nameId == 0 || (preserveTrademark && nameId == 7))
			return true;

		if (lookupNameIds == null)
			return false;

		for (int i = 0; i < lookupNameIds.length; i++)
		{
			if (nameId == lookupNameIds[i])
				return true;
		}

		return false;
	}
	/**
	 * Add a subset of this name table to 'tables'.
	 * 
	 * The subset contains all the Copyright (nameId 0) strings of the original table.
	 * If the 'preserveTrademark' parameter is true, all of the Trademark strings are also
	 * preserved.
	 * 
	 * Those entries are necessary to follow the Adobe embedding guidelines (to
	 * preserve the record of the IP claims in the font).
	 * 
	 * Additionally, we may need to preserve other lookupNameIds (those in 'lookupNameIds')
	 * in order to preserve references made in other tables (notably by the 'size' feature in
	 * the GPOS).
	 * 
	 * It is also necessary to remove some entries: Acrobat/CoolType will reject
	 * fonts which have name entries with errors in them, i.e. if the bytes are
	 * not a valid encoding of some string. The particular case we ran into
	 * is a font stored under the name FgFumi.ttf, with font name entries
	 * under MacRoman, but where the data is actually some kind of Shift-JIS 
	 * encoding. Our subset does not contain a cmap, so CoolType gives a 
	 * non-Japanese primary script to this font, and consequently insists
	 * that font name entries be a valid encoding. (See also W#1223876.)
	 */
	public void subsetAndStream (Subset subset, boolean preserveTrademark, int[] lookupNameIds, Map tables) 
	throws UnsupportedFontException, InvalidFontException {

		int nbSubsetEntries = 0;
		int subsetStorageSize = 0;

		int nbOrigEntries = this.data.getuint16 (2);
		int origStringStorageOffset = this.data.getuint16 (4);

		for (int i = 0; i < nbOrigEntries; i++) {
			int nameId = this.data.getuint16 (6 + 12*i + 6);
			if (keepNameId(nameId, preserveTrademark, lookupNameIds)) {
				nbSubsetEntries++;
				subsetStorageSize += this.data.getuint16 (6 + 12*i + 8); }}

		int subsetStorageOffset = 6 + 12*nbSubsetEntries;
		OTByteArrayBuilder newData 
		= OTByteArray.getOTByteArrayBuilderInstance(subsetStorageOffset + subsetStorageSize);

		newData.setuint32 (0, 0); // format
		newData.setuint16 (2, nbSubsetEntries);
		newData.setuint16 (4, subsetStorageOffset);

		int curOffsetInSubset = 6;
		int curOffsetInSubsetStorage = 0;

		for (int i = 0; i < nbOrigEntries; i++) {
			int nameId = this.data.getuint16 (6 + 12*i + 6);
			if (keepNameId(nameId, preserveTrademark, lookupNameIds)) {
				int stringLengthInBytes = this.data.getuint16 (6 + 12*i + 8);
				newData.setuint16 (curOffsetInSubset, this.data.getuint16 (6 + 12*i));
				newData.setuint16 (curOffsetInSubset+2, this.data.getuint16 (6 + 12*i + 2));
				newData.setuint16 (curOffsetInSubset+4, this.data.getuint16 (6 + 12*i + 4));
				newData.setuint16 (curOffsetInSubset+6, nameId);
				newData.setuint16 (curOffsetInSubset+8, stringLengthInBytes);
				newData.setuint16 (curOffsetInSubset+10, curOffsetInSubsetStorage);
				curOffsetInSubset += 12;

				try {
					newData.replace(subsetStorageOffset + curOffsetInSubsetStorage, this.data,
							origStringStorageOffset + this.data.getuint16 (6 + 12*i + 10), stringLengthInBytes); }
				catch (ArrayIndexOutOfBoundsException e) {
					throw new InvalidFontException ("string " + i 
							+ " in name table extends beyond the end of the table"); }

				curOffsetInSubsetStorage += stringLengthInBytes; }}

		tables.put (new Integer (Tag.table_name), newData); 
	}

	public void stream(Map tables, boolean preserveTrademark, int[] lookupNameIds) 
	throws UnsupportedFontException, InvalidFontException {
		subsetAndStream(null, preserveTrademark, lookupNameIds, tables);
	}

	public String selectName(int nameid) 
	throws UnsupportedEncodingException, InvalidFontException {

		String name;
		name = getName(Name.PlatformId.MICROSOFT, Name.MicrosoftEncodingId.UTF16_BE_BMP_ONLY, Name.MicrosoftLCID.ENGLISH_UNITED_STATES.getLanguageCode(), nameid);
		if (name != null) return name;
		name = getName(Name.PlatformId.MICROSOFT, Name.MicrosoftEncodingId.SYMBOL, Name.MicrosoftLCID.ENGLISH_UNITED_STATES.getLanguageCode(), nameid);
		if (name != null) return name;
		getName(Name.PlatformId.MACINTOSH, Name.MacintoshEncodingId.ROMAN, 0, nameid);
		return name;

	}
}
