/*
 * File: PDFEncodingBuilderImpl.java
 * 
 *	ADOBE CONFIDENTIAL
 *	___________________
 *
 *	Copyright 2005 Adobe Systems Incorporated
 *	All Rights Reserved.
 *
 *	NOTICE: All information contained herein is, and remains the property of
 *	Adobe Systems Incorporated and its suppliers, if any. The intellectual
 *	and technical concepts contained herein are proprietary to Adobe Systems
 *	Incorporated and its suppliers and may be covered by U.S. and Foreign
 *	Patents, patents in process, and are protected by trade secret or
 *	copyright law. Dissemination of this information or reproduction of this
 *	material is strictly forbidden unless prior written permission is
 *      obtained from Adobe Systems Incorporated.
 *
 */

package com.adobe.fontengine.font.opentype;

import java.util.ArrayList;
import java.util.List;
import java.util.ListIterator;
import java.util.Iterator;

import com.adobe.fontengine.font.InvalidFontException;
import com.adobe.fontengine.font.PDFEncodingBuilder;
import com.adobe.fontengine.font.Subset;
import com.adobe.fontengine.font.UnsupportedFontException;

final public class PDFEncodingBuilderImpl extends PDFEncodingBuilder implements Cmap.CmapSelector {

  private static class RevMap {
    final Encoding encoding;
    final int[] revMap;
    final int numGlyphsCovered;

    RevMap (Encoding e, int[] revMap, int numGlyphsCovered) {
      this.revMap = revMap;
      this.numGlyphsCovered = numGlyphsCovered;
      this.encoding = e;
    }
  }

  private final List /* RevMap */ revMapList;
  private Subset s;
  private OpenTypeFont f; // only temporarily non-null during construction.

  public PDFEncodingBuilderImpl (OpenTypeFont font, Subset s)
  throws UnsupportedFontException, InvalidFontException {
    this.s = s;
    this.f = font;
    revMapList = new ArrayList(5);
    font.cmap.enumerateCmaps(this);
    this.f = null;
  }

  public int getCodePoint (int subsetGlyphID, Encoding e) {
    Iterator iter = revMapList.iterator();
    int fullGid = s.getFullGid (subsetGlyphID);
    while (iter.hasNext ()) {
      RevMap map = (RevMap)iter.next();
      if (map.encoding == e && map.revMap[fullGid] != -1) {
        return map.revMap[fullGid]; }}
    return Integer.MAX_VALUE;
  }

  public Encoding getEncoding (int subsetGlyphID) {
    Iterator iter = revMapList.iterator ();
    int fullGid = s.getFullGid (subsetGlyphID);
    while (iter.hasNext ()) {
      RevMap map = (RevMap)iter.next();
      if (map.revMap[fullGid] != -1) {
        return map.encoding; }}
    return null;
  }

  public void cmapFound (int pid, int eid, int index)
  throws UnsupportedFontException, InvalidFontException {
    int[] invertedCmap = null;
    Encoding e = null;

    switch (pid) {
      case Cmap.PlatformID.MICROSOFT: {
        switch (eid) {
          case Cmap.MS_EncodingID.UNICODE_BMP:
          case Cmap.MS_EncodingID.UNICODE_FULL: {
            e = Encoding.UNICODE;
            break; }

          case Cmap.MS_EncodingID.BIG5: {
            e = Encoding.MS_BIG5;
            break; }

          case Cmap.MS_EncodingID.JOHAB: {
            e = Encoding.MS_JOHAB;
            break; }

          case Cmap.MS_EncodingID.PRC: {
            e = Encoding.MS_PRC;
            break; }

          case Cmap.MS_EncodingID.SHIFTJIS: {
            e = Encoding.MS_SHIFT_JIS;
            break; }

          case Cmap.MS_EncodingID.WANSUNG: {
            e = Encoding.MS_WANSUNG;
            break; }}
        
        if (e != null) {
          invertedCmap = f.cmap.glyph2char (f.getNumGlyphs(), index); }
        break; }

      case Cmap.PlatformID.UNICODE: {
        e = Encoding.UNICODE;
        invertedCmap = f.cmap.glyph2char (f.getNumGlyphs(), index);
        break; }}

    if (e != null) {
      int numGlyphsCovered = 0;
      for (int i = 0; i < s.getNumGlyphs(); i++) {
        if (invertedCmap [s.getFullGid(i)] != -1) {
          numGlyphsCovered++; }}


      ListIterator iter = this.revMapList.listIterator();
      while (iter.hasNext ()) {
        RevMap map = (RevMap)iter.next();

        if (map.numGlyphsCovered <= numGlyphsCovered) {
          break; }}

      RevMap newMap = new RevMap (e, invertedCmap, numGlyphsCovered);
      if (iter.hasPrevious ()) {
        iter.previous(); }
      else {
        iter = this.revMapList.listIterator(); }

      iter.add(newMap); }
  }
}
