/**
 * 
 */
package com.adobe.fontengine.font.opentype;

import java.util.Arrays;
import java.util.HashMap;
import java.util.Map;

import com.adobe.fontengine.font.InvalidFontException;
import com.adobe.fontengine.font.Subset;
import com.adobe.fontengine.font.UnsupportedFontException;
import com.adobe.fontengine.font.opentype.LayoutTable.CoverageConsumer;
import com.adobe.fontengine.font.opentype.LookupTableSubsetter.LookupSubtableGenerator;
import com.adobe.fontengine.font.opentype.OTByteArray.OTByteArrayBuilder;

abstract class SetGenerator implements CoverageConsumer, LookupSubtableGenerator
{
	protected final Subset subset;
	protected final OTByteArrayBuilder builder;
	protected final LayoutTable origTable;
	protected final int newSTOffset;
	protected final int origSTOffset;
	protected int subtableSize;
	protected int thisSet = 0;
	
	/*
	 * How many bytes from the start of the subtable is the set count? Most subtables have it 4 bytes in, but 
	 *  tables with class def tables have it 6 bytes in...
	 */
	protected final int origSetCountDelta;
	
	/*
	 * Class based tables need to keep the original indexing since the classes aren't renumbered. With other table
	 * types, sets that can no longer be reached can entirely go away and the ones that remain may need to be written out 
	 * in a different order (eg, if subset gids are in a different order than the original gids).
	 */
	protected final boolean preserveRuleSetIndexing;
	private int[] subsetGIDToOrigSetOffset;
	private Map /*<Integer, Integer>*/ origSetOffsetToNewSetOffset;
	
	/*
	 * There are 2 kinds of LookupTable sets: those whose set elements have variable size (eg, Ligatures) and those
	 * with fixed size (eg, PairPos format1). For the former, set elements have offsets that point to the elements. For the 
	 * latter, the records are stored directly in the set. The boolean is true for the former case, false for the latter.
	 */
	private final boolean setElementsHaveOffsets;
	
	SetGenerator(LayoutTable origTable, Subset subset, OTByteArrayBuilder newData, int newSTOffset, 
			int origSTOffset, int setCount, int origSetCountDelta, 
			boolean preserveRuleSetIndexing, boolean setElementsHaveOffsets) throws InvalidFontException
	{
		this.subset = subset;
		this.builder = newData;
		this.newSTOffset = newSTOffset;
		this.origSTOffset = origSTOffset;
		this.subtableSize =  origSetCountDelta + 2 + 2*setCount;
		this.origSetCountDelta = origSetCountDelta;
		this.preserveRuleSetIndexing = preserveRuleSetIndexing;
		this.origTable = origTable;
		this.setElementsHaveOffsets = setElementsHaveOffsets;
		this.origSetOffsetToNewSetOffset = new HashMap();
		
		if (preserveRuleSetIndexing)
			setCount = origTable.data.getuint16(origSTOffset + origSetCountDelta);
		else 
		{
			subsetGIDToOrigSetOffset = new int[subset.getNumGlyphs()];
			Arrays.fill(subsetGIDToOrigSetOffset, -1);
		}
		
		builder.ensureCapacity(newSTOffset + origSetCountDelta + 2 + 2*setCount);
		
		for (int i = 0; i < setCount; i++)
		{
			builder.setuint16(newSTOffset + origSetCountDelta + 2 + 2*i, 0);
		}
	}
	
	public boolean glyphInfo(int gid, int coverageIndex) throws InvalidFontException, UnsupportedFontException {
		
		int ruleSetIndex = whichRuleSetIndexApplies(gid, coverageIndex);
		int origSetOffset = origTable.data.getOffset(origSTOffset, origSetCountDelta + 2 + 2*ruleSetIndex);
		if (preserveRuleSetIndexing)
		{
			writeSubtableEntry(ruleSetIndex, origSetOffset);
		}
		else
		{
			subsetGIDToOrigSetOffset[subset.getExistingSubsetGid(gid)] = origSetOffset;
		}
			
		return true;
	}
	
	public int writeSubtable() throws InvalidFontException, UnsupportedFontException
	{
		if (preserveRuleSetIndexing)
			return subtableSize; // we wrote it out as we iterated over the coverage
		
		for (int i = 0, j=0; i < subsetGIDToOrigSetOffset.length; i++)
		{
			if (subsetGIDToOrigSetOffset[i] != -1)
			{
				writeSubtableEntry(j, subsetGIDToOrigSetOffset[i]);
				j++;
			}
		}
		
		return subtableSize;
	}
	
	private void writeSubtableEntry(int ruleSetIndex, int origSetOffset) throws InvalidFontException, UnsupportedFontException
	{

		// if we have already written out this rule set, skip it and go on.
		if (builder.getuint16(newSTOffset + origSetCountDelta + 2 + 2*ruleSetIndex) != 0)
			return;
		
		// class based sets can have some original offsets being NULL. If that is the case, there is nothing
		// to write out.
		if (origSetOffset != 0)
		{
			Integer newOffset = (Integer)origSetOffsetToNewSetOffset.get(new Integer(origSetOffset));
			if (newOffset == null)
			{
				origSetOffsetToNewSetOffset.put(new Integer(origSetOffset), new Integer(subtableSize));
				builder.setuint16(newSTOffset + origSetCountDelta + 2 + 2*ruleSetIndex, subtableSize); // set offset
				subtableSize += writeSet(newSTOffset + subtableSize, origSetOffset);
			} else
			{
				builder.setuint16(newSTOffset + origSetCountDelta + 2 + 2*ruleSetIndex, newOffset.intValue()); // set offset
			}
			
		} 
	}
	
	private int writeSet(int newSetOffset, int origSetOffset) throws InvalidFontException, UnsupportedFontException
	{
		// figure out which elements of this set we will keep and just emit them.
		
		boolean[] keepMember = computeMembersToKeep(newSetOffset, origSetOffset);
		int newMemberCount = 0;
		for (int i = 0; i < keepMember.length; i++)
		{
			if (keepMember[i])
				newMemberCount++;
		}
		int setSize;
		
		if (setElementsHaveOffsets)
			setSize = 2 + 2*newMemberCount;
		else
			setSize = 2;
		
		
		builder.ensureCapacity(newSetOffset + setSize);
		builder.setuint16(newSetOffset, newMemberCount);
		
		for (int j = 0, i = 0; i < keepMember.length; i++)
		{
			if (keepMember[i])
			{
				int origOffset;
				if (setElementsHaveOffsets)
				{
					builder.setuint16(newSetOffset + 2 + 2*j, setSize );
					origOffset = origTable.data.getOffset(origSetOffset, 2 + 2*i);
				} else
					origOffset = origSetOffset + 2 + getOrigRecordSize()*i;
				
				setSize += writeMember(newSetOffset + setSize, origOffset);
				j++;
			}
		}
		return setSize;
	}

	protected int writeByteArrayAtOffset(int newOffset, OTByteArray byteArrayToWrite) throws InvalidFontException
	{
		builder.replace(newOffset, byteArrayToWrite, 0, byteArrayToWrite.getSize());
		return byteArrayToWrite.getSize();
	}
	
	abstract int whichRuleSetIndexApplies(int gid, int coverageIndex) throws InvalidFontException;
	abstract boolean[] computeMembersToKeep(int newSetOffset, int origSetOffset) throws InvalidFontException, UnsupportedFontException;
	abstract int writeMember(int newMemberOffset, int origMemberOffset) throws InvalidFontException, UnsupportedFontException;
	
	/**
	 * Needs to be implemented iff setElementsHaveOffsets == false
	 */
	abstract int getOrigRecordSize();
}