/*
 *
 *	File: TTParser.java
 *
 * ****************************************************************************
 *
 *	ADOBE CONFIDENTIAL
 *	___________________
 *
 *	Copyright 2004-2005 Adobe Systems Incorporated
 *	All Rights Reserved.
 * 
 *	NOTICE: All information contained herein is, and remains the property of
 *	Adobe Systems Incorporated and its suppliers, if any. The intellectual
 *	and technical concepts contained herein are proprietary to Adobe Systems
 *	Incorporated and its suppliers and may be covered by U.S. and Foreign
 *	Patents, patents in process, and are protected by trade secret or
 *	copyright law. Dissemination of this information or reproduction of this
 *	material is strictly forbidden unless prior written permission is obtained
 *	from Adobe Systems Incorporated.
 *
 */

package com.adobe.fontengine.font.opentype;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import com.adobe.fontengine.font.InvalidFontException;
import com.adobe.fontengine.font.InvalidGlyphException;
import com.adobe.fontengine.font.Matrix;
import com.adobe.fontengine.font.OutlineConsumer;
import com.adobe.fontengine.font.UnsupportedFontException;

/**
 * Converts truetype outlines to unhinted cubic beziers with type1
 * winding order. Overlapping paths are NOT corrected in this transformation.
 */
final class TTParser {

  private static class Coord {
    int x;
    int y;
    int flags;
        
    private void setXY(Coord c) {
    	this.x = c.x;
    	this.y = c.y;
    }
    
    private void setXYAtMidPoint(Coord a, Coord b) {
  	  this.x = (int) Math.round((a.x + b.x)/ 2.0);
	  this.y = (int) Math.round((a.y + b.y)/ 2.0);	      	
    }
    
    public String toString() {
    	String fStr = ((flags & Glyf.CoordFlags.ON_CURVE) != 0) ? "OnCurve" : "OffCurve";
    	return ("(" + x + "," + y + ") flags: " + fStr );
    }

  }

  List /*<Coord>*/ points = new ArrayList(); // the points in absolute coordinates
  List /*<Integer>*/ endPoints = new ArrayList(); // indices of last point in each contour

  public static final int EXACT_PATH = 1;
  public static final int APPROX_PATH = 2;
  
  private void getSimpleGlyph(Glyf glyf, int index, int numContours)
  throws InvalidFontException {
    if (numContours == 0)
      return;

    int startNumPoints = points.size();

    int lastEndPoint = 0;
    for (int i = 0; i < numContours; i++) {
      int thisEndPoint = glyf.glyf.data.getuint16 (index);
      index += 2;
      if (thisEndPoint < lastEndPoint) {
        throw new InvalidGlyphException ("endpoints in a simple TT glyph must be increasing"); }
      lastEndPoint = thisEndPoint;
      endPoints.add (new Integer (thisEndPoint + startNumPoints)); }

    int numPoints = ((Integer)endPoints.get(endPoints.size()-1)).intValue() + 1 - startNumPoints;

    // skip the instructions
    index += glyf.glyf.data.getuint16(index) + 2;

    // fetch the flags
    for (int i = 0; i < numPoints; i++) {
      Coord coord = new Coord();
      points.add(coord);
      coord.flags = glyf.glyf.data.getuint8(index++);

      if ((coord.flags & Glyf.CoordFlags.REPEAT) != 0) {
        int flags = coord.flags;
        int numRepeats = glyf.glyf.data.getuint8(index++);

        while (numRepeats-- > 0) {
          coord = new Coord();
          points.add(coord);
          coord.flags = flags;
          i++; }}}

    // fetch the x coords
    short x = 0;
    for (int i = 0; i < numPoints; i++) {

      Coord coord = (Coord)points.get(i + startNumPoints);
      if ((coord.flags & Glyf.CoordFlags.X_SHORT_VECTOR) != 0) {
        if ((coord.flags & Glyf.CoordFlags.THIS_X_IS_SAME) == 0) {
          x -= glyf.glyf.data.getuint8(index++); }
        else {
          x += glyf.glyf.data.getuint8(index++); }}
      else if ((coord.flags & Glyf.CoordFlags.THIS_X_IS_SAME) == 0) {
        x += glyf.glyf.data.getint16(index);
        index+=2; }

      coord.x = x; }

    // fetch the y coords
    short y = 0;
    for (int i = 0; i < numPoints; i++) {

      Coord coord = (Coord)points.get(i + startNumPoints);
      if ((coord.flags & Glyf.CoordFlags.Y_SHORT_VECTOR) != 0) {
        if ((coord.flags & Glyf.CoordFlags.THIS_Y_IS_SAME) == 0) {
          y -= glyf.glyf.data.getuint8(index++); }
        else {
          y += glyf.glyf.data.getuint8(index++); }}
      else if ((coord.flags & Glyf.CoordFlags.THIS_Y_IS_SAME) == 0) {
        y += glyf.glyf.data.getint16(index);
        index+=2; }

      coord.y = y; }
  }

  private int getCompositeGlyph(Glyf glyf, int index, int thisGid)
  throws UnsupportedFontException, InvalidFontException {
    int metricsGid = thisGid;
    for (;;) {
      int x, y, i;
      double a,b,c,d;
      Matrix transMatrix = null;
      int firstCoordToTransform;
      int savedPosInGlyfTable;

      int flags = glyf.glyf.data.getuint16 (index);
      index +=2;
      int gid = glyf.glyf.data.getuint16 (index);
      index +=2;

      boolean translate = (flags & Glyf.Flags.ARGS_ARE_XY_VALUES) != 0;

      if ((flags & Glyf.Flags.ARG_1_AND_2_ARE_WORDS) != 0) {
        if (translate) {
          x = glyf.glyf.data.getint16 (index);
          y = glyf.glyf.data.getint16 (index+2);  }
        else {
          // these aren't really an x/y in this case...they are indices into
          // the points array
          x = glyf.glyf.data.getuint16 (index);
          y = glyf.glyf.data.getuint16 (index+2); }

        index += 4; }
      
      else {
        if (translate) {
          x = glyf.glyf.data.getint8 (index);
          y = glyf.glyf.data.getint8 (index+1); }
        else {
          // these aren't really an x/y in this case...they are indices into
          // the points array
          x = glyf.glyf.data.getuint8 (index);
          y = glyf.glyf.data.getuint8 (index+1); }

        index += 2; }

      if ((flags & Glyf.Flags.WE_HAVE_A_SCALE) != 0) {
        a = glyf.glyf.data.getint16 (index)/16384.0;
        index += 2;
        transMatrix = new Matrix (a,0,0,a,0,0); }
      else if ((flags & Glyf.Flags.WE_HAVE_A_TWO_BY_TWO) != 0) {
        a = glyf.glyf.data.getint16 (index)/16384.0; index += 2;
        b = glyf.glyf.data.getint16 (index)/16384.0; index += 2;
        c = glyf.glyf.data.getint16 (index)/16384.0; index += 2;
        d = glyf.glyf.data.getint16 (index)/16384.0; index += 2;
        transMatrix = new Matrix(a,b,c,d,0,0); }
      else if ((flags & Glyf.Flags.WE_HAVE_AN_X_AND_Y_SCALE)!= 0) {
        a = glyf.glyf.data.getint16 (index)/16384.0;
        d = glyf.glyf.data.getint16 (index+2)/16384.0;
        transMatrix = new Matrix (a,0,0,d,0,0);
        index += 4; } 

      if ((flags & Glyf.Flags.USE_MY_METRICS) != 0) {
        metricsGid = gid; }

      firstCoordToTransform = points.size();
      savedPosInGlyfTable = index;

      index = glyf.getGlyphLocation (gid);
      if (glyf.getGlyphLocation (gid+1) != index) {
    	  
	      int numberOfContours = glyf.glyf.data.getint16(index);
	      index += 10;
	
	      if (numberOfContours < 0) {
	        metricsGid = getCompositeGlyph (glyf, index, gid); }
	      else {
	        getSimpleGlyph(glyf, index, numberOfContours); }
	
	      if (!translate) {
	        /* Convert matched points to a translation */
	        Coord first = (Coord)points.get(x);
	        Coord second = (Coord)points.get(y + firstCoordToTransform);
	        x = first.x - second.x;
	        y = first.y - second.y; }
	
	      if (transMatrix != null) {
	        /* Apply transformation to component */
	        for (i = firstCoordToTransform; i < points.size(); i++) {
	          Coord coord = (Coord) points.get (i);
	          int origX = coord.x;
	          coord.x = (int)Math.round (transMatrix.applyToXYGetX(coord.x, coord.y) + x);
	          coord.y = (int)Math.round (transMatrix.applyToXYGetY(origX, coord.y) + y); }}
	      else if (x != 0 || y != 0) {
	        /* Apply translation to component */
	        for (i = firstCoordToTransform; i < points.size(); i++) {
	          Coord coord = (Coord)points.get (i);
	          coord.x += x;	
	          coord.y += y; }}}

      if ((flags & Glyf.Flags.MORE_COMPONENTS) == 0) {
        return metricsGid; }

      index = savedPosInGlyfTable; }
  }

  private void adjustPointsBySideBearing(int gid, Hmtx hmtx, int xMin)
  throws InvalidFontException {
    int lsb = hmtx.getLeftSideBearing(gid);

    int xScale = lsb - xMin;
    if (xScale != 0) {
      Iterator iter = points.iterator  ();
      while (iter.hasNext ()) {
        Coord c = (Coord) iter.next ();
        c.x += xScale; }}
  }

  private void addCurve(Coord[] pnts, OutlineConsumer consumer) {
	  consumer.curveto(Math.round((pnts[0].x + 2*pnts[1].x)/3.0), 
			  Math.round((pnts[0].y + 2*pnts[1].y)/3.0),
			  Math.round ((2*pnts[1].x + pnts[2].x)/3.0),
			  Math.round((2*pnts[1].y + pnts[2].y)/3.0),
			   pnts[2].x, pnts[2].y);
  }
  

  /* Test if curve pair should be combined. If true, callback combined curve else
  callback first curve of pair. Return true if curves combined else false. */
  private boolean combinePair(Coord[] pnts, OutlineConsumer consumer) {
	  
 	double a = pnts[3].y - pnts[1].y;
	double b = pnts[1].x - pnts[3].x;
	if ((a != 0 || pnts[1].y != pnts[2].y) && (b != 0 || pnts[1].x != pnts[2].x)) {
		
		/* Not a vertical or horizontal join... */
		double absq = a*a + b*b;
		if (absq != 0) {
			
			double sr = a*(pnts[2].x - pnts[1].x) + b*(pnts[2].y - pnts[1].y);
			if ((sr*sr)/absq < 1) {
				
				/* ...that is straight... */
				if ((a*(pnts[0].x - pnts[1].x) + b*(pnts[0].y - pnts[1].y) < 0) ==
					(a*(pnts[4].x - pnts[1].x) + b*(pnts[4].y - pnts[1].y) < 0)) {
					
					/* ...and without inflexion... */
					double d0 = (pnts[2].x - pnts[0].x)*(pnts[2].x - pnts[0].x) + 
						(pnts[2].y - pnts[0].y)*(pnts[2].y - pnts[0].y);
					double d1 = (pnts[4].x - pnts[2].x)*(pnts[4].x - pnts[2].x) + 
						(pnts[4].y - pnts[2].y)*(pnts[4].y - pnts[2].y);
					if (d0 <= 3*d1 && d1 <= 3*d0) {
						
						/* ...and small segment length ratio; combine curve */
						consumer.curveto(
										Math.round((4*pnts[1].x - pnts[0].x)/3.0),
										Math.round((4*pnts[1].y - pnts[0].y)/3.0),
										Math.round((4*pnts[3].x - pnts[4].x)/3.0),
										Math.round((4*pnts[3].y - pnts[4].y)/3.0),
										pnts[4].x, pnts[4].y);
						pnts[0].setXY(pnts[4]);
						return true;
						}
					}
				}
			}
		}

	/* Callback first curve then replace it by second curve */
	addCurve(pnts, consumer);
	pnts[0].setXY(pnts[2]);
	pnts[1].setXY(pnts[3]);
	pnts[2].setXY(pnts[4]);

	return false;
	}

 
  /* Convert path using an approximate conversion of quadratic curve segments to
  cubic curve segments and send it via callbacks. 

  Points are either on or off the curve and are accumulated in an array until
  there is enough context to make a decision about how to convert the point
  sequence. This is implemented using a state machine as follows:

  state	sequence		points
  			0=off,1=on		accumulated
  0		1				0
  1		1 0				0-1
  2		1 0 0			0-3 (p[2] is mid-point of p[1] and p[3])
  3		1 0 1			0-2
  4		1 0 1 0			0-3

  One curve is described by points 0-2 and another by point 2-4. Point 5 is a
  temporary. 

  States 2 and 4 are complicated by the fact that a test must be performed to
  decide if the 2 curves decribed by the point data can be combined into a
  single curve or must retained as 2 curves. */
  private void 	emitContoursApproxPath(OutlineConsumer consumer) {
	  int contour;
	  int startOfContour; // index of first point of current contour
      Coord[] p = new Coord[6];
      for (int i = 0; i < 6; i++) {
    	  p[i] = new Coord();
      }
      
	  for (startOfContour = 0, contour = 0; contour < endPoints.size(); contour++) {
	      int endOfContour = ((Integer)endPoints.get(contour)).intValue();
	      if (startOfContour >= endOfContour) {
	          startOfContour = endOfContour + 1;
	          continue; } // no points in contour. go to the next contour

	      Coord current = null;
	      int state = 0;
	      int numOfCoords = endOfContour - startOfContour + 1;
	      Coord begin = (Coord)points.get(startOfContour);
	      Coord end = (Coord)points.get(endOfContour);
	      int currentIndex = startOfContour;

	      if ((begin.flags & Glyf.CoordFlags.ON_CURVE) != 0) {
	    	  current = begin;
	    	  p[0].setXY(current);
	      } else if ((end.flags & Glyf.CoordFlags.ON_CURVE) != 0) {
	    	  current = end;
	    	  currentIndex = endOfContour;
	    	  p[0].setXY(current);
	      } else {
	    	  //start at mid-point
	    	  current = end;
	    	  currentIndex = endOfContour;
	    	  numOfCoords++;
	    	  p[0].setXYAtMidPoint(begin, end);
	      }
	      
	      consumer.moveto(p[0].x, p[0].y);
	      
	      while (numOfCoords > 0) {
	    	  numOfCoords--;
	    	  //Advance to next point
	    	  if (current == end) {
	    		  current =  begin; 
		    	  currentIndex = startOfContour;
	    	  }
	    	  else { 
	    		  currentIndex++;
	    		  current = (Coord)points.get(currentIndex);
	    	  }

	    	  if ((current.flags & Glyf.CoordFlags.ON_CURVE) != 0)
	    		  /* On-curve */
	    		  switch (state) {
	    		  case 0:
	    			  if (numOfCoords > 0) {
	    				  consumer.lineto(current.x, current.y);
	    		    	  p[0].setXY(current);
	    				  /* stay in state 0 */
	    			  }
	    			  break;
	    		  case 1:
	    	    	  p[2].setXY(current);
	    			  state = 3;
	    			  break;
	    		  case 2:
	    	    	  p[4].setXY(current);
	    			  state = combinePair(p, consumer)? 0: 3;
	    			  break;
	    		  case 3:
	    			  addCurve(p, consumer);
	    			  if (numOfCoords > 0) {
	    				  consumer.lineto(current.x, current.y);
	    		    	  p[0].setXY(current); }
	    			  state = 0;
	    			  break;
	    		  case 4:
	    	    	  p[4].setXY(current);
	    			  state = combinePair(p, consumer)? 0: 3;
	    			  break;
	    		  }
	    	  else
	    		  /* Off-curve */
	    		  switch (state) {
	    		  case 0:
	    	    	  p[1].setXY(current);
	    			  state = 1;
	    			  break;
	    		  case 1:
	    	    	  p[3].setXY(current);
	    	    	  p[2].setXYAtMidPoint(p[1], p[3]);
	    			  state = 2;
	    			  break;
	    		  case 2:
	    	    	  p[5].setXY(current);
	    	    	  p[4].setXYAtMidPoint(p[3], p[5]);
	    			  if (combinePair(p, consumer)) {
	    				  p[1].setXY(p[5]);
	    				  state = 1;
	    			  } else {
	    				  p[3].setXY(p[5]);
	    				  state = 4;
	    			  }
	    			  break;
	    		  case 3:
	    	    	  p[3].setXY(current);
	    			  state = 4;
	    			  break;
	    		  case 4:
	    	    	  p[5].setXY(current);
	    	    	  p[4].setXYAtMidPoint(p[3], p[5]);
	    			  if (combinePair(p, consumer)) {
	    		    	  p[1].setXY(p[5]);
	    				  state = 1;
	    			  }else {
	    		    	  p[3].setXY(p[5]);
	    				  state = 2;
	    			  }
	    			  break;
	    		  }
	      }

	      /* Finish up */
	      switch (state) {
	      case 2:
	    	  p[3].setXY(current);
	    	  p[2].setXYAtMidPoint(p[1], p[3]);
	    	  /* Fall through */
	      case 3:
	      case 4:
	    	  addCurve(p, consumer);
	    	  break;
	      }

	      startOfContour = endOfContour+1;		

	  }
	  consumer.endchar();	  
  }


  private void emitContoursExactPath (OutlineConsumer consumer) {
    int contour;
    int startOfContour; // index of first point of current contour
 
    for (startOfContour = 0, contour = 0; contour < endPoints.size(); contour++) {
      double x1 = 0, y1 = 0; // first control point of bezier
      int endOfContour = ((Integer)endPoints.get(contour)).intValue();

      if (startOfContour >= endOfContour) {
        startOfContour = endOfContour + 1;
        continue; } // no points in contour. go to the next contour

      int start = startOfContour; // first control point we write out.
      boolean onCurveStartNotFound = true;
      Coord current = null, next = null;
      int currentIndex, nextIndex;

      // look for the first oncurve point. that is the first one we will
      // emit. moveto that point
      while (onCurveStartNotFound && start < endOfContour) {
        current = (Coord)points.get(start);
        if ((current.flags & Glyf.CoordFlags.ON_CURVE) != 0) {
          consumer.moveto(current.x, current.y);
          onCurveStartNotFound = false; }
        else {
          start++; }}

      if (onCurveStartNotFound) {
        Coord c2 = (Coord)points.get(endOfContour);

        start = startOfContour;
        current = (Coord)points.get(start);

        // start in the middle of the start and end points (an implicit
        // oncurve point).
        consumer.moveto ((current.x + c2.x)/2.0, (current.y + c2.y)/2.0);

        // find first point of next bezier
        x1 = (c2.x + 5*current.x)/6.0;
        y1 = (c2.y + 5*current.y)/6.0; }

      nextIndex = currentIndex = start;

      do {
        if (nextIndex++ == endOfContour) {
          nextIndex = startOfContour; }

        next = (Coord)points.get(nextIndex);

        if ((current.flags & Glyf.CoordFlags.ON_CURVE) != 0) {
          if ((next.flags & Glyf.CoordFlags.ON_CURVE) != 0) {
            // on on...a line
            consumer.lineto(next.x, next.y); }
          else  {
            // on off...the start of a new bezier
            x1 = next.x;
            y1 = next.y; }}
        else  {
          if ((next.flags & Glyf.CoordFlags.ON_CURVE) != 0) {
            // off on...the end of a bezier
            consumer.curveto(x1, y1, next.x, next.y); }
          else {
            // off off. the end of a bezier (via an implicit on-curve point)
            // and the start of a new one
            consumer.curveto(x1, y1, (current.x + next.x)/2.0, (current.y + next.y)/2.0);
            x1 = next.x;
            y1 = next.y; } }

        currentIndex = nextIndex;
        current = (Coord)points.get(currentIndex);

      } while (nextIndex != start);

      
      startOfContour = endOfContour+1; }
    
      consumer.endchar();
  }

  private void emitContours (OutlineConsumer consumer, int pathType) {

	  // reverse the points...tt outlines have the opposite winding order
	  // of type1 beziers.
	  int startOfContour = 0; // index of first point of current contour
	  for (int contour = 0; contour < endPoints.size(); contour++) {
		  int endOfContour = ((Integer)endPoints.get(contour)).intValue();
		  if (startOfContour >= endOfContour) {
			  startOfContour = endOfContour + 1;
			  continue; }

		  int lastPoint = endOfContour;
		  int swapLocation = endOfContour + startOfContour + 1 - lastPoint;
		  for (;lastPoint > swapLocation; lastPoint--, swapLocation++) {
			  Object swapObj = points.set(swapLocation, points.get(lastPoint));
			  points.set(lastPoint, swapObj); }
		  startOfContour = endOfContour + 1; 
	  }

	  if (pathType == EXACT_PATH) {
		  emitContoursExactPath(consumer);
	  } else {
		  emitContoursApproxPath(consumer);
	  }
 }
	  
  void parse(OpenTypeFont font, int gid, OutlineConsumer consumer, int pathType)
  throws UnsupportedFontException, InvalidGlyphException {
    Glyf glyf = font.glyf;
    points.clear();
    endPoints.clear();

    int indexIntoData;
    
    try {
    	indexIntoData = glyf.getGlyphLocation (gid);
    } catch (InvalidFontException e) {
    	consumer.endchar();
    	return; } // If the location is past the end of the table, consider it as having no outlines.
    
    try { 
    	int numberOfContours;

	    try {
		    if (glyf.getGlyphLocation (gid+1) == indexIntoData) {
		      consumer.endchar();
		      return; }
		    
		    numberOfContours = glyf.glyf.data.getint16(indexIntoData);
	    } catch (InvalidFontException e)
	    {
	    	consumer.endchar();
	    	return; // if the loca takes us past the end of the glyf table, consider it having no outlines.
	    }
	    
	    int metricsGID = gid;
	    
	    int xMin;
	
	    indexIntoData += 10;
	    if (font.head == null) {
	      throw new InvalidFontException ("OpenType font with 'glyf' table needs a 'head' table"); }
	    consumer.setMatrix (new Matrix (1.0d / font.head.getUnitsPerEm (), 0.0d, 0.0d, 1.0d / font.head.getUnitsPerEm (), 0.0d, 0.0d));

	    if (numberOfContours < 0) {
	      metricsGID = getCompositeGlyph (glyf, indexIntoData, gid);
	      xMin = glyf.glyf.data.getint16  (glyf.getGlyphLocation(metricsGID) + 2); }
	    else {
	      getSimpleGlyph (glyf, indexIntoData, numberOfContours);
	      xMin = glyf.glyf.data.getint16 (indexIntoData-8); }
	
	    adjustPointsBySideBearing(metricsGID, font.hmtx, xMin);
	
	    emitContours(consumer, pathType);
    } catch(InvalidFontException e) {
    	throw new InvalidGlyphException(e);
    }
  }
  
  void parse(OpenTypeFont font, int gid, OutlineConsumer consumer)
  throws UnsupportedFontException, InvalidGlyphException {
	  parse(font, gid, consumer, EXACT_PATH);
  }

  
}
