/*
 * File: TTScaler.java
 * 
 *	ADOBE CONFIDENTIAL
 *	___________________
 *
 *	Copyright 2006 Adobe Systems Incorporated
 *	All Rights Reserved.
 *
 *	NOTICE: All information contained herein is, and remains the property of
 *	Adobe Systems Incorporated and its suppliers, if any. The intellectual
 *	and technical concepts contained herein are proprietary to Adobe Systems
 *	Incorporated and its suppliers and may be covered by U.S. and Foreign
 *	Patents, patents in process, and are protected by trade secret or
 *	copyright law. Dissemination of this information or reproduction of this
 *	material is strictly forbidden unless prior written permission is
 *      obtained from Adobe Systems Incorporated.
 *
 */

package com.adobe.fontengine.font.opentype;

import com.adobe.fontengine.font.BitmapConsumer;
import com.adobe.fontengine.font.InvalidFontException;
import com.adobe.fontengine.font.Matrix;
import com.adobe.fontengine.font.OutlineConsumer;
import com.adobe.fontengine.font.Scaler;
import com.adobe.fontengine.font.ScalerDebugger;
import com.adobe.fontengine.font.ScanConverter;
import com.adobe.fontengine.font.UnsupportedFontException;

/** A {@link Scaler} for OpenType fonts with TrueType outlines.
 */

public class TTScaler implements Scaler {

  protected final OpenTypeFont font;
  protected final TTInterpreter interpreter;
  protected final ScanConverter scanConverter;
  
  
  public TTScaler (OpenTypeFont font, ScanConverter scanConverter)
  throws InvalidFontException, UnsupportedFontException {
    
    this.font = font;
    
    if (font.maxp == null || font.maxp.getTableMajorVersion () < 1) {
      throw new InvalidFontException ("need maxp table with major version at least 1"); }
    if (font.head == null) {
      throw new InvalidFontException ("need a head table"); }
    
    this.interpreter = new TTInterpreter (
        font.maxp.getMaxStorage(), font.maxp.getMaxStackElements (),
        font.cvt == null ? 0 : font.cvt.getNbEntries (),
            font.cvt == null ? null : font.cvt.getData (),
            font.maxp.getMaxFunctionDefs (),
            font.maxp.getMaxTwilightPoints ());

    this.scanConverter = scanConverter;
    
    interpreter.setUnitsPerEm (font.head.getUnitsPerEm ());
    
    if (font.fpgm != null) {
      OTByteArray fpgm = font.fpgm.getData ();
      interpreter.runFpgm (fpgm, 0, fpgm.getSize ()); }
  }
  
  double pointSize = Double.NaN;
  double ppemX = Double.NaN;
  double ppemY = Double.NaN;
  double dX = Double.NaN;
  double dY = Double.NaN;
  Matrix em2px = null;

  boolean gridfit;
  
  public void setScale (double pointSize, double ppemX, double ppemY, double dX, double dY)
  throws InvalidFontException, UnsupportedFontException {
//    if (pointSize == this.pointSize
//        && ppemX == this.ppemX
//        && ppemY == this.ppemY
//        && dX == this.dX
//        && dY == this.dY) {
//      return; }

    this.pointSize = pointSize;
    this.ppemX = ppemX;
    this.ppemY = ppemY;
    this.dX = dX;
    this.dY = dY;   
    
    if (false) { // should be true if we pay attention to the 'gasp' table
      gridfit = (font.gasp.getBehavior (ppemX) & Gasp.BehaviorBits.GRIDFIT) != 0; }
    else {
      gridfit = true; }

    em2px = new Matrix (ppemX, 0, 0, ppemY, dX, dY);

    interpreter.setScaling (pointSize, ppemX, ppemY, dX, dY);

    if (font.prep != null) {
      OTByteArray prep = font.prep.getData ();
      interpreter.runPrep (prep, 0, prep.getSize ()); }
    else {
      interpreter.runPrep (null, 0, 0); }
  }
     
  public void getOutline (int gid, OutlineConsumer outlineConsumer)
  throws UnsupportedFontException, InvalidFontException {
    TTOutline outline = font.glyf.createTTOutline (font, gid);
    
    outline.scale (font.head.getUnitsPerEm (), em2px);

    if (ScalerDebugger.debugOn && debugger != null) {
      debugger.unhintedTTOutline (outline); }
    
    if (gridfit) {
      outline.instruct (interpreter); }

    outline.translate();
    
    if (ScalerDebugger.debugOn && debugger != null) {
      debugger.hintedTTOutline (outline); }
    
    outline.toConsumer (outlineConsumer);
  }
  
  public void getBitmap (int gid, BitmapConsumer bitmapConsumer)
  throws UnsupportedFontException, InvalidFontException {  
    TTOutline outline = font.glyf.createTTOutline (font, gid);
    
    outline.scale (font.head.getUnitsPerEm (), em2px);

    if (ScalerDebugger.debugOn && debugger != null) {
      debugger.unhintedTTOutline (outline); }
    
    if (gridfit) {
      outline.instruct (interpreter); }
    
    outline.translate();
    
    if (ScalerDebugger.debugOn && debugger != null) {
      debugger.hintedTTOutline (outline); }
    
    scanConverter.setScanType (outline.getScanType ());
    outline.toConsumer2 (scanConverter.getOutlineConsumer2 ());
    scanConverter.getBitmap (bitmapConsumer);
  }

  
  protected ScalerDebugger debugger;
  public void setDebugger (ScalerDebugger debugger) {
    this.debugger = debugger;
    interpreter.setDebugger (debugger);
    if (scanConverter != null) {
    	scanConverter.setDebugger (debugger); }
  }
}
