/*
 *
 *	File: Table.java
 *
 *
 *	ADOBE CONFIDENTIAL
 *	___________________
 *
 *	Copyright 2004-2005 Adobe Systems Incorporated
 *	All Rights Reserved.
 *
 *	NOTICE: All information contained herein is, and remains the property of
 *	Adobe Systems Incorporated and its suppliers, if any. The intellectual
 *	and technical concepts contained herein are proprietary to Adobe Systems
 *	Incorporated and its suppliers and may be covered by U.S. and Foreign
 *	Patents, patents in process, and are protected by trade secret or
 *	copyright law. Dissemination of this information or reproduction of this
 *	material is strictly forbidden unless prior written permission is obtained
 *	from Adobe Systems Incorporated.
 *
 */

package com.adobe.fontengine.font.opentype;

import java.io.IOException;

import com.adobe.fontengine.font.FontByteArray;
import com.adobe.fontengine.font.InvalidFontException;
import com.adobe.fontengine.font.UnsupportedFontException;
import com.adobe.fontengine.font.opentype.OTByteArray.OTByteArrayBuilder;

/** Base class for an OpenType table */
public class Table {
  OTByteArray data;

  protected static Table factory (int tag, FontByteArray buffer) 
  throws InvalidFontException, UnsupportedFontException, IOException  {
    switch (tag) {
      case Tag.table_cmap: return new Cmap (buffer);
      case Tag.table_hhea: return new Hhea (buffer);
      case Tag.table_hmtx: return new HmtxRaw (buffer);
      case Tag.table_maxp: return new Maxp (buffer);
      case Tag.table_name: return new Name (buffer);
      case Tag.table_os2:  return new Os2  (buffer);
      case Tag.table_GPOS: return new Gpos (buffer);
      case Tag.table_GSUB: return new Gsub (buffer);
      case Tag.table_GDEF: return new Gdef (buffer);
      case Tag.table_BASE: return new Base (buffer);
      case Tag.table_head: return new Head (buffer);
      case Tag.table_post: {
		try { return new Post (buffer); }
		catch (UnsupportedFontException unsupported) { return null; }}

      case Tag.table_glyf: return new GlyfRaw (buffer);
      case Tag.table_loca: return new LocaRaw (buffer);

      case Tag.table_CFF:  return new Cff (buffer);

      case Tag.table_vhea: return new Vhea (buffer);
      case Tag.table_vmtx: return new VmtxRaw (buffer);
      case Tag.table_cvt:  return new Cvt (buffer);
      case Tag.table_fpgm: return new Fpgm (buffer);
      case Tag.table_prep: return new Prep (buffer);
      case Tag.table_kern: return new Kern (buffer);
      case Tag.table_gasp: return new Gasp (buffer);
      case Tag.table_LTSH: return new Ltsh (buffer);
      case Tag.table_VORG: return new Vorg (buffer);
      case Tag.table_fond: return new Fond (buffer);

      default:             return new Table (buffer); }
  }
  
  /**
   * Return true iff 'tag' is a table that AFE processes.
   */
  static protected boolean knownTable(int tag) {
	   switch (tag) {
	      case Tag.table_cmap: 
	      case Tag.table_hhea: 
	      case Tag.table_hmtx: 
	      case Tag.table_maxp:
	      case Tag.table_name: 
	      case Tag.table_os2: 
	      case Tag.table_GPOS: 
	      case Tag.table_GSUB: 
	      case Tag.table_GDEF: 
	      case Tag.table_BASE:
	      case Tag.table_head:
	      case Tag.table_post:
	      case Tag.table_glyf: 
	      case Tag.table_loca:
	      case Tag.table_CFF: 
	      case Tag.table_vhea:
	      case Tag.table_vmtx:
	      case Tag.table_cvt: 
	      case Tag.table_fpgm: 
	      case Tag.table_prep: 
	      case Tag.table_kern: 
	      case Tag.table_gasp: 
	      case Tag.table_LTSH: 
	      case Tag.table_VORG:
	      case Tag.table_fond:
	    	  return true;
	   }
	   return false;
  }

  protected Table (FontByteArray buffer) 
  throws IOException, InvalidFontException, UnsupportedFontException {
    this.data = new OTByteArray(buffer);
  }

  public Table () {
    super ();
  }  
  
  protected OTByteArrayBuilder getDataAsByteArray()
  {
	  return OTByteArray.getOTByteArrayBuilderInstance(this.data);
  }

  long checksum (int from, int to, long checksum) 
  throws InvalidFontException {
    return this.data.checksum(from, to, checksum);
  }
}
