/*
 * File: CIDtoUnicodeResourceBuilder.java
 * 
 *	ADOBE CONFIDENTIAL
 *	___________________
 *
 *	Copyright 2004-2005 Adobe Systems Incorporated
 *	All Rights Reserved.
 *
 *	NOTICE: All information contained herein is, and remains the property of
 *	Adobe Systems Incorporated and its suppliers, if any. The intellectual
 *	and technical concepts contained herein are proprietary to Adobe Systems
 *	Incorporated and its suppliers and may be covered by U.S. and Foreign
 *	Patents, patents in process, and are protected by trade secret or
 *	copyright law. Dissemination of this information or reproduction of this
 *	material is strictly forbidden unless prior written permission is obtained
 *	from Adobe Systems Incorporated.
 *
 */
package com.adobe.fontengine.font.postscript;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.FileReader;
import java.io.IOException;
import java.io.LineNumberReader;
import java.io.ObjectOutputStream;
import java.io.OutputStream;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

final public class CIDtoUnicodeResourceBuilder {
  
  // This is *NOT* a general purpose CMap parser. It only understands
  // the CMaps it is supposed to work with
  
  static Pattern registryPattern
    = Pattern.compile (" */Registry \\(([A-Za-z0-9]*)\\) def");
  static Pattern orderingPattern
    = Pattern.compile (" */Ordering \\(([A-Za-z0-9]*)\\) def");
  static Pattern cmapVersionPattern
    = Pattern.compile ("/CMapVersion ([^ ]*) def");
  
  static Pattern begincidcharPattern 
    = Pattern.compile ("([0-9]*) begincidchar");
  static Pattern cidcharPattern 
    = Pattern.compile ("<([0-9a-f]*)> ([0-9]*)");

  static Pattern begincidrangePattern
    = Pattern.compile ("([0-9]*) begincidrange");
  static Pattern cidrangePattern
    = Pattern.compile ("<([0-9a-f]*)> <([0-9a-f]*)> ([0-9]*)");

  static CIDtoUnicode parseCmapFile (String fileName)
      throws IOException, IllegalArgumentException {
    
    LineNumberReader f = new LineNumberReader (new FileReader (fileName));
    try {
      int highestCid = -1;
      Map /*<Integer, Integer>*/ map = new HashMap ();
      String registry = "?";
      String ordering = "?";
      String cmapVersion = "?";
    
      String line;
      
      while ((line = f.readLine ()) != null) {
        Matcher m;
        
        m = registryPattern.matcher (line);
        if (m.matches ()) {
          registry = m.group (1); 
          continue; }
        
        m = orderingPattern.matcher (line);
        if (m.matches ()) {
          ordering = m.group (1); 
          continue; }
        
        m = cmapVersionPattern.matcher (line);
        if (m.matches ()) {
          cmapVersion = m.group (1); 
          continue; }
        
        m = begincidcharPattern.matcher (line);
        if (m.matches ()) {
          int cidCount = Integer.parseInt (m.group (1));
          for (int i = 0; i < cidCount; i++) {
            line = f.readLine ();
            Matcher m2 = cidcharPattern.matcher (line);
            if (!m2.matches ()) {
              throw new IllegalArgumentException (); }
            int usv = Integer.parseInt (m2.group (1), 16);
            int cid = Integer.parseInt (m2.group (2), 10);
            if (cid > highestCid) {
              highestCid = cid; }
            map.put (new Integer (cid), new Integer (usv)); }
          continue; }
       
        m = begincidrangePattern.matcher (line);
        if (m.matches ()) {
          int rangeCount = Integer.parseInt (m.group (1));
          for (int i = 0; i < rangeCount; i++) {
            line = f.readLine ();
            Matcher m2 = cidrangePattern.matcher (line);
            if (! m2.matches ()) {
              throw new IllegalArgumentException (); }
            int usv1 = Integer.parseInt (m2.group (1), 16);
            int usv2 = Integer.parseInt (m2.group (2), 16);
            int cid = Integer.parseInt (m2.group (3), 10); 
            for (int usv = usv1; usv <= usv2; usv++) {
              if (cid > highestCid) {
                highestCid = cid; }
              map.put (new Integer (cid), new Integer (usv)); 
              cid++; }}}
          continue; }
      
      System.out.println ("    handling " + registry + " " + ordering + " " + cmapVersion);
      
      int [] a = new int [highestCid + 1];
      Arrays.fill(a,-1);
      for (Iterator it = map.keySet().iterator(); it.hasNext (); ) {
        Integer cid = (Integer) it.next ();
        Integer usv = (Integer) map.get (cid);
        a [cid.intValue ()] = usv.intValue (); }
      
      return new CIDtoUnicode (registry, ordering, a); }

    finally {
      f.close (); }
  }

  static void dumpMap (CIDtoUnicode m, String dirname) 
       throws FileNotFoundException, IOException {
    
    File f = new File (dirname + "cid2usv_" + m.registry + "_" + m.ordering);
    f.delete ();   
    OutputStream is = new FileOutputStream (f);
    ObjectOutputStream ois = new ObjectOutputStream (is);
    ois.writeObject (m);
    ois.close ();
    
    System.out.println ("      dumped " + f.length () + " bytes to " + f.getName ());

  }
  
  static void doOne (String projectPath, String outputPath, String s)
      throws FileNotFoundException, IOException {

    dumpMap (parseCmapFile (projectPath + "outside/cmaps/" + s),
             outputPath + "com/adobe/fontengine/font/postscript/"); 
  }
    
  
  public static void main (String args []) throws Exception {
    String projectPath = args [0] + File.separator;
    String outputPath = args[1] + File.separator;
    
    System.out.println ("--- generating cid->Unicode maps");
    
    doOne (projectPath, outputPath, "UniCNS-UTF32-H");
    doOne (projectPath, outputPath, "UniGB-UTF32-H");
    doOne (projectPath, outputPath, "UniJIS-UTF32-H");
    doOne (projectPath, outputPath, "UniKS-UTF32-H");

    System.out.println ("    done");
  }
  

}
