/*
 * File: ScriptHeuristics.java
 * 
 *	ADOBE CONFIDENTIAL
 *	___________________
 *
 *	Copyright 2005 Adobe Systems Incorporated
 *	All Rights Reserved.
 *
 *	NOTICE: All information contained herein is, and remains the property of
 *	Adobe Systems Incorporated and its suppliers, if any. The intellectual
 *	and technical concepts contained herein are proprietary to Adobe Systems
 *	Incorporated and its suppliers and may be covered by U.S. and Foreign
 *	Patents, patents in process, and are protected by trade secret or
 *	copyright law. Dissemination of this information or reproduction of this
 *	material is strictly forbidden unless prior written permission is
 *      obtained from Adobe Systems Incorporated.
 *
 */
package com.adobe.fontengine.font.postscript;

import java.util.regex.Pattern;

import com.adobe.fontengine.font.CoolTypeScript;
import com.adobe.fontengine.font.InvalidFontException;
import com.adobe.fontengine.font.UnsupportedFontException;

final public class ScriptHeuristics {
  static final Pattern cePattern = Pattern.compile (".*[^A-Z]CE[^A-Za-z].*");
  static final Pattern cyrPattern = Pattern.compile (".*[^A-Z]Cyr[^A-Za-z].*");
  
  static private CoolTypeScript getCoolTypeScriptFromPostscriptName (String psName) {   
    if (psName == null) {
      return null; }
    
    if (cePattern.matcher (psName).find ()) {
      return CoolTypeScript.EAST_EUROPEAN_ROMAN; }
    
    if (cyrPattern.matcher (psName).find ()) {
      return CoolTypeScript.CYRILLIC; }
    
    // CoolType comment: recognize ALBAYANPlain and ALBAYANBold as Arabic
    if (psName.startsWith ("ALBAYAN")) {
      return CoolTypeScript.ARABIC; }
    
    return null;
  }
  
  static private CoolTypeScript getCoolTypeScriptFromEncodedGlyphComplement 
    (int numGlyphs, GlyphNamesAccessor glyphNames)
  throws InvalidFontException, UnsupportedFontException {
    
    // This heuristic looks at the encoded glyphs. If some specific
    // glyphs are present, then the script can be determined.
    // Those glyphs are not present in the standard encoding, 
    // so we only need to look at an explicit encoding.
      
    if (glyphNames == null) {
      return null; }
    
    boolean hasShin = false;
    boolean hasPatah = false;
    boolean hasSheen = false;
    boolean hasFatha = false;
    
    for (int gid = 0; gid < numGlyphs; gid++) {
      String glyphName = glyphNames.getAGlyphName (gid);
      
      if ("Tcaron".equals (glyphName)) {
        return CoolTypeScript.EAST_EUROPEAN_ROMAN; }
      
      if ("afii10049".equals (glyphName)) {
        return CoolTypeScript.CYRILLIC; }
      
      if (glyphName.startsWith ("patah")) {
        if (hasShin) {
          return CoolTypeScript.HEBREW; }
        hasPatah = true; }
      
      if (glyphName.startsWith ("shin")) {
        if (hasPatah) {
          return CoolTypeScript.HEBREW; }
        hasShin = true; }
      
      if (glyphName.startsWith ("fatha")) {
        if (hasSheen) {
          return CoolTypeScript.ARABIC; }
        hasFatha = true; }
      
      if (glyphName.startsWith ("sheen")) {
        if (hasFatha) {
          return CoolTypeScript.ARABIC; }
        hasSheen = true; }}
      
    return null;
  }
  
  static public CoolTypeScript getCoolTypeScript (String psName, int numGlyphs, GlyphNamesAccessor glyphNames)
  throws InvalidFontException, UnsupportedFontException {
      
    CoolTypeScript ctScript;
    
    if ((ctScript = getCoolTypeScriptFromPostscriptName (psName)) != null) {
      return ctScript; }
    
    if ((ctScript = getCoolTypeScriptFromEncodedGlyphComplement (numGlyphs, glyphNames)) != null) {
      return ctScript; }
    
    return CoolTypeScript.ROMAN;
  }
}
