/*
 *
 *	File: PFBInputStream.java
 *
 * ****************************************************************************
 *
 *	ADOBE CONFIDENTIAL
 *	___________________
 *
 *	Copyright 2004-2005 Adobe Systems Incorporated
 *	All Rights Reserved.
 * 
 *	NOTICE: All information contained herein is, and remains the property of
 *	Adobe Systems Incorporated and its suppliers, if any. The intellectual
 *	and technical concepts contained herein are proprietary to Adobe Systems
 *	Incorporated and its suppliers and may be covered by U.S. and Foreign
 *	Patents, patents in process, and are protected by trade secret or
 *	copyright law. Dissemination of this information or reproduction of this
 *	material is strictly forbidden unless prior written permission is obtained
 *	from Adobe Systems Incorporated.
 *
 */
package com.adobe.fontengine.font.type1;

import com.adobe.fontengine.font.FontInputStream;

import java.io.IOException;

/**
 * Reads segmented pfb data.
 *
 * <h4>Synchronization</h4>
 * 
 * This class is NOT threadsafe. Multiple instances can safely
 * coexist without threadsafety issues, but each must only be accessed 
 * from one thread (or must be guarded by the client).
 */
final class PFBInputStream extends FontInputStream {
    private int bytesLeftInSegment;
    
    PFBInputStream(FontInputStream stream){
        super(stream);
    }
    
    public int read (byte[] b, int off, int len) throws IOException
    {
        int retVal=0;
        
        while (len > 0)
        {
	        
	        if (bytesLeftInSegment > 0)
	        {
	            // return whatever is left in the current segment (up to the
	            // number of requested bytes)
	            
	            if (bytesLeftInSegment < len)
	                len = bytesLeftInSegment;
	            
	            retVal = super.read(b, off, len);
	            if (retVal == -1)
	                throw new IOException("unexpected EOF");
	            bytesLeftInSegment -= retVal;
	            return retVal;
	        } 
	        else
	        {
		        // read in the header data from the start of the next segment
		
				int escape, type;
				
				escape = super.read();
				type = super.read();;
				
				/* Check segment id */
				if (escape != 128 || (type != 1 && type != 2 && type != 3))
					throw new IOException("invalid segment");
				
				/* EOF */
				if (type == 3)
				{
					if (retVal > 0)
					    return retVal;
					else
					    return -1;
				}
				else
				{
					/* Read segment length (little endian) */
				    if ((type = super.read()) == -1)
				        throw new IOException("unexpected bytes read");
				    bytesLeftInSegment = type & 0xff;
				    
				    if ((type = super.read()) == -1)
				        throw new IOException("unexpected bytes read");
				    bytesLeftInSegment |= (type & 0xff) << 8;
				    
				    if ((type = super.read()) == -1)
				        throw new IOException("unexpected bytes read");
				    bytesLeftInSegment |= (type & 0xff) << 16;
				    
				    if ((type = super.read()) == -1)
				        throw new IOException("unexpected bytes read");
				    bytesLeftInSegment |= (type & 0xff) << 24;
				    
				}
	        }

        }
        
        return retVal;
        
    }

      public int read() throws IOException{
          byte[] b = new byte[1];
          if (read(b, 0, 1) != 1)
              return -1;
          return b[0] & 0xff;
      }

      public void unread(int b) throws IOException {
          super.unread(b);
          bytesLeftInSegment++;
      }
 
    
    public void unread (byte[] b, int off, int len) throws IOException
    {
        if (off != 0)
        {
            throw new IOException("unsupported unread offset");
        }
        
        super.unread(b, off, len);
        bytesLeftInSegment += len;
    }
}
