/*
 *	File: Type1Font.java
 *
 *	ADOBE CONFIDENTIAL
 *	___________________
 *
 *	Copyright 2004-2006 Adobe Systems Incorporated
 *	All Rights Reserved.
 * 
 *	NOTICE: All information contained herein is, and remains the property of
 *	Adobe Systems Incorporated and its suppliers, if any. The intellectual
 *	and technical concepts contained herein are proprietary to Adobe Systems
 *	Incorporated and its suppliers and may be covered by U.S. and Foreign
 *	Patents, patents in process, and are protected by trade secret or
 *	copyright law. Dissemination of this information or reproduction of this
 *	material is strictly forbidden unless prior written permission is
 *      obtained from Adobe Systems Incorporated.
 */

/*
 * Adobe Patent and/or Adobe Patent Pending invention included within this file:
 *
 * Adobe patent application tracking # P376,
 * entitled 'Method for calculating CJK emboxes in fonts',
 * invented by Nathaniel McCully
 * Issued US Patent 7,071,941 on July 4, 2006.
 *
 * Adobe patent application tracking # P376,
 * entitled 'A LINE COMPOSITION CONTROLLABLE DTP SYSTEM, A LINE
 * COMPOSITION CONTROLLING METHOD, A LINE COMPOSITION CONTROL 
 * PROGRAM AND A RECORDING MEDIUM STORING THE SAME',
 * invented by Nathaniel McCully
 * Issued Japanese Patent 3708828 on August 12, 2005.
 *
 * Adobe patent application tracking # P377,
 * entitled 'LINE PREEMPT CONTROLLABLE DTP SYSTEM, A LINE
 * PREEMPT CONTROL METHOD, A LINE PREEMPT CONTROL PROGRAM
 * AND A RECORDING MEDIUM STORING THE SAME'
 * invented by Nathaniel McCully
 * Issued Japanese Patent 3598070 on September 17, 2004.
 */

package com.adobe.fontengine.font.type1;


import java.io.IOException;
import java.io.OutputStream;
import java.net.URL;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

import com.adobe.agl.util.ULocale;
import com.adobe.fontengine.font.CScan;
import com.adobe.fontengine.font.CatalogDescription;
import com.adobe.fontengine.font.CodePage;
import com.adobe.fontengine.font.CoolTypeScript;
import com.adobe.fontengine.font.EmbeddingPermission;
import com.adobe.fontengine.font.Font;
import com.adobe.fontengine.font.FontData;
import com.adobe.fontengine.font.FontInputStream;
import com.adobe.fontengine.font.InvalidFontException;
import com.adobe.fontengine.font.InvalidGlyphException;
import com.adobe.fontengine.font.LineMetrics;
import com.adobe.fontengine.font.Matrix;
import com.adobe.fontengine.font.OrigFontType;
import com.adobe.fontengine.font.OutlineConsumer;
import com.adobe.fontengine.font.PDFFontDescription;
import com.adobe.fontengine.font.Permission;
import com.adobe.fontengine.font.ROS;
import com.adobe.fontengine.font.Rect;
import com.adobe.fontengine.font.SWFFont4Description;
import com.adobe.fontengine.font.SWFFontDescription;
import com.adobe.fontengine.font.Scaler;
import com.adobe.fontengine.font.ScanConverter;
import com.adobe.fontengine.font.Subset;
import com.adobe.fontengine.font.SubsetDefaultImpl;
import com.adobe.fontengine.font.SubsetSimpleTrueType;
import com.adobe.fontengine.font.SubsetSimpleType1;
import com.adobe.fontengine.font.UnderlineMetrics;
import com.adobe.fontengine.font.UnsupportedFontException;
import com.adobe.fontengine.font.XDCFontDescription;
import com.adobe.fontengine.font.cff.CIDKeyedFont;
import com.adobe.fontengine.font.cff.CharStrings;
import com.adobe.fontengine.font.cff.Dict;
import com.adobe.fontengine.font.cff.NameKeyedFont;
import com.adobe.fontengine.font.cff.Type2CStringGenerator;
import com.adobe.fontengine.font.cff.NameKeyedFont.GlyphNameFetcher;
import com.adobe.fontengine.font.postscript.GlyphNamesAccessor;
import com.adobe.fontengine.font.postscript.NameHeuristics;
import com.adobe.fontengine.font.postscript.ScriptHeuristics;
import com.adobe.fontengine.font.postscript.StandardEncoding;
import com.adobe.fontengine.font.postscript.SubArrays;
import com.adobe.fontengine.font.postscript.Token;
import com.adobe.fontengine.font.postscript.TokenType;
import com.adobe.fontengine.font.postscript.Tokenizer;
import com.adobe.fontengine.font.postscript.UnicodeCmap;
import com.adobe.fontengine.fontmanagement.CacheSupportInfo;
import com.adobe.fontengine.fontmanagement.Platform;
import com.adobe.fontengine.fontmanagement.fxg.FXGFontDescription;
import com.adobe.fontengine.fontmanagement.platform.PlatformFontDescription;
import com.adobe.fontengine.fontmanagement.postscript.PostscriptFontDescription;
import com.adobe.fontengine.inlineformatting.css20.CSS20Attribute.CSSStretchValue;

/**
 * Gives access to a Type1 font.
 *
 * <h4>Synchronization</h4>
 * 
 * This class can have a two-step construction. First,
 * the constructor is called, then setMetricFile may be called. This is
 * required because the fontname must be used to determine what metric file
 * aligns with this font. setMetricFile shall only be called one time. It is
 * called before the modules outside of AFE can see the Type1Font object. 
 * After the two-phase construction is complete, the class is immutable and threadsafe.
 */
final public class Type1Font extends FontData {

	private Map keyVal;
	private byte[][] subrs;
	private Type1Glyph[] charStrings; 
	private MetricFile metricFile;
	private final URL fontLocation;
	private boolean seenPrivate;

	/** if we are looking at the base font for a synthetic,
        store the info in this Map.
	 */
	private Map syntheticBaseMap;

	/** points to either the synthetic base or keyval map depending on if we
       are looking at the base font for the current font.
	 */
	private Map currentMap;


	//These need to be fixed if mm is to be handled.
	/** the longest integer array in type1 is for XUIDs, which has up to 16 entries. */
	private final static int maxIntArrayLen = 16; 

	/** the longest allowed double array is for blend design pos, which have up to 64 entries */
	private final static int maxDoubleArrayLen = 64;

	private final static Map literals;

	// defaults
	private final static int defaultLenIV = 4;
	final static Matrix defaultFontMatrix = new Matrix(0.001, 0, 0, 0.001, 0, 0);
	final static double[] defaultKernValue = {0, 0};
	private final T1XDCFontDescription xdcDescription;


	// operators that we need to recognize
	private static class Operators {

		final static byte[] kTRUE = {'t', 'r', 'u', 'e'};
		final static byte[] kFALSE = {'f', 'a', 'l', 's', 'e'};

		final static byte[] kCURRENTFILE = {'c', 'u', 'r', 'r', 'e', 'n', 't', 'f', 'i', 'l', 'e'};
		final static byte[] kCLOSEFILE = {'c', 'l', 'o', 's', 'e', 'f', 'i', 'l', 'e'};
		final static byte[] kEEXEC = {'e', 'e', 'x', 'e', 'c'};
		final static byte[] kSTANDARDENCODING = {'S', 't', 'a', 'n', 'd', 'a','r','d','E','n','c','o','d','i','n','g'};
		final static byte[] kNOTDEF = {'.','n','o','t','d','e','f'};
		final static byte[] kDUP = {'d','u','p'};
		final static byte[] kPUT = {'p', 'u', 't'};
		final static byte[] kDEF = {'d','e','f'};
		final static byte[] kREADONLY = {'r','e','a','d','o','n','l','y'};
		final static byte[] kBEGIN = {'b','e','g','i','n'};
		final static byte[] kEND = {'e', 'n', 'd'};
		final static byte[] kARRAY = {'a','r','r','a','y'};
		final static byte[] kNOACCESS = {'n','o','a','c','c','e','s','s'};
		final static byte[] kFONTDIRECTORY = {'F','o','n','t','D','i','r','e','c','t','o','r','y'};
	}

	// literals that aren't keys that we need to recognize
	private static class Literals {
		final static byte[] kCID_LITERAL = {'/', 'C', 'I', 'D'};
		final static byte[] kTRUETYPE_LITERAL = {'/', 'T', 'r', 'u', 'e', 'T', 'y', 'p', 'e'};
		final static byte[] kTYPE1_LITERAL = {'/', 'T', 'y', 'p', 'e', '1'};
		final static byte[] kOCF_LITERAL = {'/', 'O', 'C', 'F'};
		final static byte[] kNOTDEF = {'.', 'n','o','t','d','e','f'};
	}

	// this is the set of keys in keyVal and the set of interesting keys
	// in the font.
	private static class Type1Keys {
		private Type1Keys() {
			// namespace class, enforced by a private constructor
		}

		static String BaseFontBlend = "BaseFontBlend";
		static String BaseFontName = "BaseFontName";
		static String Blend = "Blend";
		static String BlendDesignMap = "BlendDesignMap";
		static String BlendDesignPositions = "BlendDesignPositions";
		static String BlueFuzz = "BlueFuzz";
		static String BlueScale = "BlueScale";
		static String BlueShift = "BlueShift";
		static String BlueValues = "BlueValues";
		static String CIDInit = "CIDInit";
		static String Chameleon = "Chameleon";
		static String CharStrings = "CharStrings";
		static String Copyright = "Copyright";
		static String CSSWeight = "CSSWeight";
		static String CSSWidth = "CSSWidth";
		static String Encoding = "Encoding";
		static String Erode = "Erode";
		static String ExpansionFactor = "ExpansionFactor";
		static String FSType = "FSType";
		static String FamilyBlues = "FamilyBlues";
		static String FamilyName = "FamilyName";
		static String FamilyOtherBlues = "FamilyOtherBlues";
		static String FontBBox = "FontBBox";
		static String FontMatrix = "FontMatrix";
		static String FontName = "FontName";
		static String FontType = "FontType";
		static String ForceBold = "ForceBold";
		static String ForceBoldThreshold = "ForceBoldThreshold";
		static String FullName = "FullName";
		static String GlyphDirectory = "GlyphDirectory";
		static String InvertedFontMatrix = "InvertedFontMatrix";
		static String IsSmallCaps = "IsSmallCaps";
		static String ItalicAngle = "ItalicAngle";
		static String LanguageGroup = "LanguageGroup";
		static String Notice = "Notice";
		static String OrigFontType = "OrigFontType";
		static String OtherBlues = "OtherBlues";
		static String PaintType = "PaintType";
		static String Private = "Private";
		static String PostScript = "PostScript";
		static String RndStemUp = "RndStemUp";
		static String StandardEncoding = "StandardEncoding";
		static String StdHW = "StdHW";
		static String StdVW = "StdVW";
		static String StemSnapH = "StemSnapH";
		static String StemSnapV = "StemSnapV";
		static String StrokeWidth = "StrokeWidth";
		static String Subrs = "Subrs";
		static String UnderlinePosition = "UnderlinePosition";
		static String UnderlineThickness = "UnderlineThickness";
		static String UniqueID = "UniqueID";
		static String WasEmbedded = "WasEmbedded";
		static String Weight = "Weight";
		static String WeightVector = "WeightVector";
		static String XUID = "XUID";
		static String hires = "hires";
		static String initialRandomSeed = "initialRandomSeed";
		static String isFixedPitch = "isFixedPitch";
		static String lenIV = "lenIV";
		static String version = "version";

	}

	// types of arguments that can be associated with literals
	private static class ArgumentType {
		private String description;

		private ArgumentType(String s){description = s;}

		static ArgumentType kINTEGER = new ArgumentType("Integer");
		static ArgumentType kNUMBER = new ArgumentType("Number");
		static ArgumentType kSTRING = new ArgumentType("String"); //array of bytes we want to be in unicode.
		static ArgumentType kBYTES = new ArgumentType("Bytes"); // argument is an array of bytes we don't want to translate to a string.
		static ArgumentType kBOOL = new ArgumentType("Bool");
		static ArgumentType kNUMARRAY = new ArgumentType("NumArray");
		static ArgumentType kARRAYOF1 = new ArgumentType("Array of 1"); // argument must be an array containing 1 number
		static ArgumentType kINTARRAY = new ArgumentType("IntArray");
		static ArgumentType kNAME = new ArgumentType("Name");
		static ArgumentType kUNSUPPORTEDFONT = new ArgumentType("Unsupported font"); // seeing this literal means we don't support the font.

		// the following literal/arguments must be processed specially...
		static ArgumentType kFONTTYPE = new ArgumentType("FontType"); 
		static ArgumentType kORIGFONTTYPE = new ArgumentType("OrigFontType");
		static ArgumentType kENCODING = new ArgumentType("Encoding");
		static ArgumentType kERODE = new ArgumentType("ErodeProcedure");
		static ArgumentType kCHARSTRINGS = new ArgumentType("CharStrings");
		static ArgumentType kSUBRS = new ArgumentType("Subroutines");
		static ArgumentType kFONTMATRIX = new ArgumentType("FontMatrix");
		static ArgumentType kRNDSTEMUP = new ArgumentType("RndStemUp");
		static ArgumentType kITALICANGLE = new ArgumentType("ItalicAngle");
		static ArgumentType kBBOX = new ArgumentType("BoundingBox");
		static ArgumentType kPRIVATE = new ArgumentType("Private Dictionary");

		public String toString() {return description;}
	}

	private static class CharstringCompare implements Comparator {
		private CharstringCompare() {
			// singleton class, enforced by a private constructor
		}

		static final CharstringCompare comparator = new CharstringCompare();

		public int compare(Object o1, Object o2) 
		{
			Type1Glyph first;
			Type1Glyph second;

			if (o1 == null) {
				if (o2 == null)
					return 0;

				// put null objects at the end
				return 1;
			}

			if (o2 == null)
				return -1;

			first = (Type1Glyph)o1;
			second = (Type1Glyph)o2;

			// always put .notdef first.
			if (first.glyphName.equals(".notdef"))
			{
				if (second.glyphName.equals(".notdef"))
					return 0;

				return -1;
			}

			if (second.glyphName.equals(".notdef"))
				return 1;

			return first.glyphName.compareTo(second.glyphName);
		}        
	}

	static {
		literals = new HashMap(72);

		literals.put(Type1Keys.BaseFontBlend, ArgumentType.kINTARRAY);
		literals.put(Type1Keys.BaseFontName, ArgumentType.kNAME);
		literals.put(Type1Keys.Blend, ArgumentType.kUNSUPPORTEDFONT);// mm is unsupported
		literals.put(Type1Keys.BlendDesignMap, ArgumentType.kNUMARRAY);
		literals.put(Type1Keys.BlendDesignPositions, ArgumentType.kNUMARRAY);
		literals.put(Type1Keys.BlueFuzz, ArgumentType.kINTEGER);
		literals.put(Type1Keys.BlueScale, ArgumentType.kNUMBER);
		literals.put(Type1Keys.BlueShift, ArgumentType.kNUMBER);
		literals.put(Type1Keys.BlueValues, ArgumentType.kNUMARRAY);
		literals.put(Type1Keys.CIDInit, ArgumentType.kUNSUPPORTEDFONT);// we don't support legacy cid
		literals.put(Type1Keys.Chameleon, ArgumentType.kUNSUPPORTEDFONT);
		literals.put(Type1Keys.CharStrings, ArgumentType.kCHARSTRINGS);
		literals.put(Type1Keys.Copyright, ArgumentType.kBYTES);
		literals.put(Type1Keys.Encoding, ArgumentType.kENCODING);
		literals.put(Type1Keys.Erode, ArgumentType.kERODE);
		literals.put(Type1Keys.ExpansionFactor, ArgumentType.kNUMBER);
		literals.put(Type1Keys.FSType, ArgumentType.kINTEGER);
		literals.put(Type1Keys.FamilyBlues, ArgumentType.kNUMARRAY);
		literals.put(Type1Keys.FamilyName, ArgumentType.kSTRING);
		literals.put(Type1Keys.FamilyOtherBlues, ArgumentType.kNUMARRAY);
		literals.put(Type1Keys.FontBBox, ArgumentType.kBBOX);
		literals.put(Type1Keys.FontMatrix, ArgumentType.kFONTMATRIX); 
		literals.put(Type1Keys.FontName, ArgumentType.kNAME);
		literals.put(Type1Keys.FontType, ArgumentType.kFONTTYPE);
		literals.put(Type1Keys.ForceBold, ArgumentType.kBOOL);
		literals.put(Type1Keys.ForceBoldThreshold, ArgumentType.kNUMBER);
		literals.put(Type1Keys.FullName, ArgumentType.kSTRING);
		literals.put(Type1Keys.GlyphDirectory, ArgumentType.kUNSUPPORTEDFONT); // this is a download format. we don't support it.
		literals.put(Type1Keys.ItalicAngle, ArgumentType.kITALICANGLE);
		literals.put(Type1Keys.LanguageGroup, ArgumentType.kINTEGER);
		literals.put(Type1Keys.Notice, ArgumentType.kBYTES);
		literals.put(Type1Keys.OrigFontType, ArgumentType.kORIGFONTTYPE);
		literals.put(Type1Keys.OtherBlues, ArgumentType.kNUMARRAY);
		literals.put(Type1Keys.PaintType, ArgumentType.kINTEGER);
		literals.put(Type1Keys.Private, ArgumentType.kPRIVATE);
		literals.put(Type1Keys.PostScript, ArgumentType.kBYTES);
		literals.put(Type1Keys.RndStemUp, ArgumentType.kRNDSTEMUP);
		literals.put(Type1Keys.StdHW, ArgumentType.kARRAYOF1);
		literals.put(Type1Keys.StdVW, ArgumentType.kARRAYOF1);
		literals.put(Type1Keys.StemSnapH, ArgumentType.kNUMARRAY);
		literals.put(Type1Keys.StemSnapV, ArgumentType.kNUMARRAY);
		literals.put(Type1Keys.StrokeWidth,ArgumentType.kNUMBER);
		literals.put(Type1Keys.Subrs,ArgumentType.kSUBRS);
		literals.put(Type1Keys.UnderlinePosition, ArgumentType.kNUMBER);
		literals.put(Type1Keys.UnderlineThickness, ArgumentType.kNUMBER);
		literals.put(Type1Keys.UniqueID, ArgumentType.kINTEGER);
		literals.put(Type1Keys.WasEmbedded, ArgumentType.kBOOL);
		literals.put(Type1Keys.Weight, ArgumentType.kSTRING);
		literals.put(Type1Keys.WeightVector, ArgumentType.kNUMARRAY);
		literals.put(Type1Keys.XUID, ArgumentType.kINTARRAY);
		literals.put(Type1Keys.hires, ArgumentType.kUNSUPPORTEDFONT); // we don't support hires/lores fonts
		literals.put(Type1Keys.initialRandomSeed, ArgumentType.kINTEGER);
		literals.put(Type1Keys.isFixedPitch, ArgumentType.kBOOL);
		literals.put(Type1Keys.lenIV, ArgumentType.kINTEGER);
		literals.put(Type1Keys.version, ArgumentType.kBYTES);

	}

	Type1Font(FontInputStream inStream, URL location)
	throws IOException, InvalidFontException, UnsupportedFontException
	{  
		super(null);

		keyVal = new HashMap();
		currentMap = keyVal;

		fontLocation = location;
		parse(inStream);

		xdcDescription = new T1XDCFontDescription();

	}

	public MetricFile getMetricFile()
	{
		return metricFile;   
	}

	public URL getFontLocation()
	{
		return fontLocation;
	}

	/**
	 * given the choice between alternative and the current metric file,
	 * pick one. The current impl prefers PFMs over AFMs.
	 * 
	 * @param alternative
	 */
	private boolean pickFromEqualMetricFiles(MetricFile alternative)
	{      
		if (metricFile instanceof AFM 
				&& alternative instanceof PFM)
		{
			metricFile = alternative;
			return true;
		} 
		else 
			return false;
	}

	private void removeLastFileFromPath(StringBuffer path)
	{
		for (int i = path.length()-1; i >=0; --i)
		{
			if (path.charAt(i) == '/')
			{
				path.delete(i, path.length());
				return;
			}
		}
	}

	/**
	 * This must be called only just after Type1Font construction
	 * (prior to the Type1Font being visible to the client). It is
	 * NOT to be called by clients of this object.
	 * 
	 * @param f the metric file that may be associated with this font. 
	 */
	public boolean setMetricFile(MetricFile f)
	{

		// use the first one found unless it is an AFM
		// and a PFM is later found.

		if (metricFile == null)
		{
			metricFile = f;
			return true;
		}
		else 
		{
			StringBuffer currPath = new StringBuffer(metricFile.getLocation().getPath());
			StringBuffer fPath = new StringBuffer(f.getLocation().getPath());
			StringBuffer fontPath;

			if (fontLocation == null)
			{
				return pickFromEqualMetricFiles(f);
			}            

			fontPath = new StringBuffer(fontLocation.getPath());

			removeLastFileFromPath(fontPath);

			// on the first iteration, we remove the file name. on the second, we
			// are checking for metric files in a subdirectory of the font's directory
			for (int i = 0; i < 2; i++)
			{
				removeLastFileFromPath(currPath);
				removeLastFileFromPath(fPath);

				if (currPath.length() == fontPath.length() 
						&& SubArrays.stringBufferCompare(currPath, 0, fontPath, 0, fontPath.length()))
				{
					// if both metric files are in the same path as the font, pick from among equals
					if (fPath.length() == fontPath.length() 
							&& SubArrays.stringBufferCompare(fPath, 0, fontPath, 0, fontPath.length()))
						return pickFromEqualMetricFiles(f);

					// otherwise, if the current metric file is in the font's path, keep it.

					return false;
				}
				else if (fPath.length() == fontPath.length() 
						&& SubArrays.stringBufferCompare(fPath, 0, fontPath, 0, fontPath.length()))
				{
					// if fontData is in the font's path, use it.
					metricFile = f;
					return true;
				}
			}

			return pickFromEqualMetricFiles(f);
		}
	}

	private Integer parseInteger(Tokenizer tokenizer)
	throws IOException, InvalidFontException
	{
		Token token = tokenizer.getNextPSToken();
		if (token.tokenType == TokenType.kINTEGER)
		{
			return new Integer(token.convertInteger(0));
		}

		return null;
	}

	private Double parseNumber(Tokenizer tokenizer)
	throws IOException, InvalidFontException
	{
		Token token = tokenizer.getNextPSToken();
		if (token.tokenType == TokenType.kINTEGER)
		{ 
			return new Double(token.convertInteger(0));
		} else if (token.tokenType == TokenType.kREAL)
		{
			return new Double(Double.parseDouble(new String(token.buff, 0, token.tokenLength)));
		}    
		return null;
	}

	private byte[] parseBytes(Tokenizer tokenizer)
	throws IOException, InvalidFontException
	{
		Token token = tokenizer.getNextPSToken();
		if (token.tokenType == TokenType.kSTRING)
		{
			byte[] retVal = new byte[token.tokenLength-2];
			System.arraycopy(token.buff, 1, retVal, 0, token.tokenLength-2);
			return retVal;
		}

		return new byte[0];
	}

	private String parseString(Tokenizer tokenizer)
	throws IOException, InvalidFontException
	{
		Token token = tokenizer.getNextPSToken();
		if (token.tokenType == TokenType.kSTRING)
		{
			char[] chars = new char [token.tokenLength-2];
			for (int i = 1; i < token.tokenLength-1; i++) {
				chars [i-1] = (char) token.buff[i]; }

			return new String (chars);
		}

		return null;
	}

	private String parseName(Tokenizer tokenizer)
	throws IOException, InvalidFontException
	{
		Token token = tokenizer.getNextPSToken();
		if (token.tokenType == TokenType.kSTRING)
		{
			return token.stringTokenToString(1, token.tokenLength-1);

		} else if (token.tokenType == TokenType.kLITERAL)
		{
			return token.stringTokenToString(1, token.tokenLength);
		}

		return null;
	}


	private Boolean parseBool(Tokenizer tokenizer)
	throws IOException, InvalidFontException
	{
		Token token = tokenizer.getNextPSToken();
		if (token.tokenType == TokenType.kOPERATOR)
		{
			if (SubArrays.arrayCompare(token.buff, 0, Operators.kTRUE, 0, Operators.kTRUE.length))
				return Boolean.valueOf(true);
			else if (SubArrays.arrayCompare(token.buff, 0, Operators.kFALSE, 0, Operators.kFALSE.length))
				return Boolean.valueOf(false);
		}

		return null;
	}

	private OrigFontType parseOrigFontType(Tokenizer tokenizer)
	throws IOException, InvalidFontException
	{
		Token token = tokenizer.getNextPSToken();
		if (token.tokenType == TokenType.kLITERAL)
		{
			if (SubArrays.arrayCompare(token.buff, 0, Literals.kTRUETYPE_LITERAL, 0, token.tokenLength))
				return OrigFontType.kTRUETYPE;
			else if (SubArrays.arrayCompare(token.buff, 0, Literals.kTYPE1_LITERAL, 0, token.tokenLength))
				return OrigFontType.kTYPE1;
			else if (SubArrays.arrayCompare(token.buff, 0, Literals.kCID_LITERAL, 0, token.tokenLength))
				return OrigFontType.kCID;
			else if (SubArrays.arrayCompare(token.buff, 0, Literals.kOCF_LITERAL, 0, token.tokenLength))
				return OrigFontType.kOCF;
		}
		return null;
	}

	private int[] parseIntArray(Tokenizer tokenizer)
	throws IOException, InvalidFontException
	{
		Token token = tokenizer.getNextPSToken();
		if (token.tokenType == TokenType.kARRAY || token.tokenType == TokenType.kPROCEDURE)
		{
			int i;
			int[] array = new int[maxIntArrayLen+1];
			int validEntries = 0;
			boolean inNumber = false;

			for (i = 0; i < token.tokenLength; i++)
			{
				switch(token.buff[i])
				{
				default:
					inNumber = false;
				break;

				case '0': 
				case '1':
				case '2': 
				case '3': 
				case '4':
				case '5':
				case '6':
				case '7':
				case '8':
				case '9':
				case '+':
				case '-':
					if (!inNumber)
					{
						array[validEntries++] = token.convertInteger(i);
						inNumber = true;
					}
					break;

				case '#':
					break;
				}
			}

			if (validEntries > 0)
			{
				int[] retArray;
				retArray = new int[validEntries];
				java.lang.System.arraycopy(array, 0, retArray, 0, validEntries);
				return retArray;
			}
		}

		return new int[0];
	}

	private double[] parseNumArray(Tokenizer tokenizer)
	throws IOException, InvalidFontException
	{
		Token token = tokenizer.getNextPSToken();
		if (token.tokenType == TokenType.kARRAY || token.tokenType == TokenType.kPROCEDURE)
		{
			int i;
			double[] array = new double[maxDoubleArrayLen+1];
			int validEntries = 0;

			for (i = 0; i < token.tokenLength; i++)
			{
				switch(token.buff[i])
				{                    
				case '0': 
				case '1':
				case '2': 
				case '3': 
				case '4':
				case '5':
				case '6':
				case '7':
				case '8':
				case '9':
				case '+':
				case '-':
				case '.':
					int j;
					boolean endOfNumber = false;
					for (j = i+1; j < token.tokenLength && !endOfNumber; j++)
					{
						switch (token.buff[j])
						{
						case '0': 
						case '1':
						case '2': 
						case '3': 
						case '4':
						case '5':
						case '6':
						case '7':
						case '8':
						case '9':
						case '+':
						case '-':
						case '.':
						case 'E':
						case 'e':
							break;

						default:
							endOfNumber = true;
						break;
						}
					}

					String doubleString = new String(token.buff, i, j-i-1, "US-ASCII");

					try {
						array[validEntries++] = Double.parseDouble(doubleString);
					} catch (NumberFormatException e)
					{
						throw new InvalidFontException(e);
					}
					i = j-1;
					break;
				}
			}
			if (validEntries > 0)
			{
				double[] retArray = new double[validEntries];
				java.lang.System.arraycopy(array, 0, retArray, 0, validEntries);
				return retArray;
			}
		}

		return new double[0];
	}

	private void parseEncoding(Tokenizer tokenizer)
	throws IOException, InvalidFontException
	{

		int i;
		Token token = tokenizer.getNextPSToken();
		byte[][] customEncoding;

		if (SubArrays.arrayCompare(                
				token.buff, 
				0, 
				Operators.kSTANDARDENCODING, 
				0, 
				Operators.kSTANDARDENCODING.length))

		{  
			keyVal.put(Type1Keys.StandardEncoding, Boolean.valueOf(true));
			return;
		}


		customEncoding = new byte[256][];

		/* Initialize encoding array */
		for (i = 0; i < 256; i++)
			customEncoding[i] = Operators.kNOTDEF;

		/* Search for first entry */
		for (;;)
		{
			if (SubArrays.arrayCompare(token.buff, 0, Operators.kDUP, 0, Operators.kDUP.length))
			{
				int code;
				byte[] glyphName;

				/* Parse encoding value */
				token = tokenizer.getNextPSToken();
				if (token.tokenType != TokenType.kINTEGER)
				{
					continue;
				}

				code = token.convertInteger(0);
				if (code < 0 || code > 255)
				{
					continue;
				}

				/* Parse glyph name */
				token = tokenizer.getNextPSToken();
				if (token.tokenType != TokenType.kLITERAL)
				{
					continue;
				}

				glyphName = new byte[token.tokenLength-1];
				java.lang.System.arraycopy(token.buff, 1, glyphName, 0, glyphName.length);

				/* Check for put operator */
				token = tokenizer.getNextPSToken();
				if (!SubArrays.arrayCompare(token.buff, 0, Operators.kPUT, 0, Operators.kPUT.length))
				{
					continue;
				}

				/* Save encoding */
				customEncoding[code] = glyphName;

			}
			else if (SubArrays.arrayCompare(token.buff, 0, Operators.kDEF, 0, Operators.kDEF.length) ||
					SubArrays.arrayCompare(token.buff, 0, Operators.kREADONLY, 0, Operators.kREADONLY.length))
			{

				keyVal.put(Type1Keys.Encoding, customEncoding);
				return;	/* Success */
			}

			token = tokenizer.getNextPSToken();
		}
	}

	private int readCharStringLen(Tokenizer tokenizer)
	throws IOException, InvalidFontException
	{
		/* Read initial token */
		Token token = tokenizer.getNextPSToken();
		if (token.tokenType != TokenType.kINTEGER)
		{
			return 0;	/* Unexpected token type */
		}

		/* Convert charstring length */
		int length = token.convertInteger(0);
		if (length < 1 || length > 65535)
			return 0;

		/* Parse RD operator */
		token = tokenizer.getNextPSToken();
		if (token.tokenType != TokenType.kOPERATOR)
			return 0;

		return length;

	}

	private void readLenIVBytes(byte[] lenIVBytes, Tokenizer tokenizer)
	throws IOException, InvalidFontException
	{
		// skip separator
		tokenizer.read();

		for (int i = 0; i < lenIVBytes.length; i++)
			lenIVBytes[i] = (byte)tokenizer.read();
	}

	private byte[] readCharString(Tokenizer tokenizer, int length)
	throws IOException, InvalidFontException
	{
		byte[] cstr = new byte[length];

		/* Read charstring bytes */
		for (int i = 0; i < length; i++)
			cstr[i] = (byte)tokenizer.read();

		return cstr;
	}

	private void decryptCharString(byte[] lenIVBytes, byte[] cipher, byte[] plain)
	{

		int r = 4330;	/* Initial state */
		int i;

		/* Prime state from random initial bytes */
		for (i = 0; i < lenIVBytes.length; i++)
			r = (((lenIVBytes[i] & 0xff) + r)*52845 + 22719) & 0xffff;

		/* Decrypt and copy bytes */
		for (i = 0; i < cipher.length; i++)
		{
			int c1 = cipher[i] & 0xff;
			plain[i] = (byte)(c1 ^ (r>>8));
			r = ((c1 + r)*52845 + 22719) & 0xffff;
		}
	}

	private void parseChars(Tokenizer tokenizer)
	throws IOException, InvalidFontException
	{
		boolean foundNotdef = false;
		int lenIV;
		byte[] lenIVBytes = null;
		int numChars;
		Token token;
		int i;
		final byte[] replacementName = {' '};

		Object val = keyVal.get(Type1Keys.lenIV);

		if (val == null)
			lenIV = defaultLenIV;
		else
			lenIV = ((Integer)val).intValue();

		if (lenIV >= 0)
			lenIVBytes = new byte[lenIV];

		/* Parse the upper limit of the charstring count */
		token = tokenizer.getNextPSToken();
		if (token.tokenType != TokenType.kINTEGER)
			return;

		numChars = token.convertInteger(0);

		if (numChars < 0)
			return;

		List foundCharStrings = new ArrayList(numChars);        

		tokenizer.findToken(Operators.kBEGIN);

		for (i = 0; i < numChars; ++i)
		{
			byte[] cstr;
			int length;
			byte[] glyphName;

			/* Parse glyph name */
			token = tokenizer.getNextPSToken();
			if (token.matches(Operators.kEND))
			{
				/* Reached end of CharStrings dict */
				break;
			}

			else if (token.tokenType != TokenType.kLITERAL)
				continue; // skip invalid font data...this should be the glyph name

			// hold onto the glyphname
			if (token.tokenLength > 1)
			{
				glyphName = new byte[token.tokenLength - 1];
				System.arraycopy(token.buff, 1, glyphName, 0, token.tokenLength-1);
			} else
				glyphName = replacementName;

			// read in the charstring and decrypt it
			length = readCharStringLen(tokenizer) - ((lenIVBytes != null) ? lenIV:0);

			if (length < 1 || length > 65535)
				continue;

			if (lenIVBytes != null)
				readLenIVBytes(lenIVBytes, tokenizer);	
			cstr = readCharString(tokenizer, length);	

			if (lenIVBytes != null)
				decryptCharString(lenIVBytes, cstr, cstr);


			if (glyphName.length == Literals.kNOTDEF.length
					&& SubArrays.arrayCompare(glyphName, 0, Literals.kNOTDEF, 0, Literals.kNOTDEF.length))
				foundNotdef = true; 

			foundCharStrings.add(new Type1Glyph(glyphName, cstr));

			/* skip NP operator */
			token = tokenizer.getNextPSToken();
		}

		if (!foundNotdef)
		{
			// if notdef was missing, grow the charstrings and add it.

			/* 0 250 hsbw endchar */
			byte[] notdefCharstring = {(byte)0x8b, (byte)0xf7, (byte)0x8e, 0xd, 0xe};
			foundCharStrings.add(new Type1Glyph(Literals.kNOTDEF, notdefCharstring));
		}

		// all of the charstrings have been parsed into foundCharStrings...turn it into an array.
		charStrings = new Type1Glyph[foundCharStrings.size()];
		foundCharStrings.toArray(charStrings);


		java.util.Arrays.sort(charStrings, CharstringCompare.comparator);

	}

	private void parseSubrs(Tokenizer tokenizer)
	throws IOException, InvalidFontException
	{
		int i;
		Token token;
		int cnt;
		int subrNumber;
		int lenIV;
		byte[] lenIVBytes = null;

		Object val = keyVal.get(Type1Keys.lenIV);
		if (val == null)
			lenIV = defaultLenIV;
		else
			lenIV = ((Integer)val).intValue();

		/* Parse subrs count */
		token = tokenizer.getNextPSToken();
		if (token.tokenType != TokenType.kINTEGER)
			return;

		cnt = token.convertInteger(0);

		if (cnt < 0 || cnt > 65535)
			return;

		subrs = new byte[cnt][];

		if (lenIV >= 0)
			lenIVBytes = new byte[lenIV];

		tokenizer.findToken(Operators.kARRAY);
		for (i = 0;; i++)
		{
			int length;
			byte[] cstr;

			/* Read dup token */
			token = tokenizer.getNextPSToken();
			if (!token.matches(Operators.kDUP))
			{
				return;
			}

			/* Parse subr number */
			token = tokenizer.getNextPSToken();
			if (token.tokenType != TokenType.kINTEGER)
				return;

			subrNumber = token.convertInteger(0);

			if (subrNumber < 0 || subrNumber >= cnt)
				return;


			length = readCharStringLen(tokenizer) - ((lenIVBytes != null) ? lenIV:0);

			if (lenIVBytes != null)
				readLenIVBytes(lenIVBytes, tokenizer);	
			cstr = readCharString(tokenizer, length);	
			if (lenIVBytes != null)
				decryptCharString(lenIVBytes, cstr, cstr);

			subrs[subrNumber] = cstr;

			/* Parse NP operator */
			token = tokenizer.getNextPSToken();
			if (token.tokenType != TokenType.kOPERATOR)
				return;

			/* Check for "noaccess put" sequence */
			if (token.matches(Operators.kNOACCESS))
			{
				token = tokenizer.getNextPSToken();
				if (token.tokenType != TokenType.kOPERATOR)
					break;
			}
		}
	}

	private void parseFontMatrix(Tokenizer tokenizer)
	throws IOException, InvalidFontException
	{
		double[] arg = parseNumArray(tokenizer);
		if (arg.length == 6 && 
				(arg[0] != 0.001 ||
						arg[1] != 0.0 ||
						arg[2] != 0.0 ||
						arg[3] != 0.001 ||
						arg[4] != 0.0 ||
						arg[5] != 0.0))
		{

			if (arg[4] != 0 || arg[5] != 0)
				throw new InvalidFontException("non-zero translation components in the font matrix");

			double det = arg[0] * arg[3] - arg[2] * arg[1];
			double absDet = java.lang.Math.abs(det);
			if (absDet < 1)
			{
				double mx = Double.MAX_VALUE * absDet;

				if ((java.lang.Math.abs(arg[0]) >= mx) 
						|| (java.lang.Math.abs(arg[1]) >= mx)
						|| (java.lang.Math.abs(arg[2]) >= mx) 
						|| (java.lang.Math.abs(arg[3]) >= mx))
					throw new InvalidFontException("non-invertible font matrix");
			}

			currentMap.put(Type1Keys.FontMatrix, new Matrix(arg));

			Matrix inverse = new Matrix(arg[3] / det, -arg[1] / det,
					-arg[2] / det, arg[0] / det, 0, 0);

			currentMap.put(Type1Keys.InvertedFontMatrix, inverse);

		}

	}

	private void parseErodeProc(Tokenizer tokenizer)
	throws IOException, InvalidFontException
	{
		// if we have seen a StdVW, skip this.
		if (keyVal.get(Type1Keys.StdVW) != null)
			return;

		Token token = tokenizer.getNextPSToken();
		int currToken = 0;

		if (token.tokenType == TokenType.kPROCEDURE)
		{
			for (int currPos = 1; currPos < token.tokenLength; ++currPos)
			{

				while (currPos < token.tokenLength 
						&& Tokenizer.isWhite(token.buff[currPos]))
				{currPos++;}

				if (currPos == token.tokenLength || token.buff[currPos] == '}')
					break;

				if (++currToken == 16)
				{
					Double stdvw = new Double(token.convertInteger(currPos));
					keyVal.put(Type1Keys.StdVW, stdvw);
					break;	                
				}



				while (currPos < token.tokenLength
						&& !Tokenizer.isWhite(token.buff[currPos]))
				{currPos++;}
			}
		}
	}

	private boolean doLiteral(Tokenizer tokenizer,
			Token currToken)
	throws IOException, InvalidFontException, UnsupportedFontException
	{
		ArgumentType argType;

		// all of the strings that we recognize are ascii. if
		// a non-ascii literal came in, we wouldn't match it anyway,
		// so it doesn't matter what this translates to. However, we don't
		// want an exception thrown, so we call this an ISO string so that all
		// byte values are allowed.
		String theLiteral = new String(currToken.buff, 1, currToken.tokenLength-1, "ISO-8859-1");

		argType = (ArgumentType)literals.get(theLiteral);
		if (argType == null)
		{
			return false; // skip unrecognized literals
		}

		if (argType == ArgumentType.kBOOL)
		{
			Boolean arg = parseBool(tokenizer);
			if (arg != null)
				keyVal.put(theLiteral, arg);
		}

		else if (argType == ArgumentType.kSTRING)
		{
			String arg = parseString(tokenizer);
			if (arg != null)
				currentMap.put(theLiteral, arg);
		}

		else if (argType == ArgumentType.kBYTES)
		{
			byte[] arg = parseBytes(tokenizer);
			if (arg.length != 0)
				keyVal.put(theLiteral, arg);
		}

		else if (argType == ArgumentType.kNAME)
		{
			String arg = parseName(tokenizer);
			if (arg != null)
				currentMap.put(theLiteral, arg);
		} 

		else if (argType == ArgumentType.kNUMARRAY)
		{
			double[] arg = parseNumArray(tokenizer);
			if (arg.length != 0)
				keyVal.put(theLiteral, arg);
		} 

		else if (argType == ArgumentType.kBBOX)
		{
			double[] arg = parseNumArray(tokenizer);
			if (arg.length == 4)
				keyVal.put(theLiteral, new Rect(arg[0], arg[1], arg[2], arg[3]));
		}

		else if (argType == ArgumentType.kINTARRAY)
		{
			int[] arg = parseIntArray(tokenizer);
			if (arg.length != 0)
				keyVal.put(theLiteral, arg);
		}

		else if (argType == ArgumentType.kINTEGER)
		{
			Integer arg = parseInteger(tokenizer);
			if (arg != null)
				keyVal.put(theLiteral, arg);
		}

		else if (argType == ArgumentType.kNUMBER)
		{
			Double arg = parseNumber(tokenizer);
			if (arg != null)
				keyVal.put(theLiteral, arg);
		} 

		else if (argType == ArgumentType.kARRAYOF1)
		{
			double[] arg = parseNumArray(tokenizer);
			if (arg.length == 1)
				keyVal.put(theLiteral, new Double(arg[0]));
		} 

		else if (argType == ArgumentType.kITALICANGLE)
		{
			Double arg = parseNumber(tokenizer);
			if (arg != null)
				currentMap.put(theLiteral, arg);
		}

		else if (argType == ArgumentType.kCHARSTRINGS) {
			parseChars(tokenizer);
			return true;
		} 

		else if (argType == ArgumentType.kSUBRS) {
			parseSubrs(tokenizer);
		}

		else if (argType == ArgumentType.kERODE)
		{
			parseErodeProc(tokenizer);
		} 

		else if (argType == ArgumentType.kFONTTYPE)
		{
			Integer arg = parseInteger(tokenizer);
			if (arg != null && arg.intValue() == 1)
				keyVal.put(theLiteral, arg);
			else
				throw new UnsupportedFontException("Fonttype not supported");
		} 

		else if (argType == ArgumentType.kORIGFONTTYPE)
		{
			OrigFontType arg = parseOrigFontType(tokenizer);
			if (arg != null)
				keyVal.put(theLiteral, arg);
		}  

		else if (argType == ArgumentType.kENCODING)
		{
			parseEncoding(tokenizer);
		} 

		else if (argType == ArgumentType.kFONTMATRIX)
		{
			parseFontMatrix(tokenizer);
		}

		else if (argType == ArgumentType.kRNDSTEMUP)
		{
			keyVal.put(Type1Keys.LanguageGroup, new Integer(1));
		}

		else if (argType == ArgumentType.kPRIVATE)
		{
			seenPrivate = true;
		}

		else if (argType == ArgumentType.kUNSUPPORTEDFONT)
		{
			throw new UnsupportedFontException("unsupported fonttype");
		} 

		return false;

	}


	private void setupEexec(Tokenizer tokenizer)
	throws IOException, InvalidFontException
	{    

		int nextByte, i;
		boolean binaryCipher = false;
		byte[] randomSeed = new byte[8];

		// skip the delimiter
		nextByte = tokenizer.read();

		// steal the next few bytes from the tokenizer's stream to find out what we
		// are dealing with.
		for (i = 0; i < 4; i++)
		{
			nextByte = tokenizer.read();
			if (!Tokenizer.isHex(nextByte) && nextByte != ' ' && nextByte != '\t' && nextByte != '\n' && nextByte != '\r')
				binaryCipher = true;
			randomSeed[i] = (byte)nextByte;
		}

		if (binaryCipher)
		{
			BinaryEexecReader newReader = new BinaryEexecReader();

			// set up the seed
			newReader.decryptBuffer(randomSeed, 4);
			tokenizer.setReader(newReader); 
		}       
		else
		{

			/* ASCII eexec; discard eexec delimiting whitespace */
			while (Tokenizer.isWhite(randomSeed[0]))
			{
				for (i = 0; i < 3; i++)
					randomSeed[i] = randomSeed[i + 1];
				nextByte = tokenizer.read();
				randomSeed[3] = (byte)nextByte;
			}

			/* Read remaining random nibbles */
			for (i = 4; i < 8; i++)
			{
				nextByte = tokenizer.read();
				randomSeed[i] = (byte)nextByte;
			}

			AsciiEexecReader newReader = new AsciiEexecReader();

			// set up the seed.
			newReader.decryptBuffer(randomSeed, 8);
			tokenizer.setReader(newReader);
		}


	}

	private void doOperator(Tokenizer tokenizer, Token lastToken)
	throws IOException, InvalidFontException
	{
		if (SubArrays.arrayCompare(lastToken.buff, 0, Operators.kCURRENTFILE, 0, Operators.kCURRENTFILE.length))
		{
			Token token = tokenizer.getNextPSToken();
			if (SubArrays.arrayCompare(token.buff, 0, Operators.kEEXEC, 0, Operators.kEEXEC.length))
			{
				setupEexec(tokenizer);
			}

		} else if (seenPrivate && SubArrays.arrayCompare(lastToken.buff, 0, Operators.kFONTDIRECTORY, 0, Operators.kCURRENTFILE.length))
		{
			syntheticBaseMap = new HashMap();
			currentMap = syntheticBaseMap;
		}
	}

	private void parse(FontInputStream in)
	throws IOException, InvalidFontException, UnsupportedFontException
	{
		Tokenizer tokenizer;
		Token currToken;

		// peek to see if we need a pfb stream...
		int firstByte = in.read();
		in.unread(firstByte);

		if (firstByte == 0x80)
		{
			in = new PFBInputStream(in);
		}

		seenPrivate = false;

		tokenizer = new Tokenizer(in);
		for (;;)
		{	        
			currToken = tokenizer.getNextPSToken();

			// currToken.print();

			if (currToken.tokenType == TokenType.kLITERAL)
			{
				if (doLiteral(tokenizer, currToken))
					break;
			}
			else if (currToken.tokenType == TokenType.kOPERATOR)
			{
				doOperator(tokenizer, currToken);
			}

		}

		Object o = keyVal.get(Type1Keys.FullName);
		if (o != null)
		{
			String s = (String)o;
			keyVal.put(Type1Keys.CSSWeight, new Integer(NameHeuristics.fullNameToWeight(s)));
			keyVal.put(Type1Keys.CSSWidth, NameHeuristics.fullNameToWidth(s, (String)keyVal.get("FamilyName")));
			keyVal.put(Type1Keys.IsSmallCaps, Boolean.valueOf(NameHeuristics.fullNameIndicatesSmallCaps(s)));
		}

	}

	Object getValue(String key)
	{
		return keyVal.get(key);
	}

	/**
	 * Looks up a glyph by name and returns its glyphID
	 * @param glyphName The name of the glyph
	 * @return 0 if the glyphName is .notdef or if the glyphName
	 * is not found in the font.
	 */
	public int glyphName2gid(String glyphName)
	{
		Type1Glyph tmp = new Type1Glyph(glyphName);
		int index = java.util.Arrays.binarySearch(charStrings, tmp, CharstringCompare.comparator);

		if (index < 0)
			return 0;

		if (charStrings[index].glyphName.equals(glyphName))
			return index;

		return 0;
	}

	public String charCode2GlyphName(int charCode)
	{
		Object o = getValue(Type1Keys.StandardEncoding);
		if (o != null && ((Boolean) o).booleanValue ()) {
			if (0 <= charCode && charCode < StandardEncoding.names.length) {
				return StandardEncoding.names[charCode];
			} else {
				return ".notdef"; }}

		o = getValue(Type1Keys.Encoding);
		if (o != null) {
			byte[][] encoding = (byte[][]) o;
			byte[] glyphName = encoding [charCode];
			return bytesToString (glyphName); 
		}

		return ".notdef";
	}

	public int charCode2gid (int charCode)
	{
		return glyphName2gid (charCode2GlyphName (charCode)); 
	}

	//--------------------------------------------------------- general access ---

	/** Fetch the number of glyphs in the font */
	public int getNumGlyphs()
	{
		return charStrings.length;
	}

	/** Fetch the bytes for the charstring associated with glyphID */
	byte[] getCharstring(int glyphID)
	{
		return charStrings[glyphID].charString;
	}

	public ROS getROS()
	{
		return null;
	}

	/** Fetch the glyphName associated with a given glyphID */
	public String getGlyphName(int glyphID)
	{
		return charStrings[glyphID].glyphName;
	}

	/** Fetch the number of subroutines in a font */
	int getNumSubrs()
	{
		return subrs.length;
	}

	/** Fetch the bytes for a subroutine */
	byte[] getSubr(int subrNumber)
	{
		return subrs[subrNumber];
	}

	/** Fetch the font name (aka Postscript name) for the font */
	public String getPostscriptName()
	{
		Object o = getValue(Type1Keys.FontName);
		if (o != null)
			return (String)o;
		return null;
	}

	/** Return the vertical stem width of the font.
	 * 
	 * <p>This horizontal metric is expressed in the metric space of the font.
	 */
	public double getStemV()
	{
		{ Double o = (Double) getValue(Type1Keys.StdVW);
		if (o!= null)
			return o.doubleValue() * getFontMatrix ().a * getUnitsPerEmX (); }

		{ double[] o = (double[]) getValue(Type1Keys.StemSnapV);
		if (o!=null && o.length != 0)
			return o[0] * getFontMatrix ().a * getUnitsPerEmX ();
		}

		return 0;
	}

	//------------------------------------------------------------- unitsPerEm ---
	/** The Type1 font matrix.
	 */
	public Matrix getFontMatrix()
	{      
		Matrix m = (Matrix) getValue(Type1Keys.FontMatrix);
		if (m != null)
		{
			return m;
		} else
		{ 
			return defaultFontMatrix;            
		}
	}

	// We have two good choices for the unitsPerEm{X,Y}:
	//
	// 1. use 1/a and 1/d; if the font matrix is diagonal
	//    (i.e. b = c = tx = ty = 0), then the font space
	//    and the metric space coincide.
	//
	// 2. use 1000 and 1000; this is because some of the
	//    metrics in the font are in thousands of em, i.e.
	//    already in that metric space.
	//
	// For the vast majority of fonts, which have a diagonal
	// font matrix with a = d = 0.001, the choices coincide,
	// so it does not truly matter which one we choose. 
	// Having constants is probably going to help 
	// a bit, since they can be inlined.

	// FOR NOW: use 1/a and 1/d, until the tests are fixed
	public double getUnitsPerEmX () {
		return 1.0d / getFontMatrix ().a;
	}

	public double getUnitsPerEmY () {
		return 1.0d / getFontMatrix ().d;
	}

	// This matrix allows us to convert point in the font space to the
	// metric space.
	private Matrix getFontToMetricsMatrix () {
		Matrix m = getFontMatrix ();
		double x = getUnitsPerEmX ();
		double y = getUnitsPerEmY ();
		return new Matrix (x * m.a, y * m.b, x * m.c, y * m.d, x * m.tx, y * m.ty);
	}

	//----------------------------------------------------------------- bboxes ---

	public Rect getFontBBox () {
		Rect rawBBox = (Rect) getValue ("FontBBox");
		if (rawBBox == null) {
			return null; }       
		return rawBBox.applyMatrix (getFontToMetricsMatrix ());
	}

	public Rect getCoolTypeRawFontBBox () {
		return getFontBBox ();
	}


	//----------------------------------------------------------------- script ---

	public CoolTypeScript getCoolTypeScript ()
	throws UnsupportedFontException, InvalidFontException {

		return ScriptHeuristics.getCoolTypeScript (getPostscriptName (), 
				256, 
				new GlyphNamesAccessor  ()
		{ public String getAGlyphName (int ccode) 
		throws UnsupportedFontException, InvalidFontException { 
			return charCode2GlyphName(ccode); }} );
	}

	//-------------------------------------------------------------- ideoEmBox ---

	/** Compute the IdeoEmBox from the CapHeight.
	 */
	private Rect getCoolTypeIdeoEmBoxFromCapHeight ()
	throws InvalidFontException, UnsupportedFontException {

		double capHeight = getCoolTypeCapHeight ();
		if (Double.isNaN (capHeight)) {
			return null; }

		double unitsPerEmX = getUnitsPerEmX ();
		double unitsPerEmY = getUnitsPerEmY ();
		double ymin = - (unitsPerEmY - capHeight) / 2.0d;
		return new Rect (0, ymin, unitsPerEmX,  ymin + unitsPerEmY);
	}

	/** Compute the IdeoEmBox.
	 * Various heuristics are applied until the IdeoEmBox can be determined.
	 */
	public Rect getCoolTypeIdeoEmBox () 
	throws InvalidFontException, UnsupportedFontException {

		Rect r;

		if (useCoolTypeCJKHeuristics ()) {

			if ((r = getCoolTypeIdeoEmBoxFromFullBoxCharacter ()) != null) {
				return r; }

			if ((r = getCoolTypeIdeoEmBoxFromTypicalCharacter ()) != null) {
				return r; }}

		else {
			if ((r = getCoolTypeIdeoEmBoxFromCapHeight ()) != null) {
				return r; }}

		double unitsPerEmX = getUnitsPerEmX ();
		double unitsPerEmY = getUnitsPerEmY ();
		return new Rect (0, -0.120 * unitsPerEmY, unitsPerEmX, 0.880 * unitsPerEmY);
	}

	//----------------------------------------------------------------- icfBox ---

	/** Compute the IcfBox.
	 * 
	 * <p>This metric is expressed in the metric space of the font.
	 */
	public Rect getCoolTypeIcfBox ()
	throws InvalidFontException, UnsupportedFontException {

		Rect ideoEmBox = getCoolTypeIdeoEmBox ();

		Rect r;

		if ((r = getCoolTypeIcfBoxFromTypicalCharacter (ideoEmBox)) != null) {
			return r; }

		return getCoolTypeIcfBoxFromIdeoEmBox (ideoEmBox);
	}

	//----------------------------------------------------------- line metrics ---

	/** Emulates the CoolType API CTFontDict:GetHorizontalMetrics CoolType API.
	 * 
	 * <p>This vertical metric is expressed in the metric space of the font.
	 * 
	 * <p>See also the {@link #getLineMetrics()} method.
	 * 
	 * @throws UnsupportedFontException
	 * @throws InvalidFontException
	 */
	public LineMetrics getCoolTypeLineMetrics ()
	throws UnsupportedFontException, InvalidFontException {

		return getCoolTypeLineMetricsFromFontBbox ();
	}

	//------------------------------------------------------ underline metrics ---
	public UnderlineMetrics getCoolTypeUnderlineMetrics ()
	throws UnsupportedFontException, InvalidFontException {

		double position = -150;
		Object o = getValue ("UnderlinePosition");
		if (o != null)
			position = ((Double)o).doubleValue();

		double thickness = 50;
		o = getValue ("UnderlineThickness");
		if (o != null)
			thickness = ((Double)o).doubleValue();

		double cooltypeUnitsPerEm = getCoolTypeUnitsPerEm ();
		double unitsPerEmY =  getUnitsPerEmY ();

		return new UnderlineMetrics (
				position * unitsPerEmY / cooltypeUnitsPerEm,
				thickness * unitsPerEmY / cooltypeUnitsPerEm);
	}

	//----------------------------------------------------------------- cmap ---

	int getFirstChar() throws InvalidFontException, UnsupportedFontException {
		synchronized (cmapMutex) {
			initCmap();    		
			return cmap.getFirstSupportedChar(); }
	}

	int getLastChar() throws InvalidFontException, UnsupportedFontException {
		synchronized (cmapMutex) {
			initCmap();    		
			return cmap.getLastSupportedChar(); }
	}

	// The cmap object is constructed once on demand, and is protected
	// by the cmapMutex mutex.
	UnicodeCmap cmap = null;
	Object cmapMutex = new Object ();

	private boolean isDingbat () {
		Object o = getValue(Type1Keys.FullName);
		return NameHeuristics.fullNameIndicatesDingbats (o != null ? (String) o : null);
	}

	private void initCmap() throws InvalidFontException, UnsupportedFontException {
		if (cmap == null) {
			cmap = UnicodeCmap.computeCmapFromGlyphNames 
			(getNumGlyphs (), 
					isDingbat (), 
					new GlyphNamesAccessor  ()
			{ public String getAGlyphName (int gid) 
			throws UnsupportedFontException, InvalidFontException { 
				return getGlyphName (gid); }} ); }
	}

	/** {@inheritDoc} */
	public int getGlyphForChar (int usv) 
	throws InvalidFontException, UnsupportedFontException {
		synchronized (cmapMutex) {
			initCmap();    		
			return cmap.getGlyphForChar(usv); }
	}

	//-------------------------------------------------- individual glyph data ---


	/** Get the horizontal advance of the glyph <code>glyphID</code>. 
	 * 
	 * <p>This horizontal metric is expressed in the metric space of the font.
	 */
	public double getHorizontalAdvance(int glyphID)
	throws InvalidGlyphException, UnsupportedFontException
	{
		Type1WidthFetcher fetcher= new Type1WidthFetcher();
		Type1CStringParser parser = new Type1CStringParser();
		parser.parse(fetcher, this, glyphID);

		return fetcher.getWidth() * (getFontMatrix ().a * getUnitsPerEmX ());
	}


	/** Get the horizontal kerning between two glyphs.
	 * 
	 * <p>This horizontal metric is expressed in the metric space of the font.
	 */
	public double getKernValue(int leftGID, int rightGID)
	throws InvalidFontException, UnsupportedFontException
	{
		if (metricFile == null)
		{
			return 0.0;
		}

		// the rawKernValue is expressed in the font in 1/1000th of an em.
		double rawKernValue = metricFile.getKernValue(getGlyphName(leftGID), getGlyphName(rightGID));
		return rawKernValue * getUnitsPerEmX () / 1000.0d;
	}

	public void getGlyphOutline (int gid, OutlineConsumer consumer)
	throws InvalidFontException, UnsupportedFontException {
		Type1OutlineParser parser = new Type1OutlineParser ();
		parser.parse (this, gid, consumer);
	}

	/** Get the bounding box for a glyph, in metric space. */
	public Rect getGlyphBBox (int gid) 
	throws UnsupportedFontException, InvalidFontException {

		Matrix m = new Matrix (getUnitsPerEmX (), 0, 0, getUnitsPerEmY (), 0, 0);
		return new Type1GlyphBBoxCalculator (m).calculateBBox (this, gid);
	}

	//----------------------------------------------------------------------------
	public Scaler getScaler (ScanConverter c)
	throws InvalidFontException, UnsupportedFontException {

		if (c == null) {
			c = new CScan (false, 1.0d, true); }
		return new Type1Scaler (this, c); 
	}

	//--------------------------------------------------- CSSFontDescription ---

	/** Return the set of CSS font-family names that this font matches. */
	public Set getCSSFamilyNames ()
	{
		Set s = new HashSet();
		String familyName = null;

		if (metricFile != null )
		{
			familyName = metricFile.getFamilyName();
			if (familyName != null)
				s.add(familyName);
		}

		Object o = getValue(Type1Keys.FamilyName);
		if (o != null)
		{
			s.add(o);            
		}

		return s;
	}

	public String getPreferredCSSFamilyName()
	{
		if (metricFile != null )
		{
			String familyName = metricFile.getFamilyName();
			if (familyName != null)
				return familyName;
		}

		Object o = getValue(Type1Keys.FamilyName);
		if (o != null)
			return (String)o;

		return null;
	}

	/** Tell if the font matches the CSS font-style normal. */
	public boolean isCSSStyleNormal ()
	{
		if (metricFile != null && metricFile instanceof PFM)
		{
			return !((PFM)metricFile).isItalic();
		}

		Object o = getValue(Type1Keys.ItalicAngle);
		if (o == null)
			return true;

		Double d = (Double)o;

		if (d.doubleValue() == 0)
			return true;

		return false;
	}

	/** Tell if the font matches the CSS font-style italic. */
	public boolean isCSSStyleItalic ()
	{
		return !isCSSStyleNormal();
	}

	/** Tell if the font matches the CSS font-style oblique. */
	public boolean isCSSStyleOblique ()
	{
		return !isCSSStyleNormal();
	}

	/** Tell if the font matches the CSS font-variant normal. */
	public boolean isCSSVariantNormal ()
	{
		Object o = getValue(Type1Keys.IsSmallCaps);
		if (o != null)
		{
			return !((Boolean)o).booleanValue();
		}

		return true;
	}

	/** Tell if the font matches the CSS font-style small-caps. */
	public boolean isCSSVariantSmallCaps ()
	{
		Object o = getValue(Type1Keys.IsSmallCaps);
		if (o != null)
		{
			return ((Boolean)o).booleanValue();
		}

		return false;
	}

	/** Return the CSS weight of this font. */
	public int getCSSWeight () 
	{
		if (metricFile != null)
		{
			int retval = metricFile.getWeight();
			if (retval != 0)
				return retval;
		} 

		Object o = getValue(Type1Keys.CSSWeight);
		if (o != null)
			return ((Integer)o).intValue();
		else
			return 400;

	}

	/** Return the CSS stretch associated with this font */
	public CSSStretchValue getCSSStretchValue()
	{
		Object o = getValue(Type1Keys.CSSWidth);
		if (o != null)
			return (CSSStretchValue)o;
		else
			return CSSStretchValue.NORMAL;
	}

	public CacheSupportInfo getCacheSupportInfo()
		throws InvalidFontException, UnsupportedFontException
	{
		return new CacheSupportInfo(getClass().getSimpleName(), getNumGlyphs(), false);
	}

	//-------------------------------------------- PostScriptFontDescription ---

	public PostscriptFontDescription[] getPostscriptFontDescription () 
	{
		String s = getPostscriptName();
		if (s != null) {
			return new PostscriptFontDescription[] {new PostscriptFontDescription(s)}; }
		else {
			return new PostscriptFontDescription[0]; }
	}

	//--------------------------------------------------- SWFFont3FontDescription ---
	private class T1SWFFont3Description implements SWFFontDescription
	{
		private final boolean wasEmbedded;
		private final LineMetrics metrics;

		T1SWFFont3Description(boolean wasEmbedded) throws UnsupportedFontException, InvalidFontException
		{
			this.wasEmbedded = wasEmbedded;
			this.metrics = getCoolTypeLineMetrics();
		}

		public boolean canDisplay(char c) throws UnsupportedFontException, InvalidFontException {
			return getGlyphForChar(c) != 0;
		}

		public double getAscent() throws InvalidFontException, UnsupportedFontException {

			return metrics.ascender * getUnitsPerEmX() / getUnitsPerEmY();
		}

		public double getDescent() throws InvalidFontException, UnsupportedFontException {
			return -metrics.descender * getUnitsPerEmX() / getUnitsPerEmY();
		}

		public double getLineGap() throws InvalidFontException, UnsupportedFontException {
			return metrics.linegap * getUnitsPerEmX() / getUnitsPerEmY();
		}

		public double getEmScale() throws InvalidFontException, UnsupportedFontException {
			return getUnitsPerEmX();
		}

		public String getFullName() throws InvalidFontException, UnsupportedFontException {
			return (String)Type1Font.this.getValue (Type1Keys.FullName);
		}

		public String getFamily() throws InvalidFontException, UnsupportedFontException {
			return Type1Font.this.getPreferredCSSFamilyName();
		}

		public String getCopyright() throws InvalidFontException, UnsupportedFontException {
			byte[] tmp = (byte[])getValue(Type1Keys.Copyright);
			return (tmp == null) ? null : bytesToString(tmp);
		}

		public String getTrademark() throws InvalidFontException, UnsupportedFontException {
			byte[] tmp = (byte[])getValue(Type1Keys.Notice);
			return (tmp == null) ? null : bytesToString(tmp);
		}

		public int getFirstChar() throws InvalidFontException, UnsupportedFontException {
			return Type1Font.this.getFirstChar();
		}

		public double getHorizontalAdvance(char ccode) throws UnsupportedFontException, InvalidFontException {
			int gid = getGlyphForChar(ccode);
			return Type1Font.this.getHorizontalAdvance(gid);
		}

		public int getLastChar() throws InvalidFontException, UnsupportedFontException {
			return Type1Font.this.getLastChar();
		}

		public int getNumGlyphs() throws InvalidFontException, UnsupportedFontException {
			return Type1Font.this.getNumGlyphs();
		}

		public void getOutline(char ccode, OutlineConsumer consumer) throws UnsupportedFontException, InvalidFontException {
			int gid = getGlyphForChar(ccode);
			Type1Font.this.getGlyphOutline(gid, consumer);
		}

		public Permission getPermissions() throws InvalidFontException, UnsupportedFontException {
			Permission perm = Type1Font.this.getEmbeddingPermission(wasEmbedded);
			if (perm == Permission.ILLEGAL_VALUE)
				perm = Permission.EDITABLE;

			return perm;
		}

		public String getPostscriptName() throws InvalidFontException, UnsupportedFontException {
			return Type1Font.this.getPostscriptName();
		}


		public boolean isBold() throws InvalidFontException, UnsupportedFontException {
			return Type1Font.this.getCSSWeight() >= 700;
		}

		public boolean isItalic() throws InvalidFontException, UnsupportedFontException {
			return Type1Font.this.isCSSStyleItalic();
		}

		public String getSubFamily() throws InvalidFontException, UnsupportedFontException {
			return null;
		}

	}

	//---------------------------------------------------------- FXG Properties ---
	public FXGFontDescription[] getFXGFontDescription(Platform platform,	ULocale locale) 
	throws InvalidFontException, UnsupportedFontException 
	{
		return new FXGFontDescription[0];
	}

	//---------------------------------------------------------- Platform Properties ---
	public PlatformFontDescription[] getPlatformFontDescription(Platform platform, ULocale locale)
	throws InvalidFontException, UnsupportedFontException
	{
		return new PlatformFontDescription[0];
	}

	//--------------------------------------------------- PDFFontDescription ---

	private class T1XDCFontDescription extends XDCFontDescription {
		public int getNumGlyphs ()
		throws UnsupportedFontException, InvalidFontException {
			return Type1Font.this.getNumGlyphs ();
		}


		public String getGlyphName(int gid)
		throws InvalidFontException, UnsupportedFontException
		{
			return Type1Font.this.getGlyphName(gid);
		}

		public int getGlyphCid(int glyphID)
		throws UnsupportedFontException, InvalidFontException
		{
			return -1;
		}

		public ROS getROS() { return null;}

		public boolean pdfFontIsTrueType() { return false;}

		public String getPostscriptName() throws InvalidFontException, UnsupportedFontException
		{
			return Type1Font.this.getPostscriptName();
		}

		public String getFontFamily() throws InvalidFontException, UnsupportedFontException
		{
			Object o = getValue(Type1Keys.FamilyName);
			if (o != null)
				return (String)o;
			return null;
		}

		public double getAdvance(int glyphID)
		throws InvalidGlyphException, UnsupportedFontException, InvalidFontException
		{
			return getHorizontalAdvance (glyphID) * 1000.0d / getUnitsPerEmX ();
		}

		public Rect getFontBBox() throws InvalidFontException, UnsupportedFontException
		{
			Rect bbox = (Rect) Type1Font.this.getValue ("FontBBox");
			if (bbox == null)
				return null;

			return bbox.applyMatrix (getFontMatrix ().multiply (1000.0d));
		}

		public double getStemV() throws InvalidFontException, UnsupportedFontException
		{
			return Type1Font.this.getStemV() * 1000.0d / getUnitsPerEmX ();
		}

		public double getCapHeight () throws UnsupportedFontException, InvalidFontException {
			double capHeight = getCoolTypeCapHeight ();
			if (Double.isNaN (capHeight)) {
				return 0; }
			else {
				return capHeight * 1000.0d / getUnitsPerEmY (); }
		}

		public double getXHeight() throws UnsupportedFontException, InvalidFontException {
			double xHeight = getCoolTypeXHeight();
			if (Double.isNaN(xHeight))
				return 0;
			else
				return xHeight * 1000.0d/getUnitsPerEmY();
		}

		public double getItalicAngle() throws InvalidFontException, UnsupportedFontException
		{
			Object o = getValue("ItalicAngle");
			if (o!= null)
				return ((Double)o).doubleValue();
			return 0;
		}

		public String getBase14Name()
		{
			return null;
		}


		public boolean isSerifFont() throws InvalidFontException, UnsupportedFontException {
			int gid1, gid2;
			gid1 = Type1Font.this.getGlyphForChar('l');
			if (gid1 == 0)
			{
				gid1 = Type1Font.this.glyphName2gid("Lsmall");
			}

			gid2 = Type1Font.this.getGlyphForChar('I');
			return Type1Font.this.isSerifFont(gid1, gid2, getItalicAngle());
		}


		public boolean isSmallCapFont() 
		throws InvalidFontException, UnsupportedFontException 
		{
			if (!isAllCapFont()) {
				int gid1 = Type1Font.this.getGlyphForChar('h');
				if (gid1 == 0) {
					gid1 = Type1Font.this.glyphName2gid("Hsmall");
				}

				int gid2 = Type1Font.this.getGlyphForChar('x');
				if (gid2 == 0) {
					gid2 = Type1Font.this.glyphName2gid("Xsmall");
				}

				if (gid2 != 0) {	
					Rect bbox = Type1Font.this.getGlyphBBox(gid2);
					return Type1Font.this.isSmallCapFont(gid1, bbox.ymax);
				}
			}

			return false;
		}


		public boolean isAllCapFont() throws InvalidFontException, UnsupportedFontException {
			int gid1 = Type1Font.this.getGlyphForChar('K');
			int gid2 = Type1Font.this.getGlyphForChar('k');
			if (gid2 == 0) {
				gid2 = Type1Font.this.glyphName2gid("Ksmall");
			}

			return Type1Font.this.isAllCapFont(gid1, gid2);
		}

		public int getCIDCount()
		{
			return -1;
		}

		public void subsetAndStream(Subset subset, OutputStream out, boolean preserveROS)
		throws InvalidFontException, UnsupportedFontException, IOException
		{
			Type1Font.this.subsetAndStream(subset, out, preserveROS);
		}

		public void subsetAndStream(SubsetSimpleType1 t1Subset, OutputStream out)
		throws InvalidFontException, UnsupportedFontException, IOException
		{
			Type1Font.this.subsetAndStream(t1Subset, out);
		}

		public void subsetAndStream(SubsetSimpleTrueType ttSubset, OutputStream out)
		throws UnsupportedFontException
		{
			throw new UnsupportedFontException("Not a TrueType font");
		}

		public CodePage[] getXDCCodePages()
		throws InvalidFontException, UnsupportedFontException
		{
			HashSet codePageSet = new HashSet();
			CoolTypeScript script = getCoolTypeScript();
			if (script == CoolTypeScript.ROMAN) {
				if (getGlyphForChar('A') > 0)
					codePageSet.add(CodePage.ROMAN1);
			} else if (script == CoolTypeScript.EAST_EUROPEAN_ROMAN)
				codePageSet.add(CodePage.ROMAN2);
			else if (script == CoolTypeScript.JAPANESE)
				codePageSet.add(CodePage.JAPANESE);
			else if (script == CoolTypeScript.KOREAN)
				codePageSet.add(CodePage.KOREAN);
			else if (script == CoolTypeScript.SIMPLIFIED_CHINESE)
				codePageSet.add(CodePage.SIMPLIFIED_CHINESE);
			else if (script == CoolTypeScript.TRADITIONAL_CHINESE)
				codePageSet.add(CodePage.TRADITIONAL_CHINESE);
			if (glyphName2gid("ecircumflex") > 0)
				codePageSet.add(CodePage.ROMAN1);
			if (glyphName2gid("Ccaron") > 0 || glyphName2gid("ncaron") > 0)
				codePageSet.add(CodePage.ROMAN2);
			if (glyphName2gid("akatakana") > 0)
				codePageSet.add(CodePage.JAPANESE);
			CodePage[] retVal = new CodePage[codePageSet.size()];
			Iterator iter = codePageSet.iterator();
			int index = 0;
			while (iter.hasNext())
				retVal[index++] = (CodePage)iter.next();
			return retVal;
		}


		public void stream(OutputStream out, boolean openTypeFontsAllowed) 
		throws InvalidFontException, UnsupportedFontException, IOException {
			Type1Font.this.stream(out, openTypeFontsAllowed);
		}
	}

	public PDFFontDescription getPDFFontDescription(Font font)
	{
		return xdcDescription;
	}

	public XDCFontDescription getXDCFontDescription(Font font)
	{
		return xdcDescription;
	}

	public boolean getCoolTypeProportionalRomanFromFontProperties() throws InvalidFontException
	{
		Boolean o = (Boolean)getValue(Type1Keys.isFixedPitch);
		if (o != null)
		{
			return !o.booleanValue();
		}

		return true;
	}

	//---------------------------------------------------------- permissions ---

	public Permission getEmbeddingPermission(boolean wasEmbedded)
	{
		Integer fsType = (Integer)getValue(Type1Keys.FSType);
		if (fsType != null)
		{
			return EmbeddingPermission.interpretFSType(fsType.intValue());
		}

		OrigFontType oft = (OrigFontType)getValue(Type1Keys.OrigFontType);
		if (oft == null || oft == OrigFontType.kTYPE1)
		{
			String name = this.getPostscriptName();
			byte[] tmp = (byte[])getValue (Type1Keys.Notice);
			String notice = (tmp == null) ? null : bytesToString(tmp);
			return EmbeddingPermission.getType1DefaultPermission(notice, name);

		} else if (wasEmbedded) 
		{
			return EmbeddingPermission.getDefaultWasEmbeddedPermission();
		} else if (oft == OrigFontType.kCID)
		{
			int[] v = (int[])getValue(Type1Keys.XUID);
			return EmbeddingPermission.getCIDDefaultPermission(v);
		} else if (oft == OrigFontType.kTRUETYPE)
		{
			return EmbeddingPermission.getTrueTypeDefaultPermission();
		} else
		{
			// oft == ocf
			return EmbeddingPermission.getOCFDefaultPermission();
		}
	}

	//-------------------------------------------------- cff conversion ---
	public Subset createSubset () throws UnsupportedFontException, InvalidFontException
	{
		return new Type1Subset(this, true);
	}

	private String bytesToString(byte[] bytes)
	{
		char[] chars = new char [bytes.length];
		for (int i = 0; i < bytes.length; i++) {
			chars [i] = (char)(bytes[i] & 0xff); }

		return new String (chars);
	}

	private Dict createTopDict(boolean nameKeyed)
	throws InvalidFontException
	{
		String registry, ordering;
		String fullname, familyname, weight, basefontname, fontname;
		String notice, version, copyright, postscript;
		Double italicangle, underPos, underThick, strokewidth;
		Rect bbox;
		Boolean fixedPitch;
		Integer paintType, uniqueID, fsType;
		int[] xuid, bfBlend;
		byte[] tmp;
		Matrix fontMatrix;

		if (nameKeyed) {
			registry = null;
			ordering = null;
			fontMatrix = (Matrix)getValue (Type1Keys.FontMatrix);
		} else {
			registry = "Adobe";
			ordering = "Identity";
			fontMatrix = null;
		}

		tmp = (byte[])getValue (Type1Keys.version);
		version = (tmp == null) ? null : bytesToString(tmp);

		tmp = (byte[])getValue (Type1Keys.Notice);
		notice = (tmp == null) ? null : bytesToString(tmp);

		tmp = (byte[])getValue (Type1Keys.Copyright);
		copyright = (tmp == null) ? null : bytesToString(tmp);

		tmp = (byte[])getValue (Type1Keys.PostScript);
		postscript = (tmp == null) ? null : bytesToString(tmp);

		OrigFontType oft = (OrigFontType)getValue(Type1Keys.OrigFontType);
		if (oft != null)
		{
			if (postscript == null)
				postscript = "/OrigFontType /" + oft.toString() + " def";
			else
				postscript = postscript + " /OrigFontType /" + oft.toString() + " def";
		} else
		{
			if (postscript == null)
				postscript = "/OrigFontType /Type1 def";
			else
				postscript = postscript + " /OrigFontType /Type1 def";
		}

		fullname = (String)getValue (Type1Keys.FullName);
		familyname = (String)getValue (Type1Keys.FamilyName);
		weight = (String)getValue (Type1Keys.Weight);
		fixedPitch = (Boolean)getValue(Type1Keys.isFixedPitch);
		italicangle = (Double)getValue(Type1Keys.ItalicAngle);
		underPos = (Double)getValue (Type1Keys.UnderlinePosition);
		underThick = (Double)getValue (Type1Keys.UnderlineThickness);
		paintType = (Integer)getValue(Type1Keys.PaintType);
		uniqueID = (Integer)getValue (Type1Keys.UniqueID); 
		bbox = (Rect)getValue(Type1Keys.FontBBox);
		strokewidth = (Double)getValue(Type1Keys.StrokeWidth);
		xuid = (int[])getValue (Type1Keys.XUID); 
		fsType = (Integer)getValue (Type1Keys.FSType);
		bfBlend = (int[])getValue (Type1Keys.BaseFontBlend);
		basefontname = (String)getValue (Type1Keys.BaseFontName);

		fontname = (String)getValue (Type1Keys.FontName);

		return new Dict(registry, ordering, 0, 
				version, notice, copyright, fullname, fontname, familyname,
				weight, fixedPitch, italicangle, underPos, underThick,
				paintType, uniqueID, bbox, strokewidth, xuid,
				postscript, fsType, bfBlend, basefontname, fontMatrix);
	}

	private Dict createPrivateDict(double nominalWidth, double defaultWidth)
	{
		Double blueScale, blueShift;
		Double expansionFactor, stdHW, stdVW;
		Boolean forceBold;
		Integer bluefuzz, langGroup, initRandSeed;
		double[] blues, otherBlues, familyBlues, familyOtherBlues;
		double[] stemSnapH, stemSnapV;

		blues = (double[])getValue (Type1Keys.BlueValues);
		otherBlues = (double[])getValue (Type1Keys.OtherBlues);
		familyBlues = (double[])getValue (Type1Keys.FamilyBlues);
		familyOtherBlues = (double[])getValue (Type1Keys.FamilyOtherBlues);
		blueScale = (Double)getValue (Type1Keys.BlueScale);
		blueShift = (Double)getValue (Type1Keys.BlueShift);
		bluefuzz = (Integer)getValue(Type1Keys.BlueFuzz);
		stdHW = (Double)getValue (Type1Keys.StdHW);
		stdVW = (Double)getValue (Type1Keys.StdVW);
		stemSnapH = (double[])getValue (Type1Keys.StemSnapH);
		stemSnapV = (double[])getValue (Type1Keys.StemSnapV );
		forceBold = (Boolean)getValue (Type1Keys.ForceBold);
		langGroup = (Integer)getValue (Type1Keys.LanguageGroup);
		expansionFactor = (Double)getValue (Type1Keys.ExpansionFactor);
		initRandSeed = (Integer)getValue (Type1Keys.initialRandomSeed);

		return new Dict(blues, otherBlues, familyBlues, familyOtherBlues,
				blueScale, blueShift, bluefuzz, stdHW, stdVW, stemSnapH, stemSnapV,
				forceBold, langGroup, expansionFactor, initRandSeed,
				new Double(nominalWidth), new Double(defaultWidth));
	}

	/**
	 * Creates a Type2CStringGenerator, parses the glyphs in Subset, and
	 * adds them to the Type2CStringGenerator.
	 * @param subset A Subset containing all of the glyphs that need to be converted to type2.
	 * @return a Type2CStringGenerator that knows about all of the glyphs in Subset.
	 * @throws InvalidFontException
	 * @throws UnsupportedFontException
	 */
	private Type2CStringGenerator seedGenerator(Subset subset)
	throws InvalidFontException, UnsupportedFontException
	{
		int numGlyphs = ((subset == null) ? getNumGlyphs(): subset.getNumGlyphs());

		Type2CStringGenerator cstringGenerator 
		= new Type2CStringGenerator(numGlyphs, 1);
		Type1CStringParser parser = new Type1CStringParser();

		int fullGid;

		for (int i = 0; i < numGlyphs; i++)
		{
			fullGid = ((subset == null) ? i: subset.getFullGid(i));
			cstringGenerator.newGlyph(i, 0, Type2CStringGenerator.DEFAULT_DEFAULT_WIDTH, Type2CStringGenerator.DEFAULT_NOMINAL_WIDTH);
			parser.parse(cstringGenerator, this, fullGid);
		}

		return cstringGenerator;
	}

	/**
	 * Convert this to a cid cff font containing the glyphs in subset.
	 * @param subset A subset containing all of the glyphs that need to go in the new font.
	 * @return a FontData that represents a cid cff font.
	 * @throws InvalidFontException
	 * @throws UnsupportedFontException
	 */
	private CIDKeyedFont toCID (Subset subset) 
	throws InvalidFontException, UnsupportedFontException {
		String fontname;
		CharStrings charstrings;
		double nominalWidth;
		double defaultWidth;	
		Type2CStringGenerator cstringGenerator;
		Dict topDict, privateDict, cidDict;
		Matrix matrix;

		// CFF Top dict
		topDict = createTopDict(false);

		// generate charstring index.	   
		cstringGenerator = seedGenerator(subset);
		nominalWidth = cstringGenerator.calculateNominalWidth(0);
		defaultWidth = cstringGenerator.calculateDefaultWidth(0);
		charstrings = cstringGenerator.getCharstringIndex();

		// values for CIDFontDict
		matrix = (Matrix)getValue (Type1Keys.FontMatrix);
		cidDict = new Dict(matrix);

		//private dict
		privateDict = createPrivateDict(nominalWidth, defaultWidth);

		fontname = (String)getValue (Type1Keys.FontName);
		return new com.adobe.fontengine.font.cff.CIDKeyedFont(fontname, topDict,
				charstrings, 
				subset.getNumGlyphs(),
				cidDict, privateDict);
	}

	/** Subset and stream this font for PDF use. 
	 * The stream is a CID-keyed CFF stream,.
	 * @param out the OutputStream to which the bytes are streamed
	 */
	public void subsetAndStream (Subset subset, OutputStream out, boolean preserveROS)
	throws InvalidFontException, UnsupportedFontException, IOException {
		CIDKeyedFont cid = toCID (subset);

		Integer fsType = (Integer)getValue(Type1Keys.FSType);

		// the conversion to a cid cff causes subsetting to happen. create a new subset containing
		// all the glyphs in the cid cff, and subsetAndStream it.
		Subset completeSubset = new SubsetDefaultImpl(subset.getNumGlyphs(), false);
		cid.subsetAndStream (completeSubset, out, false, fsType);
	}

	private NameKeyedFont createNKFontFromSubset(Subset subset) 
	throws InvalidFontException, UnsupportedFontException
	{
		Dict topDict = createTopDict(true);
		String fontName = (String)getValue(Type1Keys.FontName);
		Type2CStringGenerator cstringGenerator = seedGenerator(subset);
		double nominalWidth = cstringGenerator.calculateNominalWidth(0);
		double defaultWidth = cstringGenerator.calculateDefaultWidth(0);
		CharStrings charStrings = cstringGenerator.getCharstringIndex();
		Dict privateDict = createPrivateDict(nominalWidth, defaultWidth);

		return new NameKeyedFont(fontName, topDict, charStrings, privateDict);
	}

	private static class FetchGNameFromT1Subset extends GlyphNameFetcher {
		private final SubsetSimpleType1 t1Subset;

		FetchGNameFromT1Subset(SubsetSimpleType1 t1Subset) 
		{
			this.t1Subset = t1Subset;
		}

		public String getGlyphName(Subset s, int i)
		{
			if (i == 0)
				return ".notdef";
			return t1Subset.getGlyphNames()[i-1];
		}
	}

	public void subsetAndStream(SubsetSimpleType1 t1Subset, OutputStream out)
	throws InvalidFontException, UnsupportedFontException, IOException
	{
		Subset subset = createSubset();							// Create a Type1Subset
		String[] glyphNames = t1Subset.getGlyphNames();					// Subset glyph name list
		for (int i = 0; i < glyphNames.length; i++) {
			String glyphName = glyphNames[i];
			if (glyphName != null) {
				int gid = glyphName2gid(glyphName);
				if (gid == 0 && !glyphName.equals(".notdef"))
					throw new InvalidFontException("Font does not contain required codepoint");
				subset.getSubsetGid(gid);					// Add to subset if not done yet
			}
		}

		NameKeyedFont nk = createNKFontFromSubset(subset);
		Integer fsType = (Integer)getValue(Type1Keys.FSType);
		nk.subsetAndStream(subset, out, fsType, false, new FetchGNameFromT1Subset(t1Subset));
	}

	public void stream(OutputStream out, boolean openTypeOk) 
	throws InvalidFontException, UnsupportedFontException, IOException
	{
		Subset subset = new Type1Subset(this, false);
		NameKeyedFont nk = createNKFontFromSubset(subset);
		Integer fsType = (Integer)getValue(Type1Keys.FSType);

		nk.subsetAndStream(subset, out, fsType, false,
				new GlyphNameFetcher() {
			public String getGlyphName(Subset s, int i) {return Type1Font.this.getGlyphName(i);}});

	}

	//----------------------------------------------------------------------------

	public CatalogDescription getSelectionDescription () 
	throws InvalidFontException, UnsupportedFontException {

		Set s = new HashSet ();
		String bestName = null;

		if (metricFile != null ) {
			String familyName = metricFile.getFamilyName ();
			if (familyName != null) {
				bestName = familyName;
				s.add (familyName); }}

		String n = (String) getValue (Type1Keys.FullName);
		if (n != null) {
			bestName = n;
			s.add (n); } 

		return new CatalogDescription ("T1", s, bestName, "");
	}

	public SWFFontDescription getSWFFontDescription(boolean wasEmbedded) throws UnsupportedFontException, InvalidFontException {
		return new T1SWFFont3Description(wasEmbedded);
	}

	public SWFFont4Description getSWFFont4Description(boolean wasEmbedded) throws UnsupportedFontException, InvalidFontException {
		return null;
	}
}


