/*
 *
 *	File: IntelligentResolver.java
 *
 * ****************************************************************************
 *
 *	ADOBE CONFIDENTIAL
 *	___________________
 *
 *	Copyright 2004-2005 Adobe Systems Incorporated
 *	All Rights Reserved.
 *
 *	NOTICE: All information contained herein is, and remains the property of
 *	Adobe Systems Incorporated and its suppliers, if any. The intellectual
 *	and technical concepts contained herein are proprietary to Adobe Systems
 *	Incorporated and its suppliers and may be covered by U.S. and Foreign
 *	Patents, patents in process, and are protected by trade secret or
 *	copyright law. Dissemination of this information or reproduction of this
 *	material is strictly forbidden unless prior written permission is obtained
 *	from Adobe Systems Incorporated.
 *
 */
package com.adobe.fontengine.fontmanagement;

import com.adobe.fontengine.font.FontImpl;
import com.adobe.fontengine.font.Font;
import com.adobe.fontengine.font.FontLoadingException;
import com.adobe.fontengine.font.UnsupportedFontException;
import com.adobe.fontengine.font.InvalidFontException;
import com.adobe.fontengine.fontmanagement.CacheSupportInfo;

/**
 * A namespace object that contains one method: choosePreferredFont.
 *
 * <h4>Synchronization</h4>
 * This class contains no mutable static or non-static data, so it can
 * be used safely from multiple threads. This module has been rewritten
 * (and reformatted) to use the new CacheSupportInfo helper class
 * that can cache the information it needs. That allows the resolver to
 * operate without actually parsing the fonts its operating on, assuming
 * the needed CacheSupportInfo objects already exist in the cache.
 */
final public class IntelligentResolver
{
	/**
	 * Given 2 fonts that are "the same", pick one to be preferred.
	 * @param f1 the first font
	 * @param f2 the second font
	 * @param preferFirst if the fonts are "identical" which one to pick
	 * @return The preferred of f1 or f2. Note that if f1 and f2 are indistinguishable,
	 * f1 is returned, which means that choosePreferredFont(f1, f2) may not equal
	 * choosePreferredFont(f2, f1)
	 */
	public static Font choosePreferredFont(Font f1, Font f2, boolean preferFirst)
		throws UnsupportedFontException, InvalidFontException, FontLoadingException
	{
		CacheSupportInfo f1Desc = ((FontImpl)f1).getCacheSupportInfo();
		CacheSupportInfo f2Desc = ((FontImpl)f2).getCacheSupportInfo();

		String f1FontType = f1Desc.getFontType();
		String f2FontType = f2Desc.getFontType();
		if (!f1FontType.equals(f2FontType)) {
			// 2 different font types. We should always prefer 1
			// over the other.
			if (f1FontType.equals("OpenTypeFont")) {
				return f1;
			}
			if (f2FontType.equals("OpenTypeFont")) {
				return f2;
			}
			if (f1FontType.equals("Type1Font")) {
				return f1;
			}
			return f2; //f2 must be type1 since f1 has to be naked cff
		}

		// both are opentype...is exactly one cff?
		if (f1FontType.equals("OpenTypeFont")) {
			if (f1Desc.isCFF() && !f2Desc.isCFF()) {
				return f1;
			}
			if (!f1Desc.isCFF() && f2Desc.isCFF()) {
				return f2;
			}
		}

		// more glyphs is preferred
		if (f1Desc.getNumGlyphs() > f2Desc.getNumGlyphs()) {
			return f1;
		}
		if (f1Desc.getNumGlyphs() < f2Desc.getNumGlyphs()) {
			return f2;
		}

		// URL Fonts are preferred over stream fonts
		if (f1.getClass() != f2.getClass()) {
			if (f1 instanceof URLFont) {
				return f1;
			}
			return f2;
		}

		// we can't tell them apart.
		return (preferFirst ? f1 : f2);
	}
}
