/*
 *
 *	File: StreamFont.java
 *
 * ****************************************************************************
 *
 *	ADOBE CONFIDENTIAL
 *	___________________
 *
 *	Copyright 2004-2005 Adobe Systems Incorporated
 *	All Rights Reserved.
 * 
 *	NOTICE: All information contained herein is, and remains the property of
 *	Adobe Systems Incorporated and its suppliers, if any. The intellectual
 *	and technical concepts contained herein are proprietary to Adobe Systems
 *	Incorporated and its suppliers and may be covered by U.S. and Foreign
 *	Patents, patents in process, and are protected by trade secret or
 *	copyright law. Dissemination of this information or reproduction of this
 *	material is strictly forbidden unless prior written permission is obtained
 *	from Adobe Systems Incorporated.
 *
 */

package com.adobe.fontengine.fontmanagement;

import java.io.IOException;
import java.io.NotSerializableException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.util.Map;

import com.adobe.fontengine.fontmanagement.FontLoader;
import com.adobe.fontengine.font.FontData;
import com.adobe.fontengine.font.FontImpl;
import com.adobe.fontengine.font.FontInputStream;
import com.adobe.fontengine.font.FontLoadingException;
import com.adobe.fontengine.font.UnsupportedFontException;
import com.adobe.fontengine.font.InvalidFontException;

/**
 * A Font that can get a FontData from an input stream.
 */
final class StreamFont extends FontImpl 
{
	static final long serialVersionUID = 1;

	private final FontInputStream stream;
	private final int length;
	private final boolean wasEmbedded;

	// This is not a SoftReference since the input stream is consumed
	// once the FontData is created.
	private FontData fontRef;


	/**
	 * Constructor.
	 * 
	 * The stream provided to the constructor is read only during the construction of
	 * font.  Once the constructor returns it is the responsibility of the caller to
	 * close the stream.
	 * 
	 * @param stream the source of the font data
	 * @param length the length of the font stream
	 * @param wasEmbedded is the font embedded
	 * @throws InvalidFontException
	 * @throws UnsupportedFontException
	 * @throws FontLoadingException
	 */
	StreamFont(FontInputStream stream, int length, boolean wasEmbedded) 
	throws InvalidFontException, UnsupportedFontException, FontLoadingException
	{
		this.stream = stream;
		this.length = length;
		this.fontRef = loadFont(stream, length);
		this.wasEmbedded = wasEmbedded;
	}

	private FontData loadFont(FontInputStream stream, int length) 
	throws InvalidFontException, UnsupportedFontException, FontLoadingException
	{
		FontData[] arr = null;
		try
		{
			arr = FontLoader.fromStream(stream, length);
		} catch (IOException e)
		{
			throw new FontLoadingException(stream.toString(), e);
		}
		if (arr == null || arr.length == 0)
		{
			throw new FontLoadingException("Stream did not contain a font.");
		}
		return arr[0];
	}

	public boolean wasEmbedded() 
	{
		return wasEmbedded;
	}

	protected FontData retrieveFontData()
	throws InvalidFontException, UnsupportedFontException, FontLoadingException
	{
		return fontRef;
	}

	public int hashCode()
	{
		return this.stream.hashCode() ^ this.length;
	}

	public boolean equals(Object obj)
	{
		if (obj != null && this == obj)
		{
			return true;
		}
		return false;
	}

	public String toString()
	{
		return this.stream.toString();
	}

	private void readObject(ObjectInputStream s) throws NotSerializableException
	{
		throw new NotSerializableException();
	}

	private void writeObject(ObjectOutputStream s) throws NotSerializableException
	{
		throw new NotSerializableException();
	}

	/* These are the cache support methods. They are noops for StreamFont
	 * because MemoryFonts are not cached.
	 */
	public String getCanonicalPath()
	{
		return null;
	}

	public long getLength()
	{
		return 0;
	}

	public long getLastModified()
	{
		return 0;
	}

	public Object getCachedFontDescription(String key)
	{
		return null;
	}

	public Map<String, Object> getCachedFontDescriptionMap()
	{
		return null;
	}

	public void setCachedFontDescription(String key, Object value)
	{
	}
}
