/**
 * 
 */
package com.adobe.fontengine.fontmanagement.platform;

import java.util.HashMap;
import java.util.Iterator;
import java.util.TreeSet;

import com.adobe.fontengine.font.Font;
import com.adobe.fontengine.font.FontLoadingException;
import com.adobe.fontengine.font.InvalidFontException;
import com.adobe.fontengine.font.UnsupportedFontException;
import com.adobe.fontengine.fontmanagement.FontResolutionPriority;
import com.adobe.fontengine.fontmanagement.IntelligentResolver;
import com.adobe.fontengine.inlineformatting.css20.FamilyNameNormalizer;
import com.adobe.fontengine.inlineformatting.css20.PassThroughFamilyNameNormalizer;

/**
 * @author sgill
 *
 */
public class PlatformFontResolverImpl implements PlatformFontResolver 
{
	static final long serialVersionUID = 2;

	private HashMap/*<SearchKey, Font>*/ platformFonts;
	private FontResolutionPriority resolutionPriority = FontResolutionPriority.FIRST;
	private FamilyNameNormalizer normalizer;

	public PlatformFontResolverImpl () 
	{
		this(new PassThroughFamilyNameNormalizer());
	}    

	public PlatformFontResolverImpl(FamilyNameNormalizer normalizer)
	{
		this.platformFonts = new HashMap/*<SearchKey,Font>*/();
		if (normalizer == null)
		{
			normalizer = new PassThroughFamilyNameNormalizer();
		}
		this.normalizer = normalizer;
	}

	public PlatformFontResolverImpl(PlatformFontResolverImpl original)
	{
		this.platformFonts = (HashMap/*<SearchKey,Font>*/) original.platformFonts.clone();
		this.resolutionPriority = original.resolutionPriority;
		this.normalizer = original.normalizer;
	}

	public void addFont(Font font) 
	throws UnsupportedFontException, InvalidFontException, FontLoadingException
	{
		PlatformFontDescription[] fontDesc = font.getPlatformFontDescription();
		if (fontDesc == null)
		{
			throw new UnsupportedFontException("No Platform Descriptions avalable for font - " + font);
		}
		for (int i = 0; i < fontDesc.length; i++) 
		{
			this.addFont(fontDesc[i], font); 
		}
	}

	public void addFont(PlatformFontDescription fontDesc, Font font)
	throws UnsupportedFontException, InvalidFontException, FontLoadingException 
	{
		Font oldFont = (Font) platformFonts.get(fontDesc.getPlatformName());
		if (oldFont != null) 
		{
			if (this.resolutionPriority == FontResolutionPriority.FIRST) 
			{
				// don't add as the old one has priority
				return; 
			}
			else if (this.resolutionPriority == FontResolutionPriority.INTELLIGENT_LAST
					|| this.resolutionPriority == FontResolutionPriority.INTELLIGENT_FIRST) 
			{
				Font preferredFont = IntelligentResolver.choosePreferredFont(oldFont, font,
						this.resolutionPriority == FontResolutionPriority.INTELLIGENT_FIRST);
				if (preferredFont == oldFont) 
				{
					return; 
				}
			}
		}
		platformFonts.put(this.normalizer.normalize(fontDesc.getPlatformName()), font);
	}

	public boolean isEmpty()
	{
		return platformFonts.isEmpty();
	}

	public FontResolutionPriority setResolutionPriority(FontResolutionPriority priority)
	{
		FontResolutionPriority oldPriority = this.resolutionPriority;

		this.resolutionPriority = priority;
		return oldPriority;
	}

	public int hashCode() 
	{
		final int prime = 31;
		int result = 1;
		result = prime * result	+ platformFonts.hashCode();
		result = prime * result	+ resolutionPriority.hashCode();
		return result;
	}

	public boolean equals(Object obj) 
	{
		if (this == obj)
		{
			return true;
		}
		if (obj == null)
		{
			return false;
		}
		if (!(obj instanceof PlatformFontResolverImpl))
		{
			return false;
		}
		final PlatformFontResolverImpl other = (PlatformFontResolverImpl) obj;
		if (!resolutionPriority.equals(other.resolutionPriority))
		{
			return false;
		}
		if (!platformFonts.equals(other.platformFonts))
		{
			return false;
		}
		return true;
	}

	public String toString()
	{
		TreeSet tempSet = new TreeSet();
		for (Iterator it = platformFonts.keySet().iterator(); it.hasNext(); ) {
			tempSet.add(it.next());
		}
		StringBuffer sb = new StringBuffer();
		sb.append("priority = ");
		sb.append(resolutionPriority.toString());
		sb.append("; platform font descriptions = ");
		String prefix = "";
		for (Iterator it = tempSet.iterator(); it.hasNext(); ) {
			sb.append(prefix);
			sb.append((String) it.next());
			prefix = ", ";
		}
		return sb.toString();
	}

	/* (non-Javadoc)
	 * @see com.adobe.fontengine.fontmanagement.platform.PlatformNameResolver#findFont(com.adobe.fontengine.fontmanagement.platform.PlatformFontDescription)
	 */
	public Font findFont(PlatformFontSearchAttributes searchAttributes) 
	{
		Font font = (Font) this.platformFonts.get(
				this.normalizer.normalize(searchAttributes.getPlatformName()));
		return font;
	}

}
