/*
 *
 *	File: BasicFormatter.java
 *
 *
 *	ADOBE CONFIDENTIAL
 *	___________________
 *
 *	Copyright 2004-2005 Adobe Systems Incorporated
 *	All Rights Reserved.
 *
 *	NOTICE: All information contained herein is, and remains the property of
 *	Adobe Systems Incorporated and its suppliers, if any. The intellectual
 *	and technical concepts contained herein are proprietary to Adobe Systems
 *	Incorporated and its suppliers and may be covered by U.S. and Foreign
 *	Patents, patents in process, and are protected by trade secret or
 *	copyright law. Dissemination of this information or reproduction of this
 *	material is strictly forbidden unless prior written permission is obtained
 *	from Adobe Systems Incorporated.
 *
 */

package com.adobe.fontengine.inlineformatting;

import com.adobe.fontengine.font.FontLoadingException;
import com.adobe.fontengine.font.InvalidFontException;
import com.adobe.fontengine.font.UnsupportedFontException;
import com.adobe.fontengine.inlineformatting.infontformatting.InFontFormatter;

/**
 * General purpose inline formatter.
 * 
 * <p>This formatter does not do any font selection or font fallback,
 * the client must have selected fonts.
 * 
 * <p>The client must first pass a run corresponding to a whole paragraph to the
 * {@link #preFormat} method. It can then pass fragments of the run to the
 * {@link #format(AttributedRun, int, int)} or {@link #format(AttributedRun, int, int, boolean)} method.
 * 
 */
final public class BasicFormatter {
  /**
   * Constructor.  BasicFormatter instances are created with a provided factory
   * method.
   * @see com.adobe.fontengine.inlineformatting.BasicFormatter#getFormatterInstance
   */
  private BasicFormatter () {
    // all instances created view getFormatterInstance ()
  }
  
  /**
   * Create a BasicFormatter for formatting text.
   * 
   * <h4>Concurrency</h4>
   *  
   * The BasicFormatter instance that is returned from this method is not
   * in general thread safe.
   * 
   * @return A BasicFormatter to be used for formatting text
   */
  static public BasicFormatter getFormatterInstance () {
    return new BasicFormatter ();
  }
  
  /**
   * Preformat an {@link AttributedRun}.
   * 
   * This method should be called on a portion of an AttributedRun 
   * corresponding to a whole paragraph, before the 
   * {@link #format(AttributedRun, int, int)} or {@link #format(AttributedRun, int, int, boolean)} method.
   * 
   * @param run an AttributedRun that contains the text to be formatted
   * @param start the index in the AttributedRun to start formatting from
   * @param limit the index in the AttributedRun at which formatting should cease (one 
   * more than the last position that formatting should done to)
   * @return The new limit of the original range in the AttributedRun after the formatting operation.
   */
  public int preFormat (AttributedRun run, int start, int limit) 
  throws InvalidFontException, UnsupportedFontException, FontLoadingException {
    limit = InFontFormatter.preFormat (run, start, limit);
    return limit;
  }
  
  /**
   * Format an {@link AttributedRun} by resolving the styling constraints
   * to actual fonts, glyphs and positions for those glyphs.
   * 
   * <p>This method should be called after the {@link #preFormat} method.
   * 
   * <p>
   * On input:
   * <ul>
   * <li>each element in the run can be either a character or a glyph, and
   * this is recorded by the {@link ElementAttribute#isGlyph} attribute.
   * 
   * <li>each element must have {@link ElementAttribute#locale} set
   * 
   * <li>each element must have {@link ElementAttribute#bidiLevel} set
   * 
   * <li>each glyph element must have {@link ElementAttribute#joiningType} set
   * 
   * <li>each element must have {@link ElementAttribute#font} set
   * 
   * <li>each element must have {@link ElementAttribute#pointSize} set
   * 
   * <li>each character element can have {@link ElementAttribute#fontStyle} set;
   * if not set, equivalent to {@link FontStyle#NORMAL}
   * 
   * <li>each character element can have {@link ElementAttribute#typographicCase} set;
   * if not set, equivalent to {@link TypographicCase#NONE}
   * 
   * <li>each character element can have {@link ElementAttribute#digitCase} set; 
   * if not set, equivalent to {@link DigitCase#DEFAULT}
   * 
   * <li>each character element can have {@link ElementAttribute#digitWidth} set;
   * if not set, equivalent to {@link DigitWidth#DEFAULT}
   * 
   * <li>each interelement can have {@link InterElementAttribute#ligatureLevel} set;
   * if not set, equivalent to {@link LigatureLevel#COMMON}
   * </ul>
   * 
   * <p>
   * At some point during formatting, 
   * {@link AttributedRun#startWorkingWithPositions startWorkingWithPositions}
   * will be called on the run.
   * 
   * <p>
   * On output:
   * <ul>
   * <li>each element in the run will be a glyph

   * <li>each element will have a placement and advance vector
   * </ul>
   * 
   * @param run an AttributedRun that contains the text to be formatted
   * @param start the index in the AttributedRun to start formatting from
   * @param limit the index in the AttributedRun at which formatting should cease (one 
   * more than the last position that formatting should done to)
   * @return The new limit of the original range in the AttributedRun after the formatting operation.
   * @throws InvalidFontException
   * @throws UnsupportedFontException
   * @throws FontLoadingException
   */
  public int format (AttributedRun run, int start, int limit) 
  throws InvalidFontException, UnsupportedFontException, FontLoadingException {
	  return format(run, start, limit, true /*kern*/);
  }
  
  /**
   * Format an {@link AttributedRun} by resolving the styling constraints
   * to actual fonts, glyphs and positions for those glyphs.
   * 
   * <p>This method should be called after the {@link #preFormat} method.
   * 
   * <p>
   * On input:
   * <ul>
   * <li>each element in the run can be either a character or a glyph, and
   * this is recorded by the {@link ElementAttribute#isGlyph} attribute.
   * 
   * <li>each element must have {@link ElementAttribute#locale} set
   * 
   * <li>each element must have {@link ElementAttribute#bidiLevel} set
   * 
   * <li>each glyph element must have {@link ElementAttribute#joiningType} set
   * 
   * <li>each element must have {@link ElementAttribute#font} set
   * 
   * <li>each element must have {@link ElementAttribute#pointSize} set
   * 
   * <li>each character element can have {@link ElementAttribute#fontStyle} set;
   * if not set, equivalent to {@link FontStyle#NORMAL}
   * 
   * <li>each character element can have {@link ElementAttribute#typographicCase} set;
   * if not set, equivalent to {@link TypographicCase#NONE}
   * 
   * <li>each character element can have {@link ElementAttribute#digitCase} set; 
   * if not set, equivalent to {@link DigitCase#DEFAULT}
   * 
   * <li>each character element can have {@link ElementAttribute#digitWidth} set;
   * if not set, equivalent to {@link DigitWidth#DEFAULT}
   * 
   * <li>each interelement can have {@link InterElementAttribute#ligatureLevel} set;
   * if not set, equivalent to {@link LigatureLevel#COMMON}
   * </ul>
   * 
   * <p>
   * At some point during formatting, 
   * {@link AttributedRun#startWorkingWithPositions startWorkingWithPositions}
   * will be called on the run.
   * 
   * <p>
   * On output:
   * <ul>
   * <li>each element in the run will be a glyph

   * <li>each element will have a placement and advance vector
   * </ul>
   * 
   * @param run an AttributedRun that contains the text to be formatted
   * @param start the index in the AttributedRun to start formatting from
   * @param limit the index in the AttributedRun at which formatting should cease (one 
   * more than the last position that formatting should done to)
   * @param shouldKern whether or not kerning should be applied to the glyphs
   * @return The new limit of the original range in the AttributedRun after the formatting operation.
   * @throws InvalidFontException
   * @throws UnsupportedFontException
   * @throws FontLoadingException
   */
  public int format (AttributedRun run, int start, int limit, boolean shouldKern) 
  throws InvalidFontException, UnsupportedFontException, FontLoadingException {
    limit = InFontFormatter.firstPass (run, start, limit);
    limit = InFontFormatter.format (run, start, limit, shouldKern);
    return limit;
  }
}
