/*
 *
 *	File: CSSPDF16FontDescription.java
 *
 * ****************************************************************************
 *
 *	ADOBE CONFIDENTIAL
 *	___________________
 *
 *	Copyright 2004-2005 Adobe Systems Incorporated
 *	All Rights Reserved.
 * 
 *	NOTICE: All information contained herein is, and remains the property of
 *	Adobe Systems Incorporated and its suppliers, if any. The intellectual
 *	and technical concepts contained herein are proprietary to Adobe Systems
 *	Incorporated and its suppliers and may be covered by U.S. and Foreign
 *	Patents, patents in process, and are protected by trade secret or
 *	copyright law. Dissemination of this information or reproduction of this
 *	material is strictly forbidden unless prior written permission is obtained
 *	from Adobe Systems Incorporated.
 *
 */

package com.adobe.fontengine.inlineformatting.css20;

import com.adobe.fontengine.font.FontDescription;
import com.adobe.fontengine.inlineformatting.css20.CSS20Attribute.CSSStretchValue;
import com.adobe.fontengine.inlineformatting.css20.CSS20Attribute.CSSStyleValue;
import com.adobe.fontengine.inlineformatting.css20.CSS20Attribute.CSSVariantValue;

/**
 * CSSPDF16FontDescription
 * This class provides a description of the font in a CSS2.0/PDF1.6 terms.
 */
final public class CSS20FontDescription extends FontDescription {
    static final long serialVersionUID = 1;
  
    private final String familyName;
    private final CSSStyleValue style;
    private final CSSVariantValue variant;
    private final CSSStretchValue stretch;
    private final int weight;
    private final double lowPointSize;
    private final double highPointSize;
    
    /**
     * Constructor.
     * @param familyName The family name of the font.
     * @param style The style of the font.
     * @param variant The variant of the font.
     * @param stretch The stretch value of the font.
     * @param weight The weight of the font.
     * @param lowPointSize The lowest point size that the font supports (inclusive).
     * @param highPointSize The highest point size that the font supports (exclusive).
     */
    public CSS20FontDescription(String familyName,
            CSSStyleValue style, CSSVariantValue variant, CSSStretchValue stretch, int weight,
            double lowPointSize, double highPointSize)
    {     	
        this.familyName = familyName;
        this.style = style;
        this.variant = variant;
        this.stretch = stretch;
        this.weight = weight;
        this.lowPointSize = lowPointSize;
        this.highPointSize = highPointSize;
    }

    /** 
     * Get the CSS font-family name.
     * @return the CSS font-family name.
     */
    public String getFamilyName ()
    {
        return familyName;
    }
    
    /** 
     * Get the CSS font style.
     * @return the CSS font style. 
     */
    public CSSStyleValue getStyle()
    {
        return this.style;
    }
    
    /** 
     * Get the CSS font variant.
     * @return the CSS font variant
     */
    public CSSVariantValue getVariant()
    {
        return this.variant;
    }
    
    /** 
     * Get the CSS font weight.
     * @return the CSS font weight
     */
    public int getWeight()
    {
        return this.weight;
    }
    
    /**
     * Get the CSS font stretch.
     * @return the CSS font stretch
     */
    public CSSStretchValue getStretch()
    {
        return this.stretch;
    }
    
    /**
     * Get the lowest point size (inclusive) supported by this font.
     * @return the low inclusive point size
     */
    public double getLowPointSize()
    {
        return this.lowPointSize;
    }
    
    /**
     * Get the highest point size (exclusive) supported by this font.
     * @return the high exclusive point size
     */
    public double getHighPointSize()
    {
        return this.highPointSize;
    }
    
    /* (non-Javadoc)
     * @see java.lang.Object#hashCode()
     */
    public int hashCode()
    {
        int hash = this.familyName.hashCode()
        		^ this.style.hashCode()
        		^ this.variant.hashCode()
        		^ this.stretch.hashCode()
        		^ this.weight
        		^ (int) (this.lowPointSize * 100)
        		^ (int) (this.highPointSize * 100);
        
        return hash;
    }
    
    /* (non-Javadoc)
     * @see java.lang.Object#equals(java.lang.Object)
     */
    public boolean equals(Object obj)
    {
        if (obj != null)
        {
            if (this == obj)
            {
                return true;
            }
            else if (obj instanceof CSS20FontDescription)
            {
                if (this.familyName.equals(((CSS20FontDescription)obj).familyName)
                     && this.style.equals(((CSS20FontDescription)obj).style)
                     && this.variant.equals(((CSS20FontDescription)obj).variant)
                     && this.stretch.equals(((CSS20FontDescription)obj).stretch)
                     && this.weight == ((CSS20FontDescription)obj).weight
                     && this.lowPointSize == ((CSS20FontDescription)obj).lowPointSize
                     && this.highPointSize == ((CSS20FontDescription)obj).highPointSize)
                {
                    return true;
                }
            }
        }
        return false;
    }
    
    /* (non-Javadoc)
     * @see java.lang.Object#toString()
     */
    public String toString()
    {
        StringBuffer description = new StringBuffer();
        
        description.append("{familyName = " + familyName + ", ");
        
        if (style == CSSStyleValue.NORMAL)
        {
            description.append("stl = NORMAL, ");
        } 
        else if (style == CSSStyleValue.ITALIC)
        {
            description.append("stl = ITALIC, ");
        }
        else if (style == CSSStyleValue.OBLIQUE)
        {
            description.append("stl = OBLIQUE, ");
        }

        if (variant == CSSVariantValue.NORMAL)
        {
            description.append("var = NORMAL, ");
        } 
        else if (variant == CSSVariantValue.SMALL_CAPS)
        {
            description.append("var = SMALL-CAPS, ");
        }

        description.append("wt = " + weight + ", ");
        
        description.append("str = " + stretch.toString() + ", ");
        description.append("lowPt = " + lowPointSize + ", ");
        description.append("highPt = " + highPointSize + "}");        

        return description.toString();
    }
}
