/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 *  Copyright 2012 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.adobe.granite.asset.api;

import org.osgi.annotation.versioning.ProviderType;

/**
 * <code>AssetManager</code> is a primary interface which provides methods to manage {@link Asset}s.
 * <p>
 * AssetManager is created by adapting {@link org.apache.sling.api.resource.ResourceResolver} to <code>AssetManager</code>
 * <pre>
 *     AssetManager assetManager = resolver.adaptTo(AssetManager.class);
 * </pre>
 *
 */
@ProviderType
public interface AssetManager {

    /**
     * Creates a new {@link Asset} at the given <code>path</code>.
     *
     * @param path Absolute path at which asset is to be created.
     *
     * @return The newly created asset
     *
     * @throws AssetException If error occurs while creating an Asset or if Asset already exists
     */
    Asset createAsset(String path);

    /**
     * Get {@link Asset} at the given <code>assetPath</code>.
     *
     * @param assetPath An absolute path of the Asset
     *
     * @return Asset if found otherwise null
     *
     * @throws AssetException If an error occurs trying to load the Asset object
     * */
    Asset getAsset(String assetPath);

    /**
     * Get {@link Asset} specified by the given <code>Identifier</code>.
     *
     * @param id Asset identifier
     *
     * @return Asset if found otherwise null
     *
     * @throws AssetException If an error occurs trying to load the Asset object
     * */
    Asset getAssetByIdentifier(String id);

    /**
     * Checks for the existence of an {@link Asset}.
     *
     * @param assetPath  Absolute path of the asset to be checked
     *
     * @return true if the Asset exist at the specified path
     * */
    boolean assetExists(String assetPath);

    /**
     * Remove {@link Asset} if exists.
     *
     * @param assetPath  Absolute path of the asset to be removed
     *
     * @throws AssetException If asset does not exist or cant be removed
     * */
    void removeAsset(String assetPath);

    /**
     * Copy Asset to the desired location.
     *
     * <pre>
     *     {@code
     *     Eg:
     *     //To copy Asset from /document/somedocument.pdf to /document/legal with name document.pdf
     *     AssetManager#copy("/document/somedocument.pdf", "/document/legal/document.pdf");
     *     }
     * </pre>
     *
     * @param assetPath Absolute path of the asset to be copied
     * @param destAbsPath Absolute path at with asset is to be copied
     *
     * @throws AssetException If an error occurs while copying
     * */
    void copyAsset(String assetPath, String destAbsPath);

    /**
     * Move Asset to the desired location.
     *
     * <pre>
     *     {@code
     *     Eg:
     *     // To move Asset from /document/somedocument.pdf to /document/legal with name document.pdf
     *     AssetManager#move("/document/somedocument.pdf", "/document/legal/document.pdf");
     *     }
     * </pre>
     *
     * @param assetPath Absolute path of the asset to be moved
     * @param destAbsPath Absolute path at with asset is to be moved
     *
     * @throws AssetException If an error occurs while moving
     * */
    void moveAsset(String assetPath, String destAbsPath);
}
