/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 *  Copyright 2012 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.adobe.granite.asset.api;

import java.io.InputStream;
import java.util.Map;

import org.apache.sling.api.resource.Resource;
import org.osgi.annotation.versioning.ConsumerType;

/**
 * The <code>RenditionHandler</code> interface defines the API for rendition handlers. Each RenditionHandler
 * must be identifiable via a unique ID 'rendition.handler.id'.
 * <p>
 * RenditionHandler is used by the Asset API to delegate Access, Creation &amp; Deletion of {@link Rendition}.
 * <p>
 * Implementations of this interface must be registered as OSGi services.
 * </p>
 */
@ConsumerType
public interface RenditionHandler {

    /**
     * PROPERTY_ID must be set by the implementation class as an OSGi service property
     * This is used by the Asset API to identify RenditionHandler
     * */
    public static final String PROPERTY_ID = "rendition.handler.id";

    /**
     * This property constant defines a property key for the mime type. This can be used by the
     * <code>RenditionHandler</code> or the API clients to define the mime type of the Rendition
     * <blockquote><pre>
     *    For example, if the rendition handler requires mime type:
     *      final Map&lt;String, Object&gt; map = new HashMap&lt;String, Object&gt;();
     *      map.put(RenditionHandler.PROPERTY_RENDITION_MIME_TYPE, "image/gif");
     *      ....
     *      asset.setRendition("image.gif", InputStream, map);
     *
     * </pre></blockquote>
     * */
    public static final String PROPERTY_RENDITION_MIME_TYPE = "rendition.mime";

    /**
     * Get Rendition. Implementations are expected to type the given {@link Resource} to Rendition.
     *
     * @param resource Rendition resource
     *
     * @return Rendition object of the given resource
     * */
    Rendition getRendition(Resource resource);

    /**
     * Set <code>Rendition</code>. Implementations are free to define Resource data structure.
     *
     * @param resource Rendition resource
     * @param is InputStream to create the rendition binary or null
     * @param map containing values needed by this handler.
     *
     * @return newly created or updated rendition
     *
     * @throws AssetException if Rendition cannot be set
     * */
    Rendition setRendition(Resource resource, InputStream is, Map<String,Object> map);


    /**
     * Delete the rendition resource. Implementations are free to decide the semantics of deletion.
     *
     * @param resource To be deleted
     *
     * @throws AssetException if Rendition cannot be deleted
     * */
    void deleteRendition(Resource resource);
}
