/*************************************************************************
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 * Copyright 2021 Adobe
 * All Rights Reserved.
 *
 * NOTICE: All information contained herein is, and remains
 * the property of Adobe and its suppliers, if any. The intellectual
 * and technical concepts contained herein are proprietary to Adobe
 * and its suppliers and are protected by all applicable intellectual
 * property laws, including trade secret and copyright laws.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe.
 **************************************************************************/
package com.adobe.granite.auth.saml.spi;

import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import javax.jcr.Credentials;

import org.apache.jackrabbit.oak.spi.security.authentication.external.ExternalIdentityRef;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

public class SamlCredentials implements Credentials {

    private static final long serialVersionUID = -2737784478463243439L;

    protected final Map<String,Object> attributes = new HashMap<>();
    protected final String userId;

    private final String idp;

    /**
     * holds the identifiers of the SAML groups which the user must belong to.
     */
    private Set<String> samlGroupIds = new HashSet<>();

    public SamlCredentials(String userId, String idp, boolean idpNameInUserId) {
        if (idpNameInUserId) {
            this.userId = new ExternalIdentityRef(userId, idp).getString();
        } else {
            this.userId = userId;
        }
        this.idp = idp;
    }

    public @NotNull String getIdp() {
        return idp;
    }

    public @NotNull Set<String> getSamlGroupIds() {
        return samlGroupIds;
    }

    public void setSamlGroups(@NotNull Set<String> samlGroupIds) {
        this.samlGroupIds = samlGroupIds;
    }

    // AbstractCredentials code because backport to 1.6 was a nightmare

    /**
     * Returns the userId.
     *
     * @return the userId.
     */
    @NotNull
    public String getUserId() {
        return userId;
    }

    /**
     * Stores an attribute in this credentials instance. If the specified
     * {@code value} is {@code null} the attribute will be removed.
     *
     * @param name
     *            a {@code String} specifying the name of the attribute
     * @param value
     *            the {@code Object} to be stored
     */
    public void setAttribute(@NotNull String name, @Nullable Object value) {
        // name cannot be null
        if (name == null) {
            throw new IllegalArgumentException("name cannot be null");
        }

        // null value is the same as removeAttribute()
        if (value == null) {
            removeAttribute(name);
            return;
        }

        synchronized (attributes) {
            attributes.put(name, value);
        }
    }

    /**
     * Returns the value of the named attribute as an <code>Object</code>, or
     * <code>null</code> if no attribute of the given name exists.
     *
     * @param name
     *            a <code>String</code> specifying the name of the attribute
     * @return an <code>Object</code> containing the value of the attribute, or
     *         <code>null</code> if the attribute does not exist
     */
    @Nullable
    public Object getAttribute(@NotNull String name) {
        synchronized (attributes) {
            return (attributes.get(name));
        }
    }

    /**
     * Removes an attribute from this credentials instance.
     *
     * @param name
     *            a <code>String</code> specifying the name of the attribute to
     *            remove
     */
    public void removeAttribute(@NotNull String name) {
        synchronized (attributes) {
            attributes.remove(name);
        }
    }

    /**
     * @return an immutable map containing the attributes available to this credentials instance
     */
    @NotNull
    public Map<String,Object> getAttributes() {
        return Collections.unmodifiableMap(attributes);
    }

    public String getIdWithIdp(String id) {
        return new ExternalIdentityRef(id, idp).getString();
    }

}
