/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2013 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.adobe.granite.comments;

import org.apache.sling.api.resource.Resource;
import org.osgi.annotation.versioning.ProviderType;

/**
 * The <code>CommentManager</code> provides facilities to manage (create, get or remove) {@link CommentCollection}s.
 * This manager is registered as an OSGi-service and can be obtained via this interface class from the service registry.
 * The manager produces typed collections. The type-specific collections are produced by {@link CommentingProvider}s.
 * The manager binds available factories and uses them to instantiate collections based on the given collection type.
 */
@ProviderType
public interface CommentManager {

    /**
     * Retrieve a {@link CommentCollection} for the given <code>target</code>. The collection is instantiated using the
     * given <code>collectionType</code>, as provided by the type-specific {@link CommentingProvider}.
     *
     * @param target         The {@link Resource} representing the target for which to retrieve the comment collection.
     * @param collectionType The desired type of the collection.
     * @param <C>            A collection type, extending {@link CommentCollection}.
     *
     * @return The collection, or <code>null</code> if no collection of the requested type is present.
     *
     * @throws CommentException         If no {@link CommentingProvider} could be found for the given
     *                                  <code>collectionType</code>.
     * @throws IllegalArgumentException If either target or collectionType is null.
     */
    <C extends CommentCollection> C getCollection(Resource target, Class<C> collectionType);

    /**
     * Retrieve or - if none exists - create a {@link CommentCollection} for the given <code>target</code>. The
     * collection is instantiated using the given <code>collectionType</code>, as provided by the type-specific {@link
     * CommentingProvider}.
     *
     * @param target         The {@link Resource} representing the target for which to create or retrieve the comment
     *                       collection.
     * @param collectionType The desired type of the collection.
     * @param <C>            A collection type, extending {@link CommentCollection}.
     *
     * @return The collection.
     *
     * @throws CommentException         Upon encountering an error during creation of a collection, or if the collection
     *                                  already exists, or if no {@link CommentingProvider} could be found for the given
     *                                  <code>collectionType</code>.
     * @throws IllegalArgumentException If either target or collectionType is null.
     */
    <C extends CommentCollection> C getOrCreateCollection(Resource target, Class<C> collectionType);

    /**
     * Create a {@link CommentCollection} for the given <code>target</code>. The collection is instantiated using the
     * given <code>collectionType</code>, as provided by the type-specific {@link CommentingProvider}.
     *
     * @param target         The {@link Resource} representing the target for which to create the comment collection.
     * @param collectionType The desired type of the collection.
     * @param <C>            A collection type, extending {@link CommentCollection}.
     *
     * @return The newly created collection.
     *
     * @throws CommentException         Upon encountering an error during creation of a collection, or if the collection
     *                                  already exists, or if no {@link CommentingProvider} could be found for the given
     *                                  <code>collectionType</code>.
     * @throws IllegalArgumentException If either target or collectionType is null.
     */
    <C extends CommentCollection> C createCollection(Resource target, Class<C> collectionType);
}
