/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2013 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.adobe.granite.comments;

import org.apache.sling.api.adapter.AdapterFactory;
import org.apache.sling.api.resource.Resource;
import org.osgi.annotation.versioning.ConsumerType;

/**
 * A <code>CommentingProvider</code> represents a "configuration" used by the {@link CommentManager} service to create
 * and store {@link Comment}s and {@link CommentCollection}s. In this way, multiple comment collections for the same
 * target can be supported (e.g. target-located author-comments/annotations vs. user-generated comments stored in
 * /content/usergenerated). A provider operates on a target-specific resource. Providers must register themselves as
 * OSGi-components and service on this interface in order for them to be automatically picked up by the {@link
 * CommentManager}. A provider also must announce the types of comments and collections it supports via the mandatory
 * OSGi service properties {@link #PROPERTY_COMMENT_TYPES} and {@link #PROPERTY_COLLECTION_TYPES}. A provider announces
 * the supported types like this:
 * <pre>
 *     \@Properties({
 * \@Property(name = CommentingProvider.PROPERTY_COMMENT_TYPES, classValue = MyComment.class),
 * \@Property(name = CommentingProvider.PROPERTY_COLLECTION_TYPES, classValue = MyCommentCollection.class)
 * })
 * </pre>
 * Implementations should implement {@link AdapterFactory} and provide the following adaptations for their respectively
 * supported comment and collection types: <ul> <li>Adapt a target resource to any of the supported comment
 * collections</li> <li>Adapt an actual comment collection resource to a supported comment collection</li> <li>Adapt an
 * actual comment resource to a supported comment</li> </ul> For example: <ul> <li>Target (e.g. /content/geometrixx/en):
 * resource.adaptTo(CommentCollection.class)</li> <li>Collection Resource (e.g. /content/geometrixx/en/jcr:content/comments):
 * resource.adaptTo(CommentCollection.class)</li> <li>Comment Resource (e.g. /content/geometrixx/en/jcr:content/comments/my_comment):
 * resource.adaptTo(Comment.class)</li> </ul>
 */
@ConsumerType
public interface CommentingProvider {

    /**
     * The name of the OSGi service property that holds an array of classes this provider supports for collections.
     */
    public static final String PROPERTY_COLLECTION_TYPES = "provider.collectionTypes";

    /**
     * The name of the OSGi service property that holds an array of classes this provider supports for comments.
     */
    public static final String PROPERTY_COMMENT_TYPES = "provider.commentTypes";

    /**
     * Retrieves a {@link CommentCollection} of the given <code>collectionType</code> for the given
     * <code>target</code>.
     *
     * @param target         The target the collection belongs to.
     * @param collectionType The type of the collection.
     * @param <C>            A collection type, extending {@link CommentCollection}.
     *
     * @return An existing collection object of the given type, or <code>null</code> if no collection exists.
     *
     * @throws CommentException If target does not have collection of given <code>collectionType</code>.
     */
    <C extends CommentCollection> C getCollection(Resource target, Class<C> collectionType);

    /**
     * Instantiates a new {@link CommentCollection} of the given <code>collectionType</code> for the given
     * <code>target</code>.
     *
     * @param target         The target the collection belongs to.
     * @param collectionType The type of the collection.
     * @param <C>            A collection type, extending {@link CommentCollection}.
     *
     * @return A new collection object of the given type.
     *
     * @throws CommentException If a collection already exists for on the target-specific root resource, or upon
     *                          encountering an error persisting.
     */
    <C extends CommentCollection> C createCollection(Resource target, Class<C> collectionType);
}
