/*************************************************************************
* ADOBE CONFIDENTIAL
* __________________
*
* Copyright 2011 Adobe
* All Rights Reserved.
*
* NOTICE: All information contained herein is, and remains
* the property of Adobe and its suppliers, if any. The intellectual
* and technical concepts contained herein are proprietary to Adobe
* and its suppliers and are protected by all applicable intellectual
* property laws, including trade secret and copyright laws.
* Dissemination of this information or reproduction of this material
* is strictly forbidden unless prior written permission is obtained
* from Adobe.
**************************************************************************/
package com.adobe.granite.jmx.annotation;

import java.beans.IntrospectionException;
import java.beans.Introspector;
import java.beans.PropertyDescriptor;
import java.lang.annotation.Annotation;
import java.math.BigDecimal;
import java.math.BigInteger;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.management.ObjectName;
import javax.management.openmbean.ArrayType;
import javax.management.openmbean.CompositeData;
import javax.management.openmbean.CompositeType;
import javax.management.openmbean.OpenDataException;
import javax.management.openmbean.OpenType;
import javax.management.openmbean.SimpleType;
import javax.management.openmbean.TabularData;
import javax.management.openmbean.TabularType;

public abstract class OpenTypeUtils {

    /**
     * Introspects the given class to generate open type type. If the class is
     * {@link CompositeData}, {@link TabularData} or their respective array then
     * {@link OpenTypeInfo} param must not be null.
     * @param clazz the class to introspect
     * @param info the open type info
     * @return the generated open type type
     * @throws javax.management.openmbean.OpenDataException if type is null
     */
    public static OpenType createOpenType(final Class<?> clazz,
            final OpenTypeInfo info) throws OpenDataException {
        if (clazz.isArray()) {
            return new ArrayType(getArrayDim(clazz),
                createOpenType(getArrayType(clazz), info));
        }

        if (CompositeData.class.isAssignableFrom(clazz)) {
            return createCompositeType(info.value());
        }

        if (TabularData.class.isAssignableFrom(clazz)) {
            return createTabularType(info.value());
        }

        SimpleType type = getSimpleType(clazz);

        if (type == null)
            throw new OpenDataException("Illegal class for open type: " + clazz);

        return type;
    }

    private static final Map<Class<?>, SimpleType> SIMPLE_TYPES =
            new HashMap<Class<?>, SimpleType>();

    static {
        SIMPLE_TYPES.put(BigDecimal.class, SimpleType.BIGDECIMAL);
        SIMPLE_TYPES.put(BigInteger.class, SimpleType.BIGINTEGER);
        SIMPLE_TYPES.put(Boolean.class, SimpleType.BOOLEAN);
        SIMPLE_TYPES.put(Boolean.TYPE, SimpleType.BOOLEAN);
        SIMPLE_TYPES.put(Byte.class, SimpleType.BYTE);
        SIMPLE_TYPES.put(Byte.TYPE, SimpleType.BYTE);
        SIMPLE_TYPES.put(Character.class, SimpleType.CHARACTER);
        SIMPLE_TYPES.put(Character.TYPE, SimpleType.CHARACTER);
        SIMPLE_TYPES.put(Date.class, SimpleType.DATE);
        SIMPLE_TYPES.put(Double.class, SimpleType.DOUBLE);
        SIMPLE_TYPES.put(Double.TYPE, SimpleType.DOUBLE);
        SIMPLE_TYPES.put(Float.class, SimpleType.FLOAT);
        SIMPLE_TYPES.put(Float.TYPE, SimpleType.FLOAT);
        SIMPLE_TYPES.put(Integer.class, SimpleType.INTEGER);
        SIMPLE_TYPES.put(Integer.TYPE, SimpleType.INTEGER);
        SIMPLE_TYPES.put(Long.class, SimpleType.LONG);
        SIMPLE_TYPES.put(Long.TYPE, SimpleType.LONG);
        SIMPLE_TYPES.put(ObjectName.class, SimpleType.OBJECTNAME);
        SIMPLE_TYPES.put(Short.class, SimpleType.SHORT);
        SIMPLE_TYPES.put(Short.TYPE, SimpleType.SHORT);
        SIMPLE_TYPES.put(String.class, SimpleType.STRING);
        SIMPLE_TYPES.put(Void.class, SimpleType.VOID);
        SIMPLE_TYPES.put(Void.TYPE, SimpleType.VOID);
    }

    /**
     * Returns the simple type for the given class. If the class is not a simple
     * type then null is returned.
     * @param clazz the class
     * @return the simple type for the given class
     */
    public static SimpleType getSimpleType(final Class<?> clazz) {
        return SIMPLE_TYPES.get(clazz);
    }

    /**
     * Introspects the given class to generate composite type.
     * @param clazz the class
     * @return the composite type of the given class
     * @throws javax.management.openmbean.OpenDataException when an error happens
     * during inspection
     */
    public static CompositeType createCompositeType(final Class<?> clazz)
            throws OpenDataException {
        try {
            PropertyDescriptor[] properties = Introspector.getBeanInfo(clazz).getPropertyDescriptors();
            if (properties == null)
                properties = new PropertyDescriptor[0];

            List<String> itemNames = new ArrayList<String>();
            List<OpenType> itemTypes = new ArrayList<OpenType>();

            List<String> allowedClasses = Arrays.asList(OpenType.ALLOWED_CLASSNAMES);

            for (int i = 0; i < properties.length; i++) {
                PropertyDescriptor p = properties[i];

                if (!allowedClasses.contains(p.getPropertyType().getName()))
                    continue;

                OpenTypeInfo childInfo = getAnnotation(p, OpenTypeInfo.class);
                OpenType openType = createOpenType(p.getPropertyType(),
                    childInfo);

                itemNames.add(p.getName());
                itemTypes.add(openType);
            }

            Description description = clazz.getAnnotation(Description.class);
            String desc = description == null
                    ? clazz.getName()
                    : description.value();

            String[] names = itemNames.toArray(new String[itemNames.size()]);

            return new CompositeType(clazz.getName(), desc, names, names,
                itemTypes.toArray(new OpenType[itemTypes.size()]));
        } catch (IntrospectionException e) {
            throw new OpenDataException(e.toString());
        }
    }

    /**
     * Introspects the given class to generate tabular type. The class must be
     * annotated with {@link TabularTypeInfo}.
     * @param clazz the class
     * @return the tabular type of the given class
     * @throws javax.management.openmbean.OpenDataException when an error happens
     */
    public static TabularType createTabularType(final Class<?> clazz)
            throws OpenDataException {
        Description description = clazz.getAnnotation(Description.class);
        String desc = description == null
                ? clazz.getName()
                : description.value();

        TabularTypeInfo rowTypeInfo = clazz.getAnnotation(TabularTypeInfo.class);
        CompositeType rowType = createCompositeType(rowTypeInfo.rowType());

        return new TabularType(clazz.getName(), desc, rowType,
            rowTypeInfo.indexNames());
    }

    private static <T extends Annotation> T getAnnotation(PropertyDescriptor p,
            Class<T> a) {
        if (p.getReadMethod() != null) {
            T result = p.getReadMethod().getAnnotation(a);

            if (result != null) {
                return result;
            }

            if (p.getWriteMethod() == null) {
                return null;
            }
        }

        return p.getWriteMethod().getAnnotation(a);
    }

    private static final Map<String, Class<?>> PRIMITIVES = new HashMap<String, Class<?>>();
    static {
        PRIMITIVES.put("boolean", boolean.class);
        PRIMITIVES.put("byte", byte.class);
        PRIMITIVES.put("short", short.class);
        PRIMITIVES.put("int", int.class);
        PRIMITIVES.put("long", long.class);
        PRIMITIVES.put("float", float.class);
        PRIMITIVES.put("double", double.class);
    }

    static Class<?> loadClass(String type, ClassLoader loader)
            throws ClassNotFoundException {
        if (PRIMITIVES.containsKey(type)) {
            return PRIMITIVES.get(type);
        }

        return loader.loadClass(type);
    }

    static Class<?> getArrayType(Class<?> clazz) {
        return clazz.isArray() ? getArrayType(clazz.getComponentType()) : clazz;
    }

    static int getArrayDim(Class<?> clazz) {
        return clazz.isArray() ? 1 + getArrayDim(clazz.getComponentType()) : 0;
    }
}
