/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2013 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.adobe.granite.maintenance;

import java.text.ParseException;
import java.util.Calendar;
import java.util.Date;

import org.apache.sling.api.resource.Resource;
import org.quartz.CronExpression;

import com.adobe.granite.maintenance.impl.MaintenanceWindowImpl;

/**
 * Utility class for maintenance related stuff.
 */
public abstract class MaintenanceUtil {

    /**
     * Check if the start and the end time are valid time definitions ([hour]:[minute]).
     * This method does not check whether startTime is lower than endTime.
     */
    public static boolean isValidTimeInterval(final String startTime, final String endTime) {
        final int[] start = parseTime(startTime);
        final int[] end = parseTime(endTime);
        if ( start == null || end == null ) {
            return false;
        }
        return true;
    }

    /**
     * Check if the given time string is valid ([hour]:[minute])
     * @return a two value array with hour and minute if its valid, <code>null</code> otherwise.
     */
    public static int[] parseTime(final String value) throws IllegalArgumentException {
        if ( value != null && value.trim().length() > 0 ) {
            final int pos = value.indexOf(":");
            if ( pos != -1 ) {
                try {
                    final int hour = Integer.valueOf(value.substring(0, pos));
                    final int minute = Integer.valueOf(value.substring(pos + 1));
                    if ( hour >= 0 && hour <= 23 && minute >= 0 && minute <= 59 ) {
                        return new int[] {hour, minute};
                    }
                } catch ( final NumberFormatException nfe ) {
                    // ignore
                }
            }
        }
        return null;
    }

    public static String getCronExpression(final int[] timeDef) {
        if ( timeDef != null ) {
            return  "0 " + String.valueOf(timeDef[0]) + " " + String.valueOf(timeDef[1]) + " * * ?";
        }
        return null;
    }

    public static String getCronExpression(final String time) {
        return getCronExpression(parseTime(time));
    }

    /**
     * Return the next execution time for this window.
     * If the current time is currently within the window, null will be returned.
     * If the time definitions are invalid, null is returned as well.
     * @param startTime
     * @param endTime
     * @return The next execution time, if the window is currently open
     */
    public static Calendar getNextExecutionTime(final String startTime, final String endTime) {
        final String startCron = getCronExpression(parseTime(startTime));
        final String endCron = getCronExpression(parseTime(endTime));
        if ( startCron == null || endCron == null ) {
            return null;
        }
        try {
            final Date now = new Date();
            final CronExpression startExpr = new CronExpression(startCron);
            final CronExpression endExpr = new CronExpression(endCron);
            final Date nextStart = startExpr.getNextValidTimeAfter(now);
            final Date nextEnd = endExpr.getNextValidTimeAfter(now);
            if ( nextEnd.getTime() < nextStart.getTime() ) {
                return null;
            }
            final Calendar nextExecution = Calendar.getInstance();
            nextExecution.setTime(nextStart);
            return nextExecution;
        } catch ( final ParseException pe ) {
            // Sanity check
            return null;
        }
    }

    /**
     * Return the next execution time for this maintenance task
     * If the time definitions are invalid, null is returned.
     * @return The next execution time, if the window is valid
     */
    public static Calendar getNextExecutionTime(final MaintenanceTaskInfo.TaskSchedule schedule,
                                                final String startTime) {
        final String startCron = getCronExpression(parseTime(startTime));
        if ( startCron == null) {
            return null;
        }
        try {
            final Date now = new Date();
            final CronExpression startExpr = new CronExpression(startCron);
            final Date nextStart = startExpr.getNextValidTimeAfter(now);
            final Calendar nextExecution = Calendar.getInstance();
            nextExecution.setTime(nextStart);

            switch ( schedule ) {
                case WEEKLY : final int weekDay = nextExecution.get(Calendar.DAY_OF_WEEK);
                              if ( weekDay > 1 ) {
                                  nextExecution.add(Calendar.DAY_OF_WEEK, 8 - weekDay);
                              }
                              break;
                case BIWEEKLY : final int weekDay2 = nextExecution.get(Calendar.DAY_OF_WEEK);
                                if ( weekDay2 > 1 ) {
                                    nextExecution.add(Calendar.DAY_OF_WEEK, 8 - weekDay2);
                                }
                                final int week = nextExecution.getActualMaximum(Calendar.WEEK_OF_YEAR);
                                if ( week % 2 == 0 ) {
                                    nextExecution.add(Calendar.WEEK_OF_MONTH, 1);
                                }
                                break;
                case MONTHLY : int dayOfMonth = nextExecution.get(Calendar.DAY_OF_MONTH);
                               while ( dayOfMonth != 1 ) {
                                   nextExecution.add(Calendar.DAY_OF_MONTH, 1);
                                   dayOfMonth = nextExecution.get(Calendar.DAY_OF_MONTH);
                               }
                               break;
                default: // nothing to do
            }
            return nextExecution;
        } catch ( final ParseException pe ) {
            // Sanity check
            return null;
        }
    }

    /**
     * Return the next execution time for this window.
     * If the current time is currently within the window, null will be returned.
     * If the time definitions are invalid, null is returned as well.
     * @param windowResource The window resource.
     * @return The next execution time, if the window is currently open
     * @since 1.1
     */
    public static Calendar getNextWindowExecutionTime(final Resource windowResource) {
        try {
            final Date now = new Date();

            final MaintenanceWindowImpl window = new MaintenanceWindowImpl(windowResource);

            final CronExpression startExpr = new CronExpression(window.getStartCronExpression());
            final CronExpression endExpr = new CronExpression(window.getEndCronExpression());

            final Date nextStart = startExpr.getNextValidTimeAfter(now);
            final Date nextEnd = endExpr.getNextValidTimeAfter(nextStart);

            if ( nextEnd.before(nextStart) ) {
                return null;
            }
            final Calendar nextExecution = Calendar.getInstance();
            nextExecution.setTime(nextStart);
            return nextExecution;
        } catch ( final ParseException pe ) {
            // Sanity check
            return null;
        } catch ( final IllegalArgumentException iae ) {
            return null;
        }
    }
}
