/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2013 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.adobe.granite.security.user.util;

import java.util.regex.Pattern;

import javax.jcr.RepositoryException;

import org.apache.commons.collections.Predicate;
import org.apache.commons.lang3.StringUtils;
import org.apache.jackrabbit.api.security.user.Authorizable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Predicate implementation that returns true if the given object is an {@link Authorizable} that has an ID or a given name that matches a
 * certain filter.
 */
public class AuthorizableFilterPredicate implements Predicate {
    private final static Logger LOG = LoggerFactory.getLogger(AuthorizableFilterPredicate.class);
    private String filter;
    private Pattern filterPattern;

    /**
     * Creates an {@link AuthorizableFilterPredicate} object, using the given filter if it's not blank.
     *
     * @param filter to use for matching against an authorizable ID and given name
     */
    public AuthorizableFilterPredicate(final String filter) {
        if (!StringUtils.isBlank(filter)) {
            this.filter = filter;
            this.filterPattern = generateFilterPattern();
        }
    }

    /**
     * @return the filter to use for matching against an authorizable ID and given name
     */
    public String getFilter() {
        return filter;
    }

    public boolean evaluate(Object object) {
        if (filter == null) {
            return true;
        }

        if (object instanceof Authorizable) {
            Authorizable authorizable = (Authorizable) object;
            try {
                String authorizableID = authorizable.getID();
                String authorizableName = AuthorizableUtil.getName(authorizable);
                boolean isMatching = false;
                if (authorizableID != null) {
                    isMatching = filterPattern.matcher(authorizableID).matches();
                }
                if (authorizableName != null) {
                    isMatching = isMatching || filterPattern.matcher(authorizableName).matches();
                }
                return isMatching;
            } catch (RepositoryException e) {
                LOG.error("Error while accessing authorizable" + authorizable + " properties:", e);
            }
        }
        return false;
    }

    private Pattern generateFilterPattern() {
        //remove multiple % consecutive occurrences
        String filterRegex = filter.replaceAll("%+", "%");
        String[] tokens = filterRegex.split("%");
        StringBuilder sanitizedRegex = new StringBuilder();
        for (String token : tokens) {
            if (!token.isEmpty()) {
                //treat the tokens as literals in order to avoid metacharacters or escape sequences from users
                sanitizedRegex.append(Pattern.quote(token));
            }
            sanitizedRegex.append(".*");
        }
        if (!filterRegex.endsWith("%")) {
            int length = sanitizedRegex.length();
            sanitizedRegex.delete(length - 2, length);
        }
        return Pattern.compile(sanitizedRegex.toString());
    }

}
