/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2012 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.adobe.granite.socialgraph;

import java.util.Map;

import org.apache.sling.api.resource.Resource;
import org.osgi.annotation.versioning.ProviderType;

/**
 * The <code>GraphNode</code> represents a user in the social graph.
 */
@ProviderType
public interface GraphNode extends Map<String, Object> {

    /**
     * Returns the id of this node.
     * @return the id.
     */
    String getId();

    /**
     * Returns the resource of object addressed by id or <code>null</code> if it does not exist.
     * @return the resource.
     */
    Resource getResource();

    /**
     * Specifies tha this node is virtual and is not (and will never be) backed by a persistence object.
     * @return <code>true</code> if this node is virtual.
     *
     * @deprecated virtual nodes are no longer supported.
     */
    @Deprecated
    boolean isVirtual();

    /**
     * Creates a relationship to the given node.
     * @param other the other node
     * @param type the type of the relationship
     * @return the relationship
     * @throws IllegalArgumentException if the relationship already exists
     * @throws SocialGraphException if an error during this operation occurs.
     */
    Relationship createRelationshipTo(GraphNode other, String type);

    /**
     * Deletes this graph node and all outgoing relationships.
     * TODO: specify what should happen to incoming relationships
     */
    void delete();

    /**
     * Returns the relationships attached to this node of the specific type and direction.
     * @param dir the direction
     * @param types the types of relationships to return.
     * @return a collection of relationships
     */
    Iterable<Relationship> getRelationships(Direction dir, String ... types);

    /**
     * Returns the relationship between this and the give node in the specified direction and type.
     *
     * Example:
     *
     * if A --[follows]--&gt; B, then
     * B.getRelationShip(Direction.OUTGOING, A, "follows") and,
     * A.getRelationShip(Direction.INCOMING, B, "follows")
     *
     * Both return the same relationship defined above.
     *
     * @param dir direction of the relationship to return
     * @param node the other node
     * @param type the type of relationship
     * @return the desired relationship or <code>null</code>
     */
    Relationship getRelationship(Direction dir, GraphNode node, String type);

    /**
     * Get a named property and convert it into the given type.
     *
     * @param name The name of the property
     * @param type The class of the type
     * @param <T> the type
     * @return Return named value converted to type T or <code>null</code> if
     *         non existing or can't be converted.
     */
    <T> T get(String name, Class<T> type);

    /**
     * Get a named property and convert it into the given type.
     *
     * @param name The name of the property
     * @param defaultValue The default value to use if the named property does
     *            not exist or cannot be converted to the requested type. The
     *            default value is also used to define the type to convert the
     *            value to. If this is <code>null</code> any existing property is
     *            not converted.
     * @param <T> the type
     * @return Return named value converted to type T or the default value if
     *         non existing or can't be converted.
     */
    <T> T get(String name, T defaultValue);

}