/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2012 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.adobe.granite.socialgraph;

import java.util.Map;

import org.osgi.annotation.versioning.ProviderType;

/**
 * <code>Relationship</code> specifies an edge in the social graph and represents
 * the relationship between two graph nodes (users, groups, topics).
 */
@ProviderType
public interface Relationship extends Map<String, Object> {

    /**
     * Returns the starting node of this relationship
     * @return the starting node of this relationship
     */
    GraphNode getStartNode();

    /**
     * Returns the ending node of this relationship
     * @return the ending node of this relationship
     */
    GraphNode getEndNode();

    /**
     * A convenience operation that, given a node that is attached to this relationship, returns the other node.
     * @param node the start or end node of this relationship
     * @return the other node
     */
    GraphNode getOtherNode(GraphNode node);

    /**
     * Convenience operation that checks if this relationship is also established in the other direction.
     * @return <code>true</code> if this relationship is bidirectional.
     */
    boolean isBidirectional();

    /**
     * Returns the type of this relationship.
     * @return the type of this relationship
     */
    String getType();

    /**
     * Specifies tha this relationship is virtual and is not (and will never be) backed by a persistence object.
     * @return <code>true</code> if this node is virtual.
     */
    boolean isVirtual();

    /**
     * Deletes this relationship.
     */
    void delete();

    /**
     * Returns a signature of the relationship of the form: <code>startNodeId ":" type ":" endNodeId</code>,
     * with the ids and type url encoded. the equals and hash methods should use this signature for equality check.
     * @return the signature
     */
    String signature();

    /**
     * Get a named property and convert it into the given type.
     *
     * @param name The name of the property
     * @param type The class of the type
     * @param <T> the type
     * @return Return named value converted to type T or <code>null</code> if
     *         non existing or can't be converted.
     */
    <T> T get(String name, Class<T> type);

    /**
     * Get a named property and convert it into the given type.
     *
     * @param name The name of the property
     * @param defaultValue The default value to use if the named property does
     *            not exist or cannot be converted to the requested type. The
     *            default value is also used to define the type to convert the
     *            value to. If this is <code>null</code> any existing property is
     *            not converted.
     * @param <T> the type
     * @return Return named value converted to type T or the default value if
     *         non existing or can't be converted.
     */
    <T> T get(String name, T defaultValue);

}