/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2012 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.adobe.granite.taskmanagement;

import org.osgi.annotation.versioning.ProviderType;

import java.util.Iterator;

/**
 * TaskManager defines the task management API.
 */
@ProviderType
public interface TaskManager {
    /**
     * Creates a task.
     * note that if no task id is provided one will be created and assigned to the resulting task.
     * Optionally if a task id is provided then the task id will be used for the new task unless that
     * task id exists already in the system in which case an exception will be thrown for the duplicate
     * task id.
     *
     * @param task Task to create
     * @return newly created task instance
     * @throws TaskManagerException An error occurred creating the task.
     */
    Task createTask(Task task) throws TaskManagerException;

    /**
     * Creates a subtask under the specified parentTaskid
     * @param parentTaskId the id of the parent task
     * @param task the task to create
     * @return the newly created task instance
     * @throws TaskNotFoundException if the parent task specified is not found
     * @throws TaskManagerException an error occurred creating the task.
     */
    Task createTask(String parentTaskId, Task task) throws TaskNotFoundException, TaskManagerException;

    /**
     * returns all tasks for the specified task type, or all tasks if no tasktype specified. <BR>
     * Tasks assigned to the currently logged in user are returned.
     * Note that a task's subtasks are not populated by this method.
     * @param filter Only tasks matching the conditions in this filter are returned in the resulting iterator.
     * @param startIndex The index of the first task returned.  This value must be &gt; 0
     * @param length The maximum number of tasks returned.  This value must be non-zero.  A negative value indicates all results are to be returned
     * @return tasks matching the filter and paging information.
     * @throws TaskManagerException An error occurred retrieving the tasks.
     */
    Iterator<Task> getTasks(Filter filter, int startIndex, int length) throws TaskManagerException;

    /**
     * Same as calling {@link #getTasks(Filter, int, int)} with startIndex of 0, and length of -1
     * Used as a convenience for calls that do not need to limit the number of tasks being returned
     * @param filter Only tasks matching the conditions in this filter are returned in the resulting iterator.
     * @return tasks matching the filter
     * @throws TaskManagerException An error occurred retrieving the tasks
     */
    Iterator<Task> getTasks(Filter filter) throws TaskManagerException;

    /**
     * Retrieves a fully populated task instance for the given task id.
     * @param taskId String containing a task id.
     * @return Task instance retrieved for the given task id.
     * @throws TaskManagerException An error occurred retrieving the task instance.
     */
    Task getTask(String taskId) throws TaskManagerException;
    /**
     * Retrieves a fully populated task instance for the given task id.
     * @param taskUUID String containing a task UUID
     * @return Task instance retrieved for the given task id.
     * @throws TaskManagerException An error occurred retrieving the task instance.
     */
    Task getTaskByUUID(String taskUUID) throws TaskManagerException;

    /**
     * Retrieves the task identified by taskId and opotionally retrieves the task's subtasks.
     * @param taskId String containing a task id.
     * @param retrieveSubTasks true to retrieve this task's subtasks, false otherwise.
     * @return Task instance for the given taskId
     * @throws TaskManagerException an error occurred retrieving the task instance.
     */
    Task getTask(String taskId, boolean retrieveSubTasks) throws TaskManagerException;

    /**
     * Saves the given task instance.
     * Note that subtasks will not be updated by this call.
     * Note: you cannot complete a task with this method; to complete a task call
     *  {@link TaskManager#completeTask(String, String)}
     * @param task Task instance to save.
     * @return Saved copy of the given Task instance.
     * @throws TaskNotFoundException The task being saved could not be found
     * @throws TaskManagerException An error occurred saving the task instance.
     */
    Task saveTask(Task task) throws TaskNotFoundException, TaskManagerException;

    /**
     * Deletes the task with the given task ID.
     * @param taskId String containing the ID of the task to delete.
     * @throws TaskNotFoundException The task being saved could not be found
     * @throws TaskManagerException An error occurred deleting the task instance.
     */
    void deleteTask(String taskId) throws TaskNotFoundException, TaskManagerException;

    /**
     * Completes the task with the given id and action.
     * @param taskId String containing the ID of the task to complete.
     * @param actionId the Id of the action to complete the task with, or null.
     * @throws TaskNotFoundException The task being completed could not be found
     * @throws TaskManagerException An error occurred completing the task instance.
     */
    void completeTask(String taskId, String actionId) throws TaskNotFoundException, TaskManagerException;

    /**
     * Terminates the task
     * @param taskId String containing the ID of the task to complete.
     * @throws TaskNotFoundException The task being completed could not be found
     * @throws TaskManagerException An error occurred completing the task instance.
     */
    void terminateTask(String taskId) throws TaskNotFoundException, TaskManagerException;



    /**
     * Returns a task manager factory to instantiate various objects of the taskmanager interface like Task and TaskAction.
     * @see TaskManagerFactory
     * @return instance of TaskManagerFactory
     */
    TaskManagerFactory getTaskManagerFactory();

}
