/*************************************************************************
* ADOBE CONFIDENTIAL
* ___________________
*
* Copyright 2012 Adobe
* All Rights Reserved.
*
* NOTICE: All information contained herein is, and remains
* the property of Adobe and its suppliers, if any. The intellectual
* and technical concepts contained herein are proprietary to Adobe
* and its suppliers and are protected by all applicable intellectual
* property laws, including trade secret and copyright laws.
* Dissemination of this information or reproduction of this material
* is strictly forbidden unless prior written permission is obtained
* from Adobe.
**************************************************************************/
package com.adobe.granite.ui.components;

import java.io.IOException;
import java.io.Writer;

import javax.annotation.CheckForNull;
import javax.annotation.Nonnull;

/**
 * A representation of DOM tag. A tag has a name (e.g. div, span) and a set of attributes.
 */
public class Tag {
    private String name;

    @Nonnull
    private AttrBuilder attrs;

    /**
     * Creates a tag with the given name and attributes.
     * @param name the name of the tag
     * @param attrs the attributes
     */
    public Tag(@CheckForNull String name, @Nonnull AttrBuilder attrs) {
        this.name = name;
        this.attrs = attrs;
    }

    /**
     * Creates a tag with the given attributes. The name is {@code null} in this case.
     * @param attrs the attributes
     */
    public Tag(@Nonnull AttrBuilder attrs) {
        this.attrs = attrs;
    }

    /**
     * Returns the name of the tag (e.g. div, span).
     * @return the name
     */
    @CheckForNull
    public String getName() {
        return name;
    }

    /**
     * Sets the name of the tag.
     * @param name the name
     * @return the tag
     */
    @Nonnull
    public Tag setName(@CheckForNull String name) {
        this.name = name;
        return this;
    }

    /**
     * Returns the attributes of the tag.
     * @return the attributes of the tag
     */
    @Nonnull
    public AttrBuilder getAttrs() {
        return attrs;
    }

    /**
     * Sets attributes of the tag.
     * @param attrs the attributes to set
     * @return the tag with the newly set attributes
     */
    @Nonnull
    public Tag setAttrs(@Nonnull AttrBuilder attrs) {
        this.attrs = attrs;
        return this;
    }

    /**
     * Println the start tag of this tag (e.g. {@code "<div class='class1'>"}). Note that no escaping/encoding of the name
     * is performed.
     * @param out the writer
     * @return the tag
     * @throws java.io.IOException if there's a problem while printing to the writer
     */
    @Nonnull
    public Tag printlnStart(@Nonnull Writer out) throws IOException {
        out.append("<").append(name);

        if (!attrs.isEmpty()) {
            out.append(" ");
            attrs.build(out);
        }

        out.append(">\n");
        return this;
    }

    /**
     * Println the end tag of this tag (e.g. "&lt;/div&gt;"). Note that no escaping/encoding of the name is performed.
     * @param out the writer
     * @return the tag
     * @throws java.io.IOException if there's a problem while printing to the writer
     */
    @Nonnull
    public Tag printlnEnd(@Nonnull Writer out) throws IOException {
        out.append("</").append(name).append(">\n");
        return this;
    }
}
