/*************************************************************************
* ADOBE CONFIDENTIAL
* ___________________
*
* Copyright 2017 Adobe
* All Rights Reserved.
*
* NOTICE: All information contained herein is, and remains
* the property of Adobe and its suppliers, if any. The intellectual
* and technical concepts contained herein are proprietary to Adobe
* and its suppliers and are protected by all applicable intellectual
* property laws, including trade secret and copyright laws.
* Dissemination of this information or reproduction of this material
* is strictly forbidden unless prior written permission is obtained
* from Adobe.
**************************************************************************/
package com.adobe.granite.ui.components.rendercondition;

import java.io.IOException;
import java.util.HashMap;
import java.util.Map;

import javax.annotation.CheckForNull;
import javax.annotation.Nonnull;
import javax.servlet.ServletException;

import org.apache.sling.api.SlingHttpServletRequest;
import org.apache.sling.api.SlingHttpServletResponse;
import org.apache.sling.api.resource.Resource;
import org.apache.sling.api.resource.ResourceResolver;

import com.adobe.granite.ui.components.impl.SlingIncludeObjectFactory;

/**
 * A helper to get {@code RenderCondition} of a resource.
 */
public class RenderConditionHelper {
    @Nonnull
    private static final String NAME_RENDERCONDITION = "granite:rendercondition";

    private static final String ATTRIBUTE_CACHE_RC = RenderConditionHelper.class.getName();

    @Nonnull
    private SlingHttpServletRequest request;

    @Nonnull
    private SlingHttpServletResponse response;

    public RenderConditionHelper(@Nonnull SlingHttpServletRequest request, @Nonnull SlingHttpServletResponse response) {
        this.request = request;
        this.response = response;
    }

    /**
     * Returns the render condition of the given resource.
     *
     * This method is an overload of {@link #getRenderCondition(Resource, boolean)}
     * with cache = false.
     *
     * @param resource
     *            The resource whose render condition needs to be fetched
     * @return The render condition
     * @throws ServletException
     *             When there's a servlet error during include
     * @throws IOException
     *             When there's an I/O error during include
     */
    @Nonnull
    public RenderCondition getRenderCondition(@Nonnull Resource resource) throws ServletException, IOException {
        return getRenderCondition(resource, false);
    }

    /**
     * Returns the render condition of the given resource.
     *
     * The render condition is specified by {@code granite:rendercondition}
     * subresource. If there is no such subresource,
     * {@link SimpleRenderCondition#TRUE} is returned.
     *
     * @param resource
     *            The resource whose render condition needs to be fetched
     * @param cache
     *            {@code true} to cache the result; Use it when checking render
     *            condition of other resource (typically the item resource) so that
     *            the render condition is only resolved once.
     * @return The render condition
     * @throws ServletException
     *             When there's a servlet error during include
     * @throws IOException
     *             When there's an I/O error during include
     */
    @Nonnull
    public RenderCondition getRenderCondition(@Nonnull Resource resource, boolean cache)
            throws ServletException, IOException {
        Map<String, RenderCondition> cacheMap = getRenderConditionCache();

        String key = resource.getPath();

        RenderCondition rc = cacheMap.get(key);

        if (rc != null) {
            return rc;
        }

        Resource condition = resource.getChild(NAME_RENDERCONDITION);

        if (condition != null) {
            SlingIncludeObjectFactory factory = new SlingIncludeObjectFactory(request, response);
            rc = factory.get(condition, getResourceType(condition), RenderCondition.class);
        }

        if (rc == null) {
            rc = SimpleRenderCondition.TRUE;
        }

        if (cache) {
            cacheMap.put(key, rc);
        }

        return rc;
    }

    @Nonnull
    private Map<String, RenderCondition> getRenderConditionCache() {
        @SuppressWarnings("unchecked")
        Map<String, RenderCondition> cache = (Map<String, RenderCondition>) request.getAttribute(ATTRIBUTE_CACHE_RC);

        if (cache == null) {
            cache = new HashMap<String, RenderCondition>();
            request.setAttribute(ATTRIBUTE_CACHE_RC, cache);
        }

        return cache;
    }

    @SuppressWarnings("null")
    @CheckForNull
    private static String getResourceType(@Nonnull Resource resource) {
        return resource.getValueMap().get(ResourceResolver.PROPERTY_RESOURCE_TYPE, String.class);
    }
}
