/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2012 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.adobe.granite.workflow.job;

import com.adobe.granite.workflow.exec.WorkItem;
import org.apache.sling.event.EventUtil;
import org.osgi.service.event.Event;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.Serializable;
import java.util.Calendar;
import java.util.Date;
import java.util.Dictionary;
import java.util.Hashtable;

/**
 * The <code>TimeoutJob</code> class provides a utility for creating event
 * triggered job used for programmed/scripted workflow steps.
 */
public class TimeoutJob implements Serializable {

    private static final Logger log = LoggerFactory.getLogger(TimeoutJob.class);

    /**
     * Generated serial version UID.
     */
    private static final long serialVersionUID = 5670996916430565635L;

    /**
     * The job topic for adding an entry to the audit log.
     */
    public static final String TIMEOUT_JOB_TOPIC = "com/adobe/granite/workflow/timeout/job";

    /**
     * The event property holding the {@link TimeoutJob}.
     */
    public static final String TIMEOUT_JOB = "com.adobe.granite.workflow.console.timeout.job";

    /**
     * The {@link com.adobe.granite.workflow.exec.WorkItem} <code>Id</code> of the job.
     */
    protected String itemId;

    /**
     * The timeout job handler
     */
    String handler;

    protected long timeStarted = 0;

    /**
     * Creates a new WorkflowJob.
     * @param item the work item
     * @param handler the timeout job handler
     */
    public TimeoutJob(WorkItem item, String handler) {
        if (item == null || handler == null) {
            throw new IllegalArgumentException("item or handler must not be null.");
        }
        this.itemId = item.getId();
        this.timeStarted = item.getTimeStarted().getTime();
        this.handler = handler;
    }

    /**
     * Returns the {@link com.adobe.granite.workflow.exec.WorkItem} {@code Id} of the job
     * @return the {@link com.adobe.granite.workflow.exec.WorkItem} {@code Id} of the job
     */
    public String getWorkItemId() {
        return itemId;
    }

    /**
     * Returns the timeout handler
     * @return the timeout handler
     */
    public String getHandler() {
        return handler;
    }

    /**
     * Convenience method to create a job event for the workflow job.
     * @param executeParallel whether to execute in parallel or not
     * @param seconds number of seconds
     * @param addOffset whether to add the current time to the specified number of seconds
     * @return the created event
     */
    public Event createEvent(boolean executeParallel, long seconds, boolean addOffset) {
        final Dictionary<String, Object> props = new Hashtable<String, Object>();
        props.put(TIMEOUT_JOB, this);
        props.put(EventUtil.PROPERTY_JOB_PARALLEL, executeParallel);
        props.put(EventUtil.PROPERTY_TIMED_EVENT_TOPIC, EventUtil.TOPIC_JOB);
        props.put(EventUtil.PROPERTY_JOB_TOPIC, TIMEOUT_JOB_TOPIC);
        props.put(EventUtil.PROPERTY_TIMED_EVENT_ID, itemId + "_" + timeStarted);
        Date date = calculateDate(seconds, addOffset);
        props.put(EventUtil.PROPERTY_TIMED_EVENT_DATE, date);
        log.debug("create timeout event {} ", props);
        return new Event(EventUtil.TOPIC_TIMED_EVENT, props);
    }

    public Event cancelEvent(boolean executeParallel) {
        final Dictionary<String, Object> props = new Hashtable<String, Object>();
        props.put(TIMEOUT_JOB, this);
        props.put(EventUtil.PROPERTY_JOB_PARALLEL, executeParallel);
        props.put(EventUtil.PROPERTY_TIMED_EVENT_TOPIC, EventUtil.TOPIC_JOB);
        props.put(EventUtil.PROPERTY_JOB_TOPIC, TIMEOUT_JOB_TOPIC);
        props.put(EventUtil.PROPERTY_TIMED_EVENT_ID, itemId + "_" + timeStarted);
        log.debug("create timeout cancel event {}", props);
        return new Event(EventUtil.TOPIC_TIMED_EVENT, props);
    }

    private Date calculateDate(long seconds, boolean addOffset) {
        Calendar cal = Calendar.getInstance();
        long millies = seconds * 1000;
        if (addOffset) {
            millies += cal.getTimeInMillis();
        }
        cal.setTimeInMillis(millies);
        return cal.getTime();
    }
}
