/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2012 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.adobe.granite.workflow.model;

import java.util.List;
import java.util.Map;

import com.adobe.granite.workflow.HasMetaData;
import org.osgi.annotation.versioning.ProviderType;

/**
 * <code>WorkflowModel</code> represents a model/definition of a workflow.
 * 
 * It provides methods for retrieving the entities of the model like
 * <code>{@link WorkflowNode}</code>s and
 * <code>{@link WorkflowTransition}</code>s as well as common model attributes
 * like name, description or version.
 */
@ProviderType
public interface WorkflowModel extends HasMetaData {
    /**
     * Returns the ID of the <code>WorkflowModel</code>.
     * 
     * @return The ID of the <code>WorkflowModel</code>.
     */
    String getId();

    /**
     * Returns the title of the <code>WorkflowModel</code>.
     * 
     * @return The title of the <code>WorkflowModel</code>.
     */
    String getTitle();

    /**
     * Sets the title of the <code>WorkflowModel</code>.
     * 
     * @param title
     *            The new title of the <code>WorkflowModel</code>.
     */
    void setTitle(String title);

    /**
     * Returns the description of the <code>WorkflowModel</code>.
     * 
     * @return The description of the <code>WorkflowModel</code>.
     */
    String getDescription();

    /**
     * Sets the description of the <code>WorkflowModel</code>.
     * 
     * @param description
     *            The new description of the <code>WorkflowModel</code>.
     */
    void setDescription(String description);

    /**
     * Returns the version of the <code>WorkflowModel</code>.
     * 
     * @return The version of the <code>WorkflowModel</code>.
     */
    String getVersion();

    /**
     * Returns the list of <code>{@link WorkflowNode}</code>s contained by the
     * <code>WorkflowModel</code>.
     * 
     * @return The list of <code>{@link WorkflowNode}</code>s of the
     *         <code>WorkflowModel</code>.
     */
    List<WorkflowNode> getNodes();

    /**
     * @return the created workflow node
     */
    WorkflowNode createNode();

    /**
     * @param title the title
     * @param type the type
     * @param description the description
     * @return the created workflow node
     */
    WorkflowNode createNode(String title, String type, String description);

    /**
     * @param node the workflow node
     */
    public void setRootNode(WorkflowNode node);

    /**
     * @param node the workflow node
     */
    public void setEndNode(WorkflowNode node);

    /**
     * Returns the <code>{@link WorkflowNode}</code> with the given id
     * 
     * @param id
     *            The ID of the <code>{@link WorkflowNode}</code>.
     * 
     * @return The <code>WorkflowNode</code> or null if not found.
     */
    WorkflowNode getNode(String id);

    /**
     * Returns the root/start <code>{@link WorkflowNode}</code>.
     * 
     * @return The root <code>{@link WorkflowNode}</code>.
     */
    WorkflowNode getRootNode();

    /**
     * Returns the end <code>{@link WorkflowNode}</code>.
     * 
     * @return The end <code>{@link WorkflowNode}</code>.
     */
    WorkflowNode getEndNode();

    /**
     * Returns the list of <code>{@link WorkflowTransition}</code>s contained by
     * the <code>WorkflowModel</code>.
     * 
     * @return The list of <code>{@link WorkflowTransition}</code>s of the
     *         <code>WorkflowModel</code>.
     */
    List<WorkflowTransition> getTransitions();

    /**
     * @return the created workflow transition
     */
    WorkflowTransition createTransition();

    /**
     * @param from the workflow node to begin the transition with
     * @param to the workflow node to end the transition with
     * @param rule the rule
     * @return the created workflow transition
     */
    WorkflowTransition createTransition(WorkflowNode from, WorkflowNode to,
            String rule);

    /**
     * Validates the model.
     * 
     * @throws ValidationException in case the validation fails
     */
    void validate() throws ValidationException;

    /**
     * @return a {@link Map} with keys as variables' names and values as <code>{@link VariableTemplate}</code>s .
     */
    Map<String, VariableTemplate> getVariableTemplates();

    /**
     * @param name the name of the variable
     * @param dataType the type of the variable in java fully qualified class name format
     * @return the created variable template
     */
    VariableTemplate createVariableTemplate(String name, String dataType);
}
