package com.adobe.internal.io;

import java.io.IOException;
import java.nio.BufferOverflowException;
import java.nio.ByteBuffer;
import java.nio.ReadOnlyBufferException;

/**
 * A {@link com.adobe.internal.io.ByteWriter ByteWriter} that wraps a
 * {@link java.nio.ByteBuffer ByteBuffer}.
 *
 * This class is <b>not</b> threadsafe.  It is not safe to pass an instance of this class
 * to multiple threads.  It is not safe to pass an instance of this class to multiple users even
 * if in the same thread.  It is not safe to give the same ByteBuffer to multiple instances
 * of this class.
 */
public class ByteBufferByteWriter implements ByteWriter 
{
	private ByteBuffer buffer;
	
	/**
	 * Create a new ByteBufferByteWriter that wraps the given ByteBuffer.
	 * @param buffer a buffer to use for writing to.
	 * @throws IOException
	 */
	public ByteBufferByteWriter(ByteBuffer buffer) throws IOException
	{
		this.buffer = buffer;
	}
	
	/**
	 * Create a new ByteBufferByteReader with the given byte array.
	 * The buffer is not copied but is wrapped with a ByteBuffer before use.
	 * @param b the byte array to use.
	 * @throws IOException 
	 * @see com.adobe.internal.io.ByteArrayByteReader
	 */
	public ByteBufferByteWriter(byte[] b) 
	throws IOException
	{
		this(ByteBuffer.wrap(b));
	}
	
	/**
	 * @see com.adobe.internal.io.ByteWriter#write(long, byte[], int, int)
	 */
	public void write(long position, byte[] b, int offset, int length) throws IOException
	{
		try {
			if (position > Integer.MAX_VALUE)
			{
				throw new IndexOutOfBoundsException("Attempt to position beyond the buffer size.");
			}
			// XXX synchronization
			//synchronized(this.buffer)
			{
				this.buffer.position((int) position);
				this.buffer.put(b, offset, length);
			}
		} catch (BufferOverflowException e)
		{
			IOException exception = new IOException();
			exception.initCause(e);
			throw exception;
		} catch (IndexOutOfBoundsException e)
		{
			IOException exception = new IOException();
			exception.initCause(e);
			throw exception;
		} catch (ReadOnlyBufferException e)
		{
			IOException exception = new IOException();
			exception.initCause(e);
			throw exception;
		}
	}
	
	/**
	 * @see com.adobe.internal.io.ByteWriter#write(long, int)
	 */
	public void write(long position, int b) throws IOException
	{
		try {
			if (position > Integer.MAX_VALUE)
			{
				throw new IndexOutOfBoundsException("Attempt to position beyond the buffer size.");
			}
			// absolute put - no need to position
			this.buffer.put((int)position, (byte) b);
		} catch (BufferOverflowException e)
		{
			IOException exception = new IOException();
			exception.initCause(e);
			throw exception;
		} catch (IndexOutOfBoundsException e)
		{
			IOException exception = new IOException();
			exception.initCause(e);
			throw exception;
		} catch (ReadOnlyBufferException e)
		{
			IOException exception = new IOException();
			exception.initCause(e);
			throw exception;
		}
	}
	
	/**
	 * @see com.adobe.internal.io.ByteWriter#length()
	 */
	public long length() throws IOException
	{
		return this.buffer.limit();
	}
	
	/**
	 * @see com.adobe.internal.io.ByteWriter#flush()
	 */
	public void flush() throws IOException
	{
		// ByteBuffer's don't implement flush()
	}
	
	/**
	 * @see com.adobe.internal.io.ByteWriter#close()
	 */
	public void close() throws IOException
	{
		this.buffer = null;
	}
	
	/**
	 * @see java.lang.Object#toString()
	 */
	public String toString()
	{
		return this.buffer.toString();
	}
	
	/*************************************** Reader Methods **********************************/
	
	/**
	 * @see com.adobe.internal.io.ByteReader#read(long)
	 */
	public int read(long position) throws IOException
	{
		if ((position < 0) || (position >= buffer.limit()))
		{
			return ByteReader.EOF;
		}
		
		// XXX synchronization
		//synchronized(this.buffer)
		{            
			// must be sure to clean up high order bits
			return this.buffer.get((int)position) & 0x00ff;
		}
	}
	
	/**
	 * @see com.adobe.internal.io.ByteReader#read(long, byte[], int, int)
	 */
	public int read(long position, byte[] b, int offset, int length) throws IOException
	{
		if ((position < 0) || (position >= buffer.limit()))
		{
			return ByteReader.EOF;
		}
		
		// XXX synchronization
		//synchronized(this.buffer)
		{
			this.buffer.position((int) position);
			length = (int) Math.min(length, buffer.limit() - position);
			this.buffer.get(b, offset, length);
		}
		return length;
	}
	
	
}
