/*
*
*	File: ByteReader.java
*
*
*	ADOBE CONFIDENTIAL
*	___________________
*
*	Copyright 2005 Adobe Systems Incorporated
*	All Rights Reserved.
*
*	NOTICE: All information contained herein is, and remains the property of
*	Adobe Systems Incorporated and its suppliers, if any. The intellectual
*	and technical concepts contained herein are proprietary to Adobe Systems
*	Incorporated and its suppliers and may be covered by U.S. and Foreign
*	Patents, patents in process, and are protected by trade secret or
*	copyright law. Dissemination of this information or reproduction of this
*	material is strictly forbidden unless prior written permission is obtained
*	from Adobe Systems Incorporated.
*
*/
package com.adobe.internal.io;

import java.io.IOException;

/**
 * <p>The <code>ByteReader</code> provides an abstraction to a linear, zero-based, semi-infinite
 * array of bytes that can be read from at any point.  The actual repository can store
 * these bytes in any manner that best fits the storage medium but the view on them
 * provided by this interface must be as above.
 */
public interface ByteReader
{
    /**
     * The value returned when reads occur outside of the legal range.
     */
    public static final int EOF = -1;
    
    /**
     * Read a single byte from the underlying bytes at the given position.
     * @param position the position to read the byte from.
     * @return the byte at the postion or EOF if the position is outside of the legal range.
     * @throws IOException if an error occurs during the read operation
     */
    int read(long position) throws IOException;
    
    /**
     * Transfers bytes from the underlying repository into the given destination array.
     * Copy up to <code>length</code> bytes from the repository starting at the postion 
     * given into the given array starting at the given offset.
     * @param position the position to read the byte from.
     * @param b the array to write the bytes into.
     * @param offset the offset in the array at which the first byte is written.
     * @param length the maximum number of bytes to write into the array.
     * @return the number of bytes actually written to the array.
     * @throws IOException if an error occurs during the read operation
     */
    int read(long position, byte[] b, int offset, int length) throws IOException;
    
    /**
     * Returns the number of total bytes in the repository that are visible through
     * this API.
     * @return the number of bytes in the repository.
     * @throws IOException if an error occurs while trying to determine the length
     */
    long length() throws IOException;
    
    /**
     * Closes this <code>ByteReader</code> and releases any system resources 
     * associated with this stream.
     * @throws IOException if an error occurs while trying to close
     */
    void close() throws IOException;    
}
