package com.adobe.internal.io;

import java.io.IOException;

/**
 * <p>The <code>ByteWriter</code> provides an abstraction to a linear, zero-based, semi-infinite
 * array of bytes that can be read from at any point.  The actual repository can store
 * these bytes in any manner that best fits the storage medium but the view on them
 * provided by this interface must be as above.
 */
public interface ByteWriter extends ByteReader 
{
	 /**
     * Write the byte given at the position given.  If during the write operation
     * the byte to be written is beyond the end of the virtual array visible through 
     * this interface then the length of that virtual array becomes the position of the
     * byte written plus one (because of zero based counting).
     * <code>length of virtual array = max(length of virtual array, position of byte written + 1)
     * @param position the zero-based offset within the byte array.
     * @param b the byte to write.
     * @throws IOException if an error occurs during the write operation
     */
    void write(long position, int b) throws IOException;
    
    /**
     * Write an array of bytes at the position given.  If during the write operation
     * any of the bytes are to be written beyond the end of the virtual array visible through 
     * this interface then the length of that virtual array becomes the position of the
     * byte written plus one (because of zero based counting).
     * for each byte written : <code>length of virtual array = max(length of virtual array, position of byte written + 1)
     * @param position the zero-based offset within the byte array.
     * @param b the array of bytes to write from.
     * @param offset the offset within the byte array to start writing from.
     * @param length the number of bytes to write from the byte array.
     * @throws IOException if an error occurs during the write operation
     */
    void write(long position, byte[] b, int offset, int length) throws IOException;
    
    /**
     * Flushes this <code>ByteWriter</code> and forces any unwritten buffered output bytes 
     * to be written out to the underlying repository.
     * @throws IOException if an error occurs while trying to flush the buffer
     */
    void flush() throws IOException;
}
