/**
 * 
 */
package com.adobe.internal.io;

import java.io.File;
import java.io.IOException;
import java.io.RandomAccessFile;

/**
 * An implementation of the ByteWriter interface that provides lazy access to a 
 * {@link java.io.RandomAccessFile RandomAccessFile}.  It will not open the
 * {@link java.io.File File} until the first attempt to read or operate on that 
 * {@link java.io.File File}.  At that point a 
 * {@link com.adobe.internal.io.RandomAccessFileByteReader RandomAccessFileByteReader}
 * will be constructed and used.
 * 
 * This class is <b>not</b> threadsafe.  It is not safe to pass an instance of this class
 * to multiple threads.  It is not safe to pass an instance of this class to multiple users even
 * if in the same thread.
 */
public class LazyRandomAccessFileByteWriter implements ByteWriter
{
	private File file;
	private ByteWriter byteWriter;
	private boolean openFailed = false;
	
	/**
	 * @param file
	 * @throws IOException
	 */
	public LazyRandomAccessFileByteWriter(File file)
	throws IOException
	{
		if (file == null)
		{
			throw new IOException("File is not valid.");
		}
		this.file = file;
	}
	
	/**
	 * @see com.adobe.internal.io.ByteReader#read(long)
	 */
	public int read(long position)
	throws IOException
	{
		return this.getByteWriter().read(position);
	}
	
	/**
	 * @see com.adobe.internal.io.ByteReader#read(long, byte[], int, int)
	 */
	public int read(long position, byte[] b, int offset, int length)
	throws IOException
	{
		return this.getByteWriter().read(position, b, offset, length);
	}
	
	/**
	 * @see com.adobe.internal.io.ByteReader#length()
	 */
	public long length()
	throws IOException
	{
		return this.getByteWriter().length();
	}
	
	/**
	 * @see com.adobe.internal.io.ByteReader#close()
	 */
	public void close()
	throws IOException
	{
		if (this.byteWriter != null)
		{
			this.byteWriter.close();
		}
		this.byteWriter = null;
		this.file = null;
	}
	
	/**
	 * Just get the <code>ByteWriter</code> if it's already been created otherwise create it first.
	 * @return the byte writer
	 * @throws IOException
	 */
	private ByteWriter getByteWriter()
	throws IOException
	{
		if (this.byteWriter != null)
		{
			return this.byteWriter; 
		}
		if (this.openFailed)
		{
			throw new IOException("Unable to open file.");
		}
		
		try
		{
			RandomAccessFile raf;
			try
			{
				raf = new RandomAccessFile(this.file, "r");
			} catch (Exception e) {
				IOException ioException = 
					new IOException("Unable to open the RandomAccessFile for the File " + this.file);
				ioException.initCause(e);
				throw ioException;
			}
			this.byteWriter = new RandomAccessFileByteWriter(raf);
		} catch (IOException e) {
			this.openFailed = true;
		}
		return this.byteWriter;		
	}
	
	/**
	 * Write the byte given at the position given.
	 * @param position the zero-based offset within the byte array.
	 * @param b the byte to write.
	 * @throws IOException
	 */
	public void write(long position, int b) throws IOException
	{
		this.getByteWriter().write(position,b);
	}
	
	/**
	 * Write an array of bytes at the position given.
	 * @param position the zero-based offset within the byte array.
	 * @param b the array of bytes to write from.
	 * @param offset the offset within the byte array to start writing from.
	 * @param length the number of bytes to write from the byte array.
	 * @throws IOException
	 */
	public void write(long position, byte[] b, int offset, int length) throws IOException
	{
		this.getByteWriter().write(position,b,offset,length);
	}
	
	/**
	 * Flushes this <code>ByteWriter</code> and forces any buffered output bytes to be written out.
	 * @throws IOException
	 */
	public void flush() throws IOException
	{
		this.getByteWriter().flush();
	}
}
