/**
 * 
 */
package com.adobe.internal.io;

import java.io.FilterOutputStream;
import java.io.IOException;
import java.io.OutputStream;

/**
 * An <code>OutputStream</code> that searches for a specific tag in the data stream 
 * and when found it calls back.
 */
public class TagSearchingOutputStream extends FilterOutputStream
{
	/**
	 * The mechanism to let the client know that the tag has been found.
	 *
	 */
	public interface Callback
	{
		/**
		 * The method which will called when the tag is found.
		 * @return <code>true</code> if the tag should continue to be searched for; <code>false</code> otherwise
		 */
		boolean tagFound(TagSearchingOutputStream oStream) throws IOException;		
	}

	protected byte[] tag;
	protected Callback callback;
	protected boolean lookForTag = true;
	protected int outputTag = 0;
	protected int inTagPosition = 0;
	
	/**
	 * Constructor.
	 * @throws IOException 
	 */
	public TagSearchingOutputStream(OutputStream out, byte[] tag,  Callback callback) 
	throws IOException
	{
		super(out);
		if (tag == null || callback == null)
		{
			throw new IOException("Invalid parameters to the constructor.");
		}
		this.tag = tag.clone();
		this.callback = callback;
	}

	/**
	 * Turn on or off the tag searching.  This must <b>not</b> be called during the
	 * {@link Callback#tagFound(TagSearchingOutputStream)}.  If tag searching should
	 * be turned off after the return from the callback then the client can use the
	 * return value from the callback method to turn searching off.
	 * @param search <code>true</code> if tag should be searched for; <code>false</code> otherwise
	 * @return the tag search state before the call
	 * @throws IOException
	 */
	public boolean searchForTag(boolean search) throws IOException
	{
		if (search == this.lookForTag)
		{
			return this.lookForTag;
		}
		this.writeTagBytes(true);
		this.lookForTag = search;
		return !this.lookForTag;	// if != above then ! current value
	}
	
	/**
	 * Query whether the tag searching is turned on or off.
	 * @return the current tag search state
	 * <code>true</code> if tag searching is on; <code>false</code> otherwise
	 * @throws IOException
	 */
	public boolean searchForTag() throws IOException
	{
		return this.lookForTag;
	}
	
	/**
	 * Sets whether the tag being searched for is output or not and if so if it is before or
	 * after the call to {@link Callback#tagFound(TagSearchingOutputStream)}.  By default
	 * the tag being searched for will not be output.
	 * @param tagOutput <code>0</code> if tag should not be output;
	 * <code>-1</code> (or any -ve number) if the tag should be output before the callback;
	 * <code>1</code> (or any +ve number) if the tag should be output after the callback
	 * @return the tag output state before the call
	 * @throws IOException
	 */
	public int outputTag(int tagOutput) throws IOException
	{
		int oldTagOutput = this.outputTag;
		this.outputTag = tagOutput;
		return oldTagOutput;
	}
	
	/**
	 * Queries whether the tag being searched for is output or not and if so if it is before or
	 * after the call to {@link Callback#tagFound(TagSearchingOutputStream)}.  By default
	 * the tag being searched for will not be output.
	 * <code>-1</code> (or any -ve number) if the tag should be output before the callback;
	 * <code>1</code> (or any +ve number) if thte tag should be output after the callback
	 * @return the current tag output state
	 * <code>0</code> if tag should not be output;
	 * <code>-1</code> (or any -ve number) if the tag should be output before the callback;
	 * <code>1</code> (or any +ve number) if the tag should be output after the callback
	 * @throws IOException
	 */
	public int outputTag() throws IOException
	{
		return this.outputTag;
	}
	
	/* (non-Javadoc)
	 * @see java.io.FilterOutputStream#write(int)
	 */
	public void write(int b) throws IOException
	{
		if (!this.checkForTag((byte) b))
		{
			if (inTagPosition != 0)
			{
				// we found a non-tag byte but had some residual tag-like bytes to write out first
				this.writeTagBytes(true);
			}
			// must check again
			// the byte may not have matched current position in the tag but may match the start
			if (!this.checkForTag((byte) b))
			{
				super.write(b);
			}
		}
	}
	
	/* (non-Javadoc)
	 * @see java.io.FilterOutputStream#close()
	 */
	public void close() throws IOException
	{
		this.writeTagBytes(true);
		super.close();
	}

	/**
	 * Check that the current byte matches the current state of the tag search.  If this byte completes
	 * the tag then the callback is made and potentially the tag bytes are written out depending on the
	 * flag setting for tag byte output.  If the tag isn't found then the current byte is not written out 
	 * nor any accumulated partial tag bytes.
	 * @param b the current byte
	 * @return <code>true</code> if the byte is potential match for the tag; <code>false</code> otherwise
	 * @throws IOException
	 */
	private boolean checkForTag(byte b) throws IOException
	{
		if (!this.lookForTag)
		{
			return false;
		}
		
		if (b == this.tag[this.inTagPosition])
		{
			this.inTagPosition++;
			if (this.inTagPosition >= this.tag.length)
			{
				int outputTag = this.outputTag;
				// output tag first?
				if (outputTag < 0)
				{
					this.writeTagBytes(true);
				}
				
				// tag found
				this.tagFound();
				
				// don't output tag?
				if (outputTag == 0)
				{
					// even if we don't output it we need to clear everything
					this.writeTagBytes(false);
				}
				
				// output tag last?
				if (outputTag > 0)
				{
					this.writeTagBytes(true);
				}
			}
			return true;
		} else {
			return false;
		}
	}
	
	/**
	 * Write out the full or partial tag bytes and reset the state to continue looking.  This method
	 * should also be called to clear the state of the tag search even if the tag shouldn't be written out.
	 * @param write <code>true</code> if the tag bytes should be written; <code>false</code> otherwise
	 * @throws IOException
	 */
	private void writeTagBytes(boolean write) throws IOException
	{
		if (write && inTagPosition != 0)
		{
			this.writeBytes(this.tag, this.inTagPosition);
		}		
		this.inTagPosition = 0;
	}
	
	/**
	 * Need this internal method because the parent FilterOutputStream will call this class's
	 * single byte write() multiple times if we call the multiple byte write().  That would
	 * cause some recursion in the single byte write() and confusion with the tag sensing code.
	 * 
	 * @param b
	 * @param length
	 * @throws IOException
	 */
	private void writeBytes(byte[] b, int length) throws IOException
	{
		for (int i = 0; i < length; i++)
		{
			super.write(b[i]);
		}
	}

	/**
	 * The tag has been found.
	 * @throws IOException
	 */
	private void tagFound() throws IOException
	{
		this.lookForTag = this.callback.tagFound(this);
	}	
}
