/*
 * File: OutputByteStreamImpl.java
 *
 * ****************************************************************************
 *
 *	ADOBE CONFIDENTIAL
 *	___________________
 *
 *	Copyright 2005 Adobe Systems Incorporated
 *	All Rights Reserved.
 *
 *	NOTICE: All information contained herein is, and remains the property of
 *	Adobe Systems Incorporated and its suppliers, if any. The intellectual
 *	and technical concepts contained herein are proprietary to Adobe Systems
 *	Incorporated and its suppliers and may be covered by U.S. and Foreign
 *	Patents, patents in process, and are protected by trade secret or
 *	copyright law. Dissemination of this information or reproduction of this
 *	material is strictly forbidden unless prior written permission is obtained
 *	from Adobe Systems Incorporated.
 *
 * ***************************************************************************/
package com.adobe.internal.io.stream;

import java.io.IOException;

import com.adobe.internal.io.ByteReader;
import com.adobe.internal.io.ByteWriter;

/**
 * Only for internal engineering use. This api can change without notice.
 * 
 * This class provides access to a seekable output stream.
 * The actual underlying seekable sink is provided by the outside 
 * client as an object that implements the <code>ByteWriter</code> interface.  
 * There can and will be many different implementations of the <code>ByteWriter</code> 
 * interface with different operating characteristics.  So, make no
 * assumptions about the speed of writing and seeking - do so judiciously.
 */
/*package protected*/ class ByteWriterOutputByteStream extends OutputByteStreamImpl
{
	private ByteWriter byteWriter = null;
	private StreamManager streamManager;
	private boolean registered = false;

	/**
	 * Only for internal engineering use. This api can change without notice.
	 * 
	 * Construct an OutputByteStream from a ByteWriter.
	 * @param byteWriter
	 */
	ByteWriterOutputByteStream(StreamManager streamManager, ByteWriter byteWriter, boolean register)
	throws IOException
	{
		if (byteWriter == null)
		{
			throw new IOException("Null ByteWriter parameter.");
		}
		this.streamManager = streamManager;
		this.byteWriter = byteWriter;
		this.registered = register;
		if (register)
		{
			this.streamManager.registerOutputByteStream(this, byteWriter);
		}
	}

	ByteWriter getByteWriter()
	{
		return this.byteWriter;
	}

	/* (non-Javadoc)
	 * @see com.adobe.internal.io.stream.OutputByteStream#write(int)
	 */
	public void write(int b)
	throws IOException
	{
		this.byteWriter.write(this.getPosition(), b);
		this.seek(this.getPosition() + 1);
	}

	/* (non-Javadoc)
	 * @see com.adobe.internal.io.stream.OutputByteStream#write(byte[], int, int)
	 */
	public void write(byte[] bytes, int offset, int length)
	throws IOException
	{
		this.byteWriter.write(this.getPosition(), bytes, offset, length);
		this.seek(this.getPosition() + length);
	}

	/* (non-Javadoc)
	 * @see com.adobe.internal.io.stream.OutputByteStream#length()
	 */
	public long length()
	throws IOException
	{
		return this.byteWriter.length();
	}

	/* (non-Javadoc)
	 * @see com.adobe.internal.io.stream.OutputByteStream#eof()
	 */
	public boolean eof()
	throws IOException
	{
		return this.getPosition() >= this.byteWriter.length();
	}

	/* (non-Javadoc)
	 * @see com.adobe.internal.io.stream.OutputByteStream#close()
	 */
	public void close()
	throws IOException
	{
		if (this.registered)
		{
			this.streamManager.deregisterOutputByteStream(this, this.byteWriter);
		} else {
			this.byteWriter.close();
		}
		this.byteWriter = null;
	}

	/* (non-Javadoc)
	 * @see com.adobe.internal.io.stream.OutputByteStreamImpl#closeAndConvert()
	 */
	public InputByteStream closeAndConvert()
	throws IOException
	{
		ByteReader br = this.byteWriter;
		InputByteStream ibs = new ByteReaderInputByteStream(this.streamManager, br, this.registered);
		if (this.registered)
		{
			this.streamManager.deregisterOutputByteStream(this, this.byteWriter);
		} //else {
			// we can't close - we gave the ByteWriter to the new IBS
		//}
		this.byteWriter = null;
		return ibs;
	}

	/* (non-Javadoc)
	 * @see com.adobe.internal.io.stream.OutputByteStream#flush()
	 */
	public void flush()
	throws IOException
	{
		this.byteWriter.flush();
	}

	/* (non-Javadoc)
	 * @see com.adobe.internal.io.stream.OutputByteStream#toOutputStream()
	 */
	public SkippingOutputStream toOutputStream() 
	throws IOException
	{
		//OutputByteStreamImpl obs = new OutputByteStreamImpl(this);
		return new OutputStreamImpl(this);
	}

	//	for debug only
	/**
	 * Only for internal engineering use. This api can change without notice.
	 */
	public String toString()
	{
		StringBuilder message = null;
		try {
			byte[] buf = new byte[1000];
			long position = Math.max(this.getPosition() - 1000, 0);
			long bytesRead = this.byteWriter.read(position, buf, 0, buf.length);
			message = new StringBuilder("OutputByteStream [ position = ").append(this.getPosition()).append(", limit = ").append(this.length()).append(" ] "). 
					append((bytesRead == OutputByteStream.EOF) ? "" : 
						new String(buf, 0, (int) bytesRead, "US-ASCII"));
		} catch (Exception e) {
			throw new RuntimeException(e);
		}
		return message.toString();
	}

	// specialized write methods
	/**
	 * Only for internal engineering use. This api can change without notice.
	 * 
	 * @param s String to write
	 * @throws IOException 
	 */
	public void write(String s)
	throws IOException
	{
		char[] chars = s.toCharArray();
		byte[] bytes = new byte[chars.length];
		for (int i = 0; i < chars.length; ++i)
			bytes[i] = (byte) chars[i];
		write(bytes);
	}	
}