/*
 *
 *	File: ChainedInputByteStream.java
 *
 *
 *	ADOBE CONFIDENTIAL
 *	___________________
 *
 *	Copyright 2005-2006 Adobe Systems Incorporated
 *	All Rights Reserved.
 *
 *	NOTICE: All information contained herein is, and remains the property of
 *	Adobe Systems Incorporated and its suppliers, if any. The intellectual
 *	and technical concepts contained herein are proprietary to Adobe Systems
 *	Incorporated and its suppliers and may be covered by U.S. and Foreign
 *	Patents, patents in process, and are protected by trade secret or
 *	copyright law. Dissemination of this information or reproduction of this
 *	material is strictly forbidden unless prior written permission is obtained
 *	from Adobe Systems Incorporated.
 *
 */

package com.adobe.internal.io.stream;

import java.io.IOException;
import java.util.ArrayList;

final class ChainedInputByteStream extends InputByteStreamImpl
{
	private InputByteStream mByteStreams[];
	private long mLengths[];
	private int mCurrentByteStreamIndex;
	private long mPosition;
	private long mTotalLength;

	/**
	 * @param byteStreams
	 * @throws IOException
	 */
	ChainedInputByteStream(InputByteStream[] byteStreams)
	throws IOException
	{
		mByteStreams = byteStreams;
		mLengths = new long[mByteStreams.length];
		for (int i = 0; i < mLengths.length; i++) 
		{
			mByteStreams[i].seek(0);
			mLengths[i] = mByteStreams[i].length();
			mTotalLength += mLengths[i];
		}
	}

	/* (non-Javadoc)
	 * @see com.adobe.internal.pdftoolkit.core.cos.util.InputByteStream#getPosition()
	 */
	public long getPosition()
	throws IOException
	{
		return mPosition;
	}

	/* (non-Javadoc)
	 * @see com.adobe.internal.pdftoolkit.core.cos.util.InputByteStream#limit()
	 */
	public long length()
	throws IOException
	{
		return mTotalLength;
	}

	/* (non-Javadoc)
	 * @see com.adobe.internal.pdftoolkit.core.cos.util.InputByteStream#setPosition(long)
	 */
	public InputByteStream seek(long position)
	throws IOException
	{
		// pin the position between 0 and the end of the streams
		if (position < 0)
		{
			position = 0;
		}
		if (position >= this.mTotalLength)
		{
			position = this.mTotalLength - 1;
		}
		
		mPosition = position;
		mCurrentByteStreamIndex = 0;
		while (position >= mLengths[mCurrentByteStreamIndex]) 
		{
			position -= mLengths[mCurrentByteStreamIndex];
			mCurrentByteStreamIndex++;
		}
		mByteStreams[mCurrentByteStreamIndex].seek(position);
			
		return this;
	}

	/* (non-Javadoc)
	 * @see com.adobe.internal.pdftoolkit.core.cos.util.InputByteStream#slice(long, long)
	 */
	public InputByteStream slice(long begin, long length)
	throws IOException
	{
		if(begin == 0 && mTotalLength == 0)
			return new ChainedInputByteStream(new InputByteStream[0]);
		if(begin < 0 || begin >= mTotalLength || length < 0 || begin + length > mTotalLength)
			throw new IOException("Invalid Parameter");
		ArrayList bufs = new ArrayList();
		long endPosition = begin + length;
		long beginPosition = begin;
		int begin_index = 0;		
		while (beginPosition >= mLengths[begin_index]) 
		{
			beginPosition -= mLengths[begin_index];
			begin_index++;
		}
		int end_index = 0;
		while (endPosition > mLengths[end_index]) 
		{
			endPosition -= mLengths[end_index];
			end_index++;
		}
		int i;
		for(i = begin_index; i < end_index; i++) 
		{
			bufs.add(mByteStreams[i].slice(beginPosition, mLengths[i] - beginPosition));
			beginPosition = 0;
		}
		bufs.add(mByteStreams[i].slice(beginPosition, endPosition - beginPosition));		
		InputByteStream[] streams = new InputByteStream[bufs.size()];
		System.arraycopy(bufs.toArray(), 0, streams, 0, bufs.size());
		return new ChainedInputByteStream(streams);
	}

	/**
	 * Only for internal engineering use. This api can change without notice.
	 */
	/* (non-Javadoc)
	 * @see com.adobe.internal.pdftoolkit.core.cos.io.InputByteStream#release()
	 */
	public void close()
	throws IOException
	{
		for (int i = 0; i < this.mByteStreams.length; i++)
		{
			this.mByteStreams[i].close();
		}
		this.mByteStreams = null;
	}

	/* (non-Javadoc)
	 * @see java.io.InputStream#read()
	 */
	// TODO - unroll the recursion
	public int read()
	throws IOException
	{
		// Already at the end?
		if (this.mCurrentByteStreamIndex >= this.mLengths.length)
		{
			return InputByteStream.EOF;
		}
		int byteRead = this.mByteStreams[mCurrentByteStreamIndex].read();

		// loop over all remaining streams tring to get a byte read
		if (byteRead == InputByteStream.EOF) 
		{
			this.mCurrentByteStreamIndex++;
			if (this.mCurrentByteStreamIndex == this.mLengths.length)
			{
				return InputByteStream.EOF;
			}
			mByteStreams[this.mCurrentByteStreamIndex].seek(0);
			byteRead = read();
		}
		else 
		{
			mPosition++;
		}
		return byteRead;
	}

	public int read(byte[] bytes, int position, int length)
	throws IOException
	{
		int bytesCopied = 0;

		if (this.mCurrentByteStreamIndex >= this.mLengths.length)
		{
			return InputByteStream.EOF;
		}

		while (bytesCopied < length)
		{
			int bytesRead = this.mByteStreams[this.mCurrentByteStreamIndex].read(bytes, position + bytesCopied, length - bytesCopied);
			if (bytesRead == InputByteStream.EOF)
			{
				this.mCurrentByteStreamIndex++;

				// we've run out of streams to read through
				if (this.mCurrentByteStreamIndex >= this.mLengths.length) 
				{
					// we also didn't copy any bytes so it must be EOF
					if (bytesCopied == 0)
					{
						return InputByteStream.EOF;
					}
					break;
				}
			} else {
				bytesCopied += bytesRead;				
			}
		}
		this.mPosition += bytesCopied;
		return bytesCopied;
	}
}
