/* ****************************************************************************
 *
 *	File: ByteArrayUtil.java
 *
 * ****************************************************************************
 *
 *	ADOBE CONFIDENTIAL
 *	___________________
 *
 *	Copyright 2003-2006 Adobe Systems Incorporated
 *	All Rights Reserved.
 *
 *	NOTICE: All information contained herein is, and remains the property of
 *	Adobe Systems Incorporated and its suppliers, if any. The intellectual
 *	and technical concepts contained herein are proprietary to Adobe Systems
 *	Incorporated and its suppliers and may be covered by U.S. and Foreign
 *	Patents, patents in process, and are protected by trade secret or
 *	copyright law. Dissemination of this information or reproduction of this
 *	material is strictly forbidden unless prior written permission is obtained
 *	from Adobe Systems Incorporated.
 *
 * ***************************************************************************/

package com.adobe.internal.pdftoolkit.core.cos;


import java.io.IOException;

import com.adobe.internal.io.stream.InputByteStream;
import com.adobe.internal.pdftoolkit.core.exceptions.PDFCosParseException;
import com.adobe.internal.pdftoolkit.core.exceptions.PDFIOException;
import com.adobe.internal.pdftoolkit.core.exceptions.PDFSecurityException;
import com.adobe.internal.pdftoolkit.core.util.ByteOps;

/**
 * Utilities for parsing literal and hexidecimal strings.
 *
 * TODO: Handle Unicode string encoding as defined in section 3.8.1
 * 		 of the PDF Reference Manual version 1.4.
 */
final class ByteArrayUtility
{
	private ByteArrayUtility(){}
	
	/**
	 * Parses a literal string as defined in section 3.2.3 of the PDF
	 * Reference Manual version 1.4.
	 *
	 * @param inBuf		Buffer to parse
	 * @return Byte array containg the string
	 * @throws PDFCosParseException
	 * @throws PDFIOException
	 * @throws PDFSecurityException
	 */	
	static byte[] readLiteral(CosDocument doc, InputByteStream inBuf, long nextObjPos)
			throws PDFCosParseException, PDFIOException, PDFSecurityException
		{
			try{
				byte[] dest = new byte[16];
				int destIndex = 0;
				int level = 1;
				byte cur;
				do {
					if(doc.getOptions().skipCorruptObjects() && inBuf.getPosition() >= nextObjPos){
						return null;// we have reached the next cos object so seems like some problem with this
									// string so returning null.
					}
					cur = (byte)inBuf.read();
					if (cur == '(') {
						level++;
					} else if (cur == ')') {
						level--;
					} else if (cur == '\\') {
						cur = (byte)inBuf.read();
						if (cur == '\n')
							continue;
						if (cur == '\r') {
							cur = (byte)inBuf.read();
							if (cur != '\n')
								inBuf.unget();
							continue;
						}
						if (cur >= '0' && cur <= '7') {
							cur -= '0';
							byte c = (byte)inBuf.read();
							if (c >= '0' && c <= '7') {
								c -= '0';
								byte d = (byte)inBuf.read();
								if (d >= '0' && d <= '7') {
									d -= '0';
									cur = (byte)(cur * 64 + c * 8 + d);
									inBuf.read();  // to make unget work below.
									// FIXME_IO check for EOF
								} else {
									cur = (byte)(cur * 8 + c);
								}
							}
							inBuf.unget();  // we read one too many bytes in all branches
						} else if (cur == 'n') {
							cur = '\n';
						} else if (cur == 'r') {
							cur = '\r';
						} else if (cur == 't') {
							cur = '\t';
						} else if (cur == 'b') {
							cur = '\b';
						} else if (cur == 'f') {
							cur = '\f';
						} else if (cur == '(') {
							cur = '(';
						} else if (cur == ')') {
							cur = ')';
						} else if (cur == '\\') {
							cur = '\\';
						}
						// TODO - should we check for unexpected characters and throw a parse exception?
					}
					if (destIndex == dest.length) {
						byte[] newDest = new byte[dest.length * 2];
						System.arraycopy(dest, 0, newDest, 0, dest.length);
						dest = newDest;
					}
					dest[destIndex++] = cur;
				} while (level != 0 && !inBuf.eof());
				byte[] result = new byte[destIndex - 1];
				System.arraycopy(dest, 0, result, 0, destIndex - 1);
				return result;
			}catch(IOException e){
				throw new PDFIOException(e);
			}
		}
	
	/**
	 * Parses a literal string as defined in section 3.2.3 of the PDF
	 * Reference Manual version 1.4.
	 *
	 * @param inBuf		Buffer to parse
	 * @throws IOException
	 */
	static void skipLiteral(InputByteStream inBuf)
		throws IOException
	{
		int level = 1;
		byte cur;
		do {
			cur = (byte)inBuf.read();
			if (cur == '(') {
				level++;
			} else if (cur == ')') {
				level--;
			} else if (cur == '\\') {
				cur = (byte)inBuf.read();				
			}
		} while (level != 0 && !inBuf.eof());
	}

	/**
	 * Parses a hexidecimal character string into a byte array. A pair of
	 * hex characters forms the value stored in a single byte of the array.
	 * Hexidecimal strings are defined in section 3.2.3 of the PDF Reference
	 * Manual version 1.4.
	 *
	 * @param buf			Buffer to parse
	 *
	 * @return Data corresponding to the hex string
	 * @throws PDFCosParseException
	 * @throws IOException
	 */
	static byte[] readHex(InputByteStream buf)
		throws PDFCosParseException, IOException, PDFIOException
	{
		// FIXME_IO - check for EOF
		byte b = (byte)buf.read();
		int begin = (int)buf.getPosition() - 1;

		
		// Find the extent of the hex string
		while (b != '>')
		{
			if (!ByteOps.isHexDigit(b))
			{
				if (!ignoreChar(b))
					throw new PDFCosParseException("Expected Hex Digit" + Long.toString(buf.getPosition() - 1));
			}
			b = (byte)buf.read();
			
		}
		int end = (int)buf.getPosition() - 1;

		// Parse the string.
		return hexToByteArray(buf, begin, end);
	}

	/**
	 * Performs the actual parsing of a string of hexadecimal characters.
	 * Each character pair is converted into a single byte value and placed
	 * in the returned byte array.
	 *
	 * @param buf		Buffer to parse
	 * @param begin		Beginning position of the string in the buffer
	 * @param end		Ending position of the string in the buffer
	 *
	 * @return Data corresponding to the hex string.
	 * @throws PDFCosParseException
	 * @throws IOException
	 */
	private static byte[] hexToByteArray(InputByteStream buf, int begin, int end)
		throws PDFCosParseException, IOException
	{
		//Let hextoByteArray ignore all characters that appear in these 3 - whitespace, \r,\n
		byte[] rslt = new byte[(end - begin) / 2];
		int idx = 0, h = 0, l = 0;
				
		buf.seek(begin);
		
		for (int i = begin; i < end; i += 2) 
		{			
			// FIXME_IO check for EOF 
			byte currentChar = (byte)buf.read();
			
			//BugFix - 1267961
			//When the content stream has newlines in Tj operands it would fail.
			//For instance
			//<0108011a010a01100110010100e700e700e800e900ea00eb00ec00ed00ee00ef00f000f100f200f300f400f500f600f700f800f900fa00fb00fc00fd00fe00ff0100010101020103010401050106010701080109010a010b010c010d010e010f0110011101
			//120113011401150116011701180119011a011b011c011d011e>Tj
			//We need to ignore all the whitespaces before the actual hex string starts.
			//This takes care of cases where the hex content looks like
			//"<
			//"0108011a010a01100......"
			
			while (ignoreChar(currentChar))
			{
				currentChar = (byte)buf.read();
				i += 1;
				continue;
			}
			
			h = CosToken.toHexDigit(currentChar);
			l = CosToken.toHexDigit((byte)buf.read());
			rslt[idx++] = (byte)(h * 16 + l);
	
		}
		buf.seek(buf.getPosition() + 1);
		return rslt;
	}
	
	/**
	 * Checks supplied byte against list of characters that
	 * need to be ignored when parsing hex data.
	 * @param byteValue
	 * @return true if byte needs to ignored.
	 */
	private static boolean ignoreChar(byte byteValue)
	{
		//TODO - Just add a list/set of chars instead of these if statements.
		if (byteValue == '\r')
			return true;
		if (byteValue == '\n')
			return true;
		if (ByteOps.isWhitespace(byteValue))
			return true;
		return false;
	}
}
