/* ****************************************************************************
 *
 *	File: CosArray.java
 *
 * ****************************************************************************
 *
 *	ADOBE CONFIDENTIAL
 *	___________________
 *
 *	Copyright 2003-2006 Adobe Systems Incorporated
 *	All Rights Reserved.
 *
 *	NOTICE: All information contained herein is, and remains the property of
 *	Adobe Systems Incorporated and its suppliers, if any. The intellectual
 *	and technical concepts contained herein are proprietary to Adobe Systems
 *	Incorporated and its suppliers and may be covered by U.S. and Foreign
 *	Patents, patents in process, and are protected by trade secret or
 *	copyright law. Dissemination of this information or reproduction of this
 *	material is strictly forbidden unless prior written permission is obtained
 *	from Adobe Systems Incorporated.
 *
 * ***************************************************************************/

package com.adobe.internal.pdftoolkit.core.cos;

import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.ListIterator;
import java.util.NoSuchElementException;

import com.adobe.internal.io.stream.InputByteStream;
import com.adobe.internal.io.stream.OutputByteStream;
import com.adobe.internal.pdftoolkit.core.exceptions.PDFCosParseException;
import com.adobe.internal.pdftoolkit.core.exceptions.PDFException;
import com.adobe.internal.pdftoolkit.core.exceptions.PDFIOException;
import com.adobe.internal.pdftoolkit.core.exceptions.PDFSecurityException;
import com.adobe.internal.pdftoolkit.core.types.ASHexString;
import com.adobe.internal.pdftoolkit.core.types.ASName;
import com.adobe.internal.pdftoolkit.core.types.ASString;

/**
 * Represents a COS array as defined in section 3.2.5 of the PDF Reference
 * Manual version 1.4.
 */
public class CosArray extends CosContainer
{
	protected ArrayList<CosObject> mData;		// List of array elements

	/**
	 * Constructs a COS array object from an ArrayList of CosObjects. Used by
	 * CosToken after it has read the CosObject members of a CosArray. We use
	 * the CosArray add(CosObject) method to insure that direct objects get
	 * the parent object assigned.
	 *
	 * @param doc		Document containing the array
	 * @param data		ArrayList of CosObjects contained in the array
	 * @param info		ObjectInfo for the array
	 */
	protected CosArray(CosDocument doc, ArrayList data, CosObjectInfo info)
	{
		super(doc, info);
		mData = data;

		// If this CosArray is indirect and there are CosObjects in the array, set
		// the parent object of any CosString or CosContainer to this object
		if (!mData.isEmpty() && isIndirect()) {
			Iterator iter = mData.iterator();
			while (iter.hasNext()) {
				CosObject obj = (CosObject)iter.next();
				if (!obj.isIndirect()) {
					if (obj instanceof CosString)
						((CosString)obj).setParentObj(this);
					else if (obj instanceof CosContainer)
						((CosContainer)obj).setParentObj(this);
				}
			}
		}
	}

	/**
	 *
	 * Create a copy of a direct CosArray which is being added to a collection.
	 * This method will recurse through the array elements, copying direct
	 * CosContainers and CosStrings as well. This is to insure that multiple
	 * collections don't hold references to a single direct CosObject.
	 *
	 * If this method is called on an indirect CosArray, it will not return a copy
	 * but rather the indirect object.
	 *
	 * @return CosArray copy of a direct object or the same indirect object
	 * @throws PDFCosParseException
	 * @throws PDFIOException
	 * @throws PDFSecurityException
	 */
	CosArray copy()
	throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		if (isIndirect())
			return this;
		CosArray copy = getDocument().createCosArray(CosObject.DIRECT);
		Iterator iter = iterator();
		while (iter.hasNext()) {
			CosObject obj = (CosObject)iter.next();
			if (!obj.isIndirect()) {
				if (obj instanceof CosArray)
					obj = ((CosArray)obj).copy();
				else if (obj instanceof CosDictionary)
					obj = ((CosDictionary)obj).copy();
				else if (obj instanceof CosString)
					obj = ((CosString)obj).copy();
			}
			copy.add(obj);
		}
		return copy;
	}

	/**
	 *
	 * return the type of this CosObject
	 */
	@Override
	public int getType()
	{
		return t_Array;
	}

	/**
	 *
	 * Returns the number of elements in the array.
	 *
	 * @return Number of elements in the array.
	 */
	public int size()
	{
		return mData.size();
	}

	/**
	 * 
	 * Tests whether the array contains any elements.
	 *
	 * @return true if the array does not contain elements; false otherwise.
	 */
	public boolean isEmpty()
	{
		return mData.isEmpty();
	}

	/**
	 * 
	 * Returns an iterator over the elements of the array.
	 *
	 * @return Iterator
	 */
	public Iterator<CosObject> iterator()
	{
		return new CosArrayListIterator();
	}

	/**
	 * 
	 * Returns a list iterator over the elements of the array.
	 *
	 * @return ListIterator
	 */
	public ListIterator<CosObject> listIterator()
	{
		return new CosArrayListIterator();
	}

	@Override
	public CosContainerValuesIterator getValuesIterator()
	{
		CosObject[] values = new CosObject[size()];
		return new CosArrayValuesIterator(mData.toArray(values));
	}

	
	@Override
	public ArrayList<Object> getValue()
	throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		ArrayList<Object> result = new ArrayList<Object>();
		if (mData != null) {
			Iterator iter = mData.iterator();
			while (iter.hasNext()) {
				CosObject obj = (CosObject)iter.next();
				result.add(obj.getValue());
			}
		}
		result.trimToSize();
		return result;
	}

	/**
	 * Returns true if this array contains the object passed, else false.
	 */
	public boolean contains(CosObject obj)
	{
		return mData.contains(obj);
	}

	/**
	 * Returns the index of name object with the same name as passed.
	 * If not found, then returns -1.
	 * @throws PDFCosParseException
	 * @throws PDFIOException
	 * @throws PDFSecurityException
	 */
	public int findName(ASName name)
	throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		int arrayInd = -1;
		int iter;
		for (iter = 0; iter < size(); iter++) {
			CosObject curElem = get(iter);
			if ((curElem instanceof CosName) && ((CosName) curElem).nameValue().equals(name)) {
				arrayInd = iter;
				break;
			}
		}
		return arrayInd;
	}

	/**
	 * 
	 * Looks for the CosString in the array.
	 *
	 * @throws PDFCosParseException
	 * @throws PDFIOException
	 * @throws PDFSecurityException
	 */
	public int findString(ASString name)
	throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		int arrayInd = -1;
		int iter;
		for (iter = 0; iter < size(); iter++) {
			CosObject curElem = get(iter);
			if ((curElem instanceof CosString) && ((CosString) curElem).stringValue().equals(name)) {
				arrayInd = iter;
				break;
			}
		}
		return arrayInd;
	}

	/**
	 * 
	 * Set this value if this container is direct and is being added to a container.
	 *
	 * @param parent - CosContainer
	 */
	@Override
	void setParentObj(CosContainer parent)
	{
		if (isIndirect())
			return;
		mParentObj = parent;
		Iterator iter = mData.iterator();
		while (iter.hasNext()) {
			CosObject obj = (CosObject)iter.next();
			if (!obj.isIndirect()) {
				if (obj instanceof CosString)
					((CosString)obj).setParentObj(parent);
				else if (obj instanceof CosContainer)
					((CosContainer)obj).setParentObj(parent);
			}
		}
	}

	/**
	 * 
	 * Obtains the specified element from the array. An indirect object
	 * reference is resolved to its referenced object before being returned.
	 *
	 * @param pos		Zero-based index of the desired array element
	 *
	 * @return Object at the specified array position.
	 * @throws PDFCosParseException
	 * @throws PDFIOException
	 * @throws PDFSecurityException
	 */
	public CosObject get(int pos)
	throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		try {
			CosObject result = mData.get(pos);
			if (result instanceof CosObjectRef)
				result = getDocument().resolveReference((CosObjectRef)result);
			return result;
		} catch (IOException e) {
			throw new PDFIOException(e);
		}
	}

	CosObjectRef getRef(int pos)
	{
		CosObject result = mData.get(pos);
		if (result instanceof CosObjectRef)
			return (CosObjectRef)result;
		return null;
	}

	/**
	 * 
	 * Returns the type of the value at the given position
	 * @param pos
	 */
	public int getType(int pos)
	{
		CosObject val = mData.get(pos);
		if (val == null)
			return t_Null;	// Treat null as CosNull
		return val.getType();
	}

	/**
	 * It assumes that the object at index "pos" is of type {@link CosName}
	 * and returns it's value.
	 * @throws PDFCosParseException
	 * @throws PDFIOException
	 * @throws PDFSecurityException
	 */
	public ASName getName(int pos)
	throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		CosObject val = get(pos);
		return val.nameValue();
	}

	/**
	 * It assumes that the object at index "pos" is of type {@link CosBoolean}
	 * and returns it's value.
	 * @throws PDFCosParseException
	 * @throws PDFIOException
	 * @throws PDFSecurityException
	 */
	public boolean getBoolean(int pos)
	throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		CosObject val = get(pos);
		return val.booleanValue();
	}

	/**
	 * It assumes that the object at index "pos" is of type {@link CosNumeric}
	 * and returns it's value.
	 * @throws PDFCosParseException
	 * @throws PDFIOException
	 * @throws PDFSecurityException
	 */
	public double getDouble(int pos)
	throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		CosObject val = get(pos);
		return val.doubleValue();
	}

	/**
	 * 
	 * getArrayDouble() presumes that the CosArray contains numeric values only
	 * and delivers an array of double values.
	 * @throws PDFCosParseException
	 * @throws PDFIOException
	 * @throws PDFSecurityException
	 */
	public double[] getArrayDouble()
	throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		double[] values = new double[size()];
		for (int i = 0; i < size(); i++)
		{
			values[i] = getDouble(i);
		}
		return values;
	}

	/**
	 * 
	 * getArrayDouble() presumes that the CosArray contains numeric values only
	 * and delivers an array of double values.
	 * @param start the index to start conversion from
	 * @param length the number of indices to convert
	 * @throws PDFCosParseException
	 * @throws PDFIOException
	 * @throws PDFSecurityException
	 */
	public double[] getArrayDouble(int start, int length)
	throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		double[] values = new double[length];
		for (int i = 0; i <  length; i++)
		{
			values[i] = getDouble(i+1);
		}
		return values;
	}

	/**
	 * It assumes that the object at index "pos" is of type {@link CosNumeric}
	 * and returns it's value.
	 * @throws PDFCosParseException
	 * @throws PDFIOException
	 * @throws PDFSecurityException
	 */
	public long getLong(int pos)
	throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		CosObject val = get(pos);
		return val.longValue();
	}

	/**
	 * It assumes that the object at index "pos" is of type {@link CosNumeric}
	 * and returns it's value.
	 * @throws PDFCosParseException
	 * @throws PDFIOException
	 * @throws PDFSecurityException
	 */
	public int getInt(int pos)
	throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		CosObject val = get(pos);
		return val.intValue();
	}

	/**
	 * 
	 * getArrayInt() presumes that the CosArray contains numeric values only
	 * and delivers an array of integer values.
	 * @throws PDFCosParseException
	 * @throws PDFIOException
	 * @throws PDFSecurityException
	 */
	public long[] getArrayLong()
	throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		long[] values = new long[size()];
		for (int i = 0; i < size(); i++)
			values[i] = getLong(i);
		return values;
	}

	/**
	 * getArrayInt() presumes that the CosArray contains numeric values only
	 * and delivers an array of integer values.
	 * @throws PDFCosParseException
	 * @throws PDFIOException
	 * @throws PDFSecurityException
	 */
	public int[] getArrayInt()
	throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		int[] values = new int[size()];
		for (int i = 0; i < size(); i++)
			values[i] = getInt(i);
		return values;
	}

	/**
	 * getArrayBytes() presumes that the CosArray contains string values only
	 * and delivers an array of array of byte values.
	 * @throws PDFCosParseException
	 * @throws PDFIOException
	 * @throws PDFSecurityException
	 */
	public byte[][] getArrayBytes()
	throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		int size = size();
		byte[][] certs = new byte[size][];
		for (int i = 0; i < size; i++)
			certs[i] = getString(i).getBytes();
		return certs;
	}

	/**
	 * It assumes that the object at index "pos" is of type {@link CosString}
	 * and returns it's value.
	 * @throws PDFCosParseException
	 * @throws PDFIOException
	 * @throws PDFSecurityException
	 */
	public ASString getString(int pos)
	throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		CosObject val = get(pos);
		return val.stringValue();
	}

	/**
	 * It assumes that the object at index "pos" is of type {@link CosString}
	 * and returns it's value.
	 * @throws PDFCosParseException
	 * @throws PDFIOException
	 * @throws PDFSecurityException
	 */
	public ASHexString getHexString(int pos)
	throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		CosObject val = get(pos);
		return val.hexStringValue();
	}

	/**
	 * It assumes that the object at index "pos" is of type {@link CosString}
	 * and returns it's value.
	 * @throws PDFCosParseException
	 * @throws PDFIOException
	 * @throws PDFSecurityException
	 */
	public String getText(int pos)
	throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		CosObject val = get(pos);
		return val.textValue();
	}

	/**
	 * It assumes that the object at index "pos" is of type {@link CosString}
	 * and returns it's value.
	 * @throws PDFCosParseException
	 * @throws PDFIOException
	 * @throws PDFSecurityException
	 */
	public CosString getCosString(int pos)
	throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		return (CosString) get(pos);
	}

	/**
	 * It assumes that the object at index "pos" is of type {@link CosArray}
	 * and returns it's value.
	 * @throws PDFCosParseException
	 * @throws PDFIOException
	 * @throws PDFSecurityException
	 */
	public CosArray getCosArray(int pos)
	throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		return (CosArray) get(pos);
	}

	/**
	 * It assumes that the object at index "pos" is of type {@link CosDictionary}
	 * and returns it's value.
	 * @throws PDFCosParseException
	 * @throws PDFIOException
	 * @throws PDFSecurityException
	 */
	public CosDictionary getCosDictionary(int pos)
	throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		return (CosDictionary)get(pos);
	}

	/**
	 * It assumes that the object at index "pos" is of type {@link CosStream}
	 * and returns it's value.
	 * @throws PDFCosParseException
	 * @throws PDFIOException
	 * @throws PDFSecurityException
	 */
	public CosStream getCosStream(int pos)
	throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		return (CosStream) get(pos);
	}

	/**
	 *
	 * This gets a <i>slice</i> of the underlying <code>InputByteStream</code>. It is the repsonsibility
	 * of the caller to close the <code>InputByteStream</code> when they are finished with it.
	 * @throws PDFCosParseException
	 * @throws PDFIOException
	 * @throws PDFSecurityException
	 */
	public InputByteStream getStream(int pos)
	throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		CosObject val = get(pos);
		return ((CosStream)val).getStreamDecoded();
	}

	/**
	 *
	 * Removes all elements from the CosArray.
	 * @throws PDFCosParseException
	 * @throws PDFIOException
	 * @throws PDFSecurityException
	 */
	public void clear()
	throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		try {
			if (mData.isEmpty())
				return;
			mData.clear();
			if (isIndirect())
				getInfo().markDirty();
			else {
				if (mParentObj != null)
					mParentObj.getInfo().markDirty();
			}
		} catch (IOException e) {
			throw new PDFIOException(e);
		}
	}

	/**
	 *
	 * Removes the element at the specified position in this list.
	 * Shifts any subsequent elements to the left (subtracts one from their
	 * indices).
	 *
	 * @param index the index of the element to removed.
	 * @return the element that was removed from the list.
	 * @throws PDFCosParseException
	 * @throws PDFIOException
	 * @throws PDFSecurityException
	 * @throws IndexOutOfBoundsException if index out of range
	 */
	public CosObject remove(int index)
	throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		try {
			CosObject removed = mData.remove(index);
			// If the element is a reference, resolve it before comparing
			if (removed instanceof CosObjectRef)
				removed = getDocument().resolveReference((CosObjectRef)removed);
			if (isIndirect())
				getInfo().markDirty();
			else {
				if (mParentObj != null)
					mParentObj.getInfo().markDirty();
			}
			return removed;
		} catch (IOException e) {
			throw new PDFIOException(e);
		}
	}

	/**
	 *
	 * Removes all instances of the specified object from the array.
	 *
	 * @param cosObject		Object to be removed from the array
	 * @return true if object successfully removed.
	 * @throws PDFCosParseException
	 * @throws PDFIOException
	 * @throws PDFSecurityException
	 */
	public boolean remove(CosObject cosObject)
	throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		try {
			boolean removed = false;
			for (int i = 0; i < size(); i++) {
				CosObject element = get(i);
				// If the element is a reference, resolve it before comparing
				if (element instanceof CosObjectRef)
					element = getDocument().resolveReference((CosObjectRef)element);
				if (cosObject == element) {
					remove(i);
					removed = true;
				}
			}
			return removed;
		} catch (IOException e) {
			throw new PDFIOException(e);
		}
	}

	/**
	 *
	 * Appends the specified CosObject to the end of this list. If the
	 * object is indirect, it adds the object reference. The CosArray
	 * is marked dirty after the operation.
	 *
	 * @param cosObject to be added to the list
	 * @return <tt>true</tt> (as per the general contract of Collection.add).
	 * @throws PDFIOException
	 * @throws PDFCosParseException
	 * @throws PDFSecurityException
	 */
	public boolean add(CosObject cosObject)
	throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		try {
			add(size(), cosObject, false);
		} catch (IOException e) {
			throw new PDFIOException(e);
		}
		return true;
	}

	/**
	 * Creates an appropriate cos object and adds it to array.
	 * @throws PDFCosParseException
	 * @throws PDFIOException
	 * @throws PDFSecurityException
	 */
	public boolean addName(ASName value)
	throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		return add(getDocument().createCosName(value));
	}

	/**
	 * Creates an appropriate cos object and adds it to array.
	 * @throws PDFCosParseException
	 * @throws PDFIOException
	 * @throws PDFSecurityException
	 */
	public boolean addBoolean(boolean value)
	throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		return add(getDocument().createCosBoolean(value));
	}

	/**
	 * Creates an appropriate cos object and adds it to array.
	 * @throws PDFCosParseException
	 * @throws PDFIOException
	 * @throws PDFSecurityException
	 */
	public boolean addDouble(double value)
	throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		return add(getDocument().createCosNumeric(value));
	}

	/**
	 * Creates an appropriate cos object and adds it to array.
	 * @throws PDFCosParseException
	 * @throws PDFIOException
	 * @throws PDFSecurityException
	 */
	public boolean addLong(long value)
	throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		return add(getDocument().createCosNumeric(value));
	}

	/**
	 * Creates an appropriate cos object and adds it to array.
	 * @throws PDFCosParseException
	 * @throws PDFIOException
	 * @throws PDFSecurityException
	 */
	public boolean addInt(int value)
	throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		return add(getDocument().createCosNumeric(value));
	}

	/**
	 * Creates an appropriate cos object and adds it to array.
	 * @throws PDFCosParseException
	 * @throws PDFIOException
	 * @throws PDFSecurityException
	 */
	public boolean addText(String value)
	throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		CosString newString = getDocument().createCosString(value);
		newString.setParentObj(isIndirect() ? this : mParentObj);
		return add(newString);
	}

	/**
	 * Creates an appropriate cos object and adds it to array.
	 * @throws PDFCosParseException
	 * @throws PDFIOException
	 * @throws PDFSecurityException
	 */
	public boolean addString(ASString value)
	throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		CosString newString = getDocument().createCosString(value);
		newString.setParentObj(isIndirect() ? this : mParentObj);
		return add(newString);
	}

	/**
	 * 
	 * Inserts the specified element at the specified position in this
	 * list. Shifts the element currently at that position (if any) and
	 * any subsequent elements to the right (adds one to their indices).
	 *
	 * @param pos index at which the specified element is to be inserted.
	 * @param cosObject element to be inserted.
	 * @throws PDFCosParseException
	 * @throws PDFIOException
	 * @throws PDFSecurityException
	 * @throws IndexOutOfBoundsException if index is out of range
	 */
	public void add(int pos, CosObject cosObject)
	throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		try {
			add(pos, cosObject, false);
		} catch (IOException e) {
			throw new PDFIOException(e);
		}
	}

	private void add(int pos, CosObject cosObject, boolean overWrite)
	throws PDFCosParseException, PDFIOException, PDFSecurityException, IOException
	{
		if (getDocument() != cosObject.getDocument())
			throw new PDFCosParseException("Object and container must be in same document");
		if (cosObject.isIndirect()) {
			if (!(cosObject instanceof CosObjectRef))
				cosObject = getDocument().getObjectRef(cosObject.getInfo());
			if (overWrite)
				mData.set(pos, cosObject);
			else
				mData.add(pos, cosObject);
		} else {
			// We're adding a direct object; if it's a CosContainer or CosString set parent object
			CosContainer parentObj = isIndirect() ? this : mParentObj;
			CosContainer oldParentObj;
			if (cosObject instanceof CosContainer) {
				// We're adding a direct CosContainer
				// If this container already belongs to another collection,
				// make a copy and add the copy
				oldParentObj = ((CosContainer)cosObject).getParentObj();
				if (oldParentObj != null) {
					if (cosObject instanceof CosArray)
						cosObject = ((CosArray)cosObject).copy();
					else
						cosObject = ((CosDictionary)cosObject).copy();
				}
				((CosContainer)cosObject).setParentObj(parentObj);
			} else if (cosObject instanceof CosString) {
				// We're adding a direct CosString
				oldParentObj = ((CosString)cosObject).getParentObj();
				if (oldParentObj != null)
					cosObject = ((CosString)cosObject).copy();
				((CosString)cosObject).setParentObj(parentObj);
			}
			if (overWrite)
				mData.set(pos, cosObject);
			else
				mData.add(pos, cosObject);
			if (isIndirect())
				getInfo().markDirty();
			else {
				if (mParentObj != null)
					mParentObj.getInfo().markDirty();
			}
		}
		if (isIndirect())
			getInfo().markDirty();
		else {
			if (mParentObj != null)
				mParentObj.getInfo().markDirty();
		}
	}

	/**
	 * Creates an appropriate cos object and adds it to array at specified position.
	 * @throws PDFCosParseException
	 * @throws PDFIOException
	 * @throws PDFSecurityException
	 */
	public void addName(int pos, ASName value)
	throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		add(pos, getDocument().createCosName(value));
	}

	/**
	 * Creates an appropriate cos object and adds it to array at specified position.
	 * @throws PDFCosParseException
	 * @throws PDFIOException
	 * @throws PDFSecurityException
	 */
	public void addBoolean(int pos, boolean value)
	throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		add(pos, getDocument().createCosBoolean(value));
	}

	/**
	 * Creates an appropriate cos object and adds it to array at specified position.
	 * @throws PDFCosParseException
	 * @throws PDFIOException
	 * @throws PDFSecurityException
	 */
	public void addDouble(int pos, double value)
	throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		add(pos, getDocument().createCosNumeric(value));
	}

	/**
	 * Creates an appropriate cos object and adds it to array at specified position.
	 * @throws PDFCosParseException
	 * @throws PDFIOException
	 * @throws PDFSecurityException
	 */
	public void addLong(int pos, long value)
	throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		add(pos, getDocument().createCosNumeric(value));
	}

	/**
	 * Creates an appropriate cos object and adds it to array at specified position.
	 * @throws PDFCosParseException
	 * @throws PDFIOException
	 * @throws PDFSecurityException
	 */
	public void addInt(int pos, int value)
	throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		add(pos, getDocument().createCosNumeric(value));
	}

	/**
	 * Creates an appropriate cos object and adds it to array at specified position.
	 * @throws PDFCosParseException
	 * @throws PDFIOException
	 * @throws PDFSecurityException
	 */
	public void addText(int pos, String value)
	throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		CosString newString = getDocument().createCosString(value);
		newString.setParentObj(isIndirect() ? this : mParentObj);
		add(pos, newString);
	}

	/**
	 * Creates an appropriate cos object and adds it to array at specified position.
	 * @throws PDFCosParseException
	 * @throws PDFIOException
	 * @throws PDFSecurityException
	 */
	public void addString(int pos, ASString value)
	throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		CosString newString = getDocument().createCosString(value);
		newString.setParentObj(isIndirect() ? this : mParentObj);
		add(pos, newString);
	}

	/**
	 * 
	 * Adds the CosObject at the specified position in this list.
	 *
	 * @param pos index of element being added.
	 * @param cosObject element to be added at the specified position.
	 * @return true
	 * @throws PDFCosParseException
	 * @throws PDFIOException
	 * @throws PDFSecurityException
	 * @throws IndexOutOfBoundsException if index out of range
	 */
	public boolean set(int pos, CosObject cosObject)
	throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		try {
			add(pos, cosObject, true);
		} catch (IOException e) {
			throw new PDFIOException(e);
		}
		return true;
	}

	/**
	 * 
	 * Adds the CosObject at the specified position in this list. If
	 * the list is too short, it is first extended with 'null'.
	 *
	 * @param pos index of element being added.
	 * @param cosObject element to be added at the specified position.
	 * @throws PDFCosParseException
	 * @throws PDFIOException
	 * @throws PDFSecurityException
	 */
	public void extendIfNecessaryAndSet (int pos, CosObject cosObject)
	throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		while (mData.size () <= pos) {
			mData.add (null); }
		set (pos, cosObject);
	}

	/**
	 * Creates an appropriate cos object and sets it to array at specified position.
	 * @throws PDFCosParseException
	 * @throws PDFIOException
	 * @throws PDFSecurityException
	 */
	public boolean setName(int pos, ASName value)
	throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		return set(pos, getDocument().createCosName(value));
	}

	/**
	 * Creates an appropriate cos object and sets it to array at specified position.
	 * @throws PDFCosParseException
	 * @throws PDFIOException
	 * @throws PDFSecurityException
	 */
	public boolean setName(int pos, String value)
	throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		return set(pos, getDocument().createCosName(ASName.create(value)));
	}

	/**
	 * Creates an appropriate cos object and sets it to array at specified position.
	 * @throws PDFCosParseException
	 * @throws PDFIOException
	 * @throws PDFSecurityException
	 */
	public boolean setBoolean(int pos, boolean value)
	throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		return set(pos, getDocument().createCosBoolean(value));
	}

	/**
	 * Creates an appropriate cos object and sets it to array at specified position.
	 * @throws PDFCosParseException
	 * @throws PDFIOException
	 * @throws PDFSecurityException
	 */
	public boolean setDouble(int pos, double value)
	throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		return set(pos, getDocument().createCosNumeric(value));
	}

	/**
	 * Creates an appropriate cos object and sets it to array at specified position.
	 * @throws PDFCosParseException
	 * @throws PDFIOException
	 * @throws PDFSecurityException
	 */
	public boolean setInt(int pos, int value)
	throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		return set(pos, getDocument().createCosNumeric(value));
	}

	/**
	 * Creates an appropriate cos object and sets it to array at specified position.
	 * @throws PDFCosParseException
	 * @throws PDFIOException
	 * @throws PDFSecurityException
	 */
	public boolean setLong(int pos, long value)
	throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		return set(pos, getDocument().createCosNumeric(value));
	}

	/**
	 * Creates an appropriate cos object and sets it to array at specified position.
	 * @throws PDFCosParseException
	 * @throws PDFIOException
	 * @throws PDFSecurityException
	 */
	public boolean setString(int pos, ASString value)
	throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		CosString newString = getDocument().createCosString(value);
		newString.setParentObj(isIndirect() ? this : mParentObj);
		return set(pos, newString);
	}

	/**
	 * Creates an appropriate cos object and sets it to array at specified position.
	 * @throws PDFCosParseException
	 * @throws PDFIOException
	 * @throws PDFSecurityException
	 */
	public boolean setText(int pos, String value)
	throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		CosString newString = getDocument().createCosString(value);
		newString.setParentObj(isIndirect() ? this : mParentObj);
		return set(pos, newString);
	}


	/**
	 *  
	 *  This method checks the inside data of CosArray and returns true if they are
	 *  clone of each other. Returns false if passed CosObject is not an instance of
	 *  CosArray. 
	 *  This maintains a continuously growing list of indirect CosObject pairs, which have already been compared, to get rid of infinite recursion.
	 *  This method can take some time if called on large Cos Objects.
	 *  @param value
	 *  @return boolean
	 */
	@Override
	public boolean equals(CosObject value){
		HashMap<Integer, HashSet<Integer>> alreadyComparedCosObjectPairsList = new HashMap<Integer, HashSet<Integer>>();
		return this.safeEquals(value, alreadyComparedCosObjectPairsList);
	}
	
	/**
	 *  
	 *  This method will be used for internal communication only after CosObject.equals() is called.
	 * @param value
	 * @param alreadyComparedCosObjectPairsList which maintains the indirect CosObject pairs which have already been compared.
	 * @return boolean
	 */
	@Override
	boolean safeEquals(CosObject value, HashMap<Integer, HashSet<Integer>> alreadyComparedCosObjectPairsList){
		if(!(value instanceof CosArray) || value.getDocument() != this.getDocument())
			return false;
		if(value == this)
			return true;
		
		////if pair successfully added then do further comparisons otherwise return true because this pair is already compared.
		if(!addCosObjectPair(value.getObjNum(), alreadyComparedCosObjectPairsList))
			return true;
		
		CosArray valueCosArray = (CosArray) value;
		if(valueCosArray.size() != this.size())
			return false;
		ArrayList<Integer> compoundEntriesIndexList = new ArrayList<Integer>();
		try{
		//// comparing scalar cos objects keeping keys corresponding to compound cos objects in list which will be compared later.
			CosObject valueAtIndex = null;
			for(int i=0; i<valueCosArray.size(); i++){
				valueAtIndex = this.get(i);
				if(valueAtIndex instanceof CosScalar){
					if(!(valueAtIndex.equals(valueCosArray.get(i))))
						return false;
				}else{
					compoundEntriesIndexList.add(i);
				}
			}
			///now comparing compound cos objects.
			for(int i=0; i<compoundEntriesIndexList.size(); i++){
				if(!((CosContainer)(this.get(compoundEntriesIndexList.get(i)))).safeEquals
						(valueCosArray.get(compoundEntriesIndexList.get(i)), alreadyComparedCosObjectPairsList))
					return false;
			}
		}catch(Exception e){
			throw new RuntimeException("problem occured while equating " + value.getObjNum() + " with " + this.getObjNum(), e);
		}
		return true;
	}
	/**
	 * 
	 * Set the specified encryption state for all strings and streams in an array recursively
	 */
	public void setEncryptionState(boolean state)
	throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		try {
			Iterator iter = iterator();
			while (iter.hasNext()){
				CosObject value = (CosObject) iter.next();
				if (value.getType() == CosObject.t_String){
					((CosString) value).setIsEncrypted(state);
					((CosString) value).setToEncrypt(state);
				}
				else if (value.getType() == CosObject.t_Stream){
					((CosStream) value).setIsEncrypted(state);
					((CosStream) value).setToEncrypt(state);
				}
				else if (value.getType() == CosObject.t_Dictionary){
					((CosDictionary) value).setEncryptionState(state);
				}
				else if (value.getType() == CosObject.t_Array){
					((CosArray) value).setEncryptionState(state);
				}
			}
		} catch (IOException e) {
			throw new PDFIOException(e);
		}
	}

	/**
	 * Returns a new array containing all the values before "pos" index in
	 * current array. 
	 * @throws PDFCosParseException
	 * @throws PDFIOException
	 * @throws PDFSecurityException
	 */
	public CosArray splitBefore(int pos)
	throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		CosArray newArray = null;
		if (isIndirect())
			newArray = getDocument().createCosArray(CosObject.INDIRECT);
		else {
			newArray = getDocument().createCosArray(CosObject.DIRECT);
			newArray.setParentObj(mParentObj);
		}
		int siz = size();
		for (int i = pos; i < siz; i++)
			newArray.add(get(i));
		for (int i = pos; i < siz; i++)
			remove(pos);
		return newArray;
	}

		
	@Override
	void writeOut(OutputByteStream outStream, boolean inString, boolean inDebugMode)
	throws PDFCosParseException, IOException, PDFSecurityException, PDFIOException
	{
		outStream.write('[');
		Iterator iter = mData.iterator();
		boolean first = true;
		while (iter.hasNext()) {
			CosObject item = (CosObject)iter.next();
			if (!first && (item.isIndirect() || item instanceof CosBoolean || item instanceof CosNumeric || item instanceof CosNull))
				outStream.write(' ');
			item.writeOut(outStream, inString, inDebugMode);
			first = false;
		}
		outStream.write(']');
	}

	public class CosArrayListIterator implements ListIterator<CosObject> {
		private int nextIndx;
		
		private boolean nextCalled = false;
		private boolean previousCalled = false;

		CosArrayListIterator()
		{
			nextIndx = 0;
		}

		/* (non-Javadoc)
		 * @see java.util.ListIterator#nextIndex()
		 */
		public int nextIndex()
		{
			return nextIndx;
		}

		/* (non-Javadoc)
		 * @see java.util.ListIterator#previousIndex()
		 */
		public int previousIndex()
		{
			return nextIndx - 1;
		}

		/* (non-Javadoc)
		 * @see java.util.ListIterator#remove()
		 */
		public void remove()
		{
			if (!(nextCalled || previousCalled))
				throw new IllegalStateException(
						"neither next nor previous have been called, or remove or add have been called after the last call to * next or previous.");
			
			try {
				if(nextCalled)
					CosArray.this.remove(--nextIndx);
				else
					CosArray.this.remove(nextIndx);
				
				nextCalled = false;
				previousCalled = false;
			} catch (PDFException e) {
				IllegalArgumentException newException = new IllegalArgumentException();
				newException.initCause(e);
				throw newException;
			}
		}

		/* (non-Javadoc)
		 * @see java.util.ListIterator#hasNext()
		 */
		public boolean hasNext()
		{
			return nextIndx < size();
		}

		/* (non-Javadoc)
		 * @see java.util.ListIterator#hasPrevious()
		 */
		public boolean hasPrevious()
		{
			return nextIndx > 0;
		}

		/* (non-Javadoc)
		 * @see java.util.ListIterator#next()
		 */
		public CosObject next()
		{
			if (nextIndx >= size())
				throw new NoSuchElementException();
			try {
				nextCalled = true;
				previousCalled = false;
				return CosArray.this.get(nextIndx++);
			} catch (PDFException e) {
				NoSuchElementException newException = new NoSuchElementException();
				newException.initCause(e);
				throw newException;
			}
		}

		/* (non-Javadoc)
		 * @see java.util.ListIterator#previous()
		 */
		public CosObject previous()
		{
			try {
				previousCalled = true;
				nextCalled = false;
				return CosArray.this.get(--nextIndx);
			} catch (PDFException e) {
				NoSuchElementException newException = new NoSuchElementException();
				newException.initCause(e);
				throw newException;
			}
		}

		/* (non-Javadoc)
		 * @see java.util.ListIterator#add(java.lang.Object)
		 */
		public void add(CosObject o)
		{
			try {
				CosArray.this.add(nextIndx++, o);
				nextCalled = false;
				previousCalled = false;
			} catch (PDFException e) {
				IllegalArgumentException newException = new IllegalArgumentException();
				newException.initCause(e);
				throw newException;
			}
		}

		/* (non-Javadoc)
		 * @see java.util.ListIterator#set(java.lang.Object)
		 */
		public void set(CosObject o)
		{
			if (!(nextCalled || previousCalled))
				throw new IllegalStateException(
						"neither next nor previous have been called, or remove or add have been called after the last call to next or previous");
			try {
				if(nextCalled)
					CosArray.this.set(nextIndx - 1, o);
				else
					CosArray.this.set(nextIndx, o);
			} catch (PDFException e) {
				IllegalArgumentException newException = new IllegalArgumentException();
				newException.initCause(e);
				throw newException;
			}
		}
	}
	
	/**
	 * 
	 * getArrayName() presumes that the CosArray contains ASName values only
	 * and delivers an array of ASName values.
	 * @throws PDFCosParseException
	 * @throws PDFIOException
	 * @throws PDFSecurityException
	 */
	public ASName[] getArrayName()
	throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		ASName[] values = new ASName[size()];
		for (int i = 0; i < size(); i++){
			values[i] = getName(i);
		}
		return values;
	}

}
