/* ****************************************************************************
 *
 *	File: CosEncryption.java
 *
 * ****************************************************************************
 *
 *	ADOBE CONFIDENTIAL
 *	___________________
 *
 *	Copyright 2003-2006 Adobe Systems Incorporated
 *	All Rights Reserved.
 *
 *	NOTICE: All information contained herein is, and remains the property of
 *	Adobe Systems Incorporated and its suppliers, if any. The intellectual
 *	and technical concepts contained herein are proprietary to Adobe Systems
 *	Incorporated and its suppliers and may be covered by U.S. and Foreign
 *	Patents, patents in process, and are protected by trade secret or
 *	copyright law. Dissemination of this information or reproduction of this
 *	material is strictly forbidden unless prior written permission is obtained
 *	from Adobe Systems Incorporated.
 *
 * ***************************************************************************/

package com.adobe.internal.pdftoolkit.core.cos;

import java.io.IOException;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

import com.adobe.internal.io.ByteWriterFactory.Fixed;
import com.adobe.internal.io.stream.InputByteStream;
import com.adobe.internal.io.stream.OutputByteStream;
import com.adobe.internal.pdftoolkit.core.exceptions.PDFCosParseException;
import com.adobe.internal.pdftoolkit.core.exceptions.PDFIOException;
import com.adobe.internal.pdftoolkit.core.exceptions.PDFSecurityAuthorizationException;
import com.adobe.internal.pdftoolkit.core.exceptions.PDFSecurityConfigurationException;
import com.adobe.internal.pdftoolkit.core.exceptions.PDFSecurityException;
import com.adobe.internal.pdftoolkit.core.exceptions.PDFUnsupportedFeatureException;
import com.adobe.internal.pdftoolkit.core.securityframework.DecryptedState;
import com.adobe.internal.pdftoolkit.core.securityframework.EncryptionHandler;
import com.adobe.internal.pdftoolkit.core.securityframework.EncryptionHandlerState;
import com.adobe.internal.pdftoolkit.core.securityframework.SecurityHandler;
import com.adobe.internal.pdftoolkit.core.securityframework.SecurityManager;
import com.adobe.internal.pdftoolkit.core.securityframework.impl.EnableEncryption;
import com.adobe.internal.pdftoolkit.core.types.ASName;

/**
 * 
 * Performs all encryption-related operations on a Cos Document
 *
 * @author itenenbo
 *
 */
public class CosEncryption
{
	private EnableEncryption mEncryptionHandle = null;	// Encryption support
	private CosDictionary mDecryptDictionary = null;
	private CosDocument mDoc;
	private SecurityManager mEncryptionManager = null;
	private Map mEncryptionMap = null;
	private Map mDecryptionStateHandlers = null;
	private Map mEncryptionStateHandlers = null;
	private boolean mNeedsDecryption;			// The encryption state of the original document [used for decryption]
	private boolean mNeedsEncryption = false;		// The encryption state of the document at Save [used for encryption]
	private boolean mDoDecryption;
	private boolean mIsNewEncryption = false;
	private boolean mInSave = false;

	private static final String ENCRYPT_IMPL = "com.adobe.internal.pdftoolkit.core.encryption.EncryptionImpl";
	private static final String ENCRYPT_METADATA = ASName.k_EncryptMetadata.asString(true);
	private static final String STMF = ASName.k_StmF.asString(true);
	private static final String CF = ASName.k_CF.asString(true);
	private static final String V = ASName.k_V.asString(true);

	public CosEncryption(CosDocument doc)
	{
		mDoc = doc;
		if (doc == null) {
			// Do not decrypt strings and streams in empty document
			// because it is used for internal purposes.
			mNeedsDecryption = false;
			mDoDecryption = false;
		}
	}

	public void setupDecryption()
		throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		//SLG - No XRef table if doc is being created so you need null check
		if (mDoc != null) {
			mNeedsDecryption = mDoc.isEncrypted(); // Checks to see if /Encrypt in trailer
			if (mNeedsDecryption) {
				CosDictionary trailer = mDoc.getTrailer();
				if (trailer != null)
					mDecryptDictionary = trailer.containsKey(ASName.k_Encrypt) ?
						trailer.getCosDictionary(ASName.k_Encrypt) : null;
			}
		}
		mDoDecryption = true; // Always true but only used if there's a dictionary
	}

	public boolean needsDecryption()
	{
		return mNeedsDecryption;
	}

	public boolean setNeedsDecryption(boolean decrypt)
	{
		boolean old = mNeedsDecryption;
		mNeedsDecryption = decrypt;
		return old;
	}

	public boolean needsEncryption()
	{
		return mNeedsEncryption;
	}

	public CosDictionary getEncryption()
	{
		return mDecryptDictionary;
	}

	public Map getDecryptionMap()
		throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		return (Map)(mDecryptDictionary.getValue());
	}

	public Map getEncryptionMap()
		throws PDFSecurityAuthorizationException
	{
		return getEncryptionImpl().getEncryptionParameters();
	}

	/**
	 * Return the default decryption handler
	 * @throws PDFCosParseException
	 * @throws PDFSecurityException
	 */
	public SecurityHandler getDefaultDecryptionHandler()
		throws PDFCosParseException, PDFSecurityException
	{
		if (mEncryptionHandle == null)
			throw new PDFSecurityConfigurationException("Security Manager is not set");
		return getDefaultHandler(mEncryptionHandle.getDecryptionParameters(), mEncryptionHandle.getDecryptionSecurityManager());
	}

	private SecurityHandler getDefaultHandler(Map encryptParams, SecurityManager secMgr)
		throws PDFCosParseException, PDFSecurityException
	{
		if ((mEncryptionHandle == null) || (secMgr == null))
			throw new PDFSecurityConfigurationException("Security Manager is not set");
		SecurityHandler handler = null;
		String handlerName = (String) encryptParams.get("Filter");
		if (handlerName != null)
			handler = secMgr.getSecurityHandler(handlerName, encryptParams);
		if (handler != null)
			return handler;
		throw new PDFCosParseException("Invalid /Encrypt dictionary");
	}

	/**
	 * Return the encryption implementation
	 * @throws PDFUnsupportedFeatureException if there is no encryption implementation
	 * This represents a "can't happen" situation
	 */
	public EnableEncryption getEncryptionImpl()
		throws PDFUnsupportedFeatureException
	{
		if (mEncryptionHandle == null) {
			try {
				mEncryptionHandle = (EnableEncryption)Class.forName(ENCRYPT_IMPL).getConstructor().newInstance();
			} catch (Exception exp) {
				throw new PDFUnsupportedFeatureException("Encryption is not supported", exp);
			}
		}
		return mEncryptionHandle;
	}

	private boolean shouldEncryptMetadata1(CosStream stream, ASName cryptFilter, boolean encrypting)
		throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		Boolean toEncrypt = null;
		if (stream.containsKey(ASName.k_Type) && (stream.getName(ASName.k_Type).equals(ASName.k_Metadata))) {
			Map encryptParams = encrypting ? getEncryptionMap() : getDecryptionMap();
			toEncrypt = (Boolean) encryptParams.get(ENCRYPT_METADATA);
			if (toEncrypt == null) {
				Map cfDict = (Map) encryptParams.get(ASName.k_CF.asString(true));
				if (cfDict != null) {
					String cryptName = cryptFilter == null ?
						(String) encryptParams.get(STMF) : cryptFilter.asString(true);
					Map cryptDict = (Map) cfDict.get(cryptName);
					if (cryptDict != null) {
						toEncrypt = (Boolean) cryptDict.get(ENCRYPT_METADATA);
					}
				}
			}
		}
		return (toEncrypt == null) || toEncrypt.booleanValue();
	}

	private boolean shouldEncryptMetadata(ASName type, ASName cryptFilter, boolean encrypting)
		throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		Boolean toEncrypt = null;
		if (ASName.k_Metadata.equals(type)) {
			Map encryptParams = encrypting ? getEncryptionMap() : getDecryptionMap();
			toEncrypt = (Boolean) encryptParams.get(ENCRYPT_METADATA);
			if (toEncrypt == null) {
				Map cfDict = (Map) encryptParams.get(ASName.k_CF.asString(true));
				if (cfDict != null) {
					String cryptName = cryptFilter == null ?
						(String) encryptParams.get(STMF) : cryptFilter.asString(true);
					Map cryptDict = (Map) cfDict.get(cryptName);
					if (cryptDict != null) {
						toEncrypt = (Boolean) cryptDict.get(ENCRYPT_METADATA);
					}
				}
			}
		}
		return (toEncrypt == null) || toEncrypt.booleanValue();
	}

	boolean shouldDecryptOrEncrypt1(CosStream stream, ASName cryptFilter, boolean encrypting)
		throws PDFCosParseException, PDFIOException, PDFSecurityException, IOException
	{
		boolean encryptMetadata = shouldEncryptMetadata1(stream, cryptFilter, encrypting);
		if (!mInSave)
			return encryptMetadata;
		if (!encryptMetadata)
			return false;
		if (mIsNewEncryption)
			return true;
		if (stream.isDirty())
			return true;
		if (!(stream instanceof CosObjectStream) && ((stream.getCryptFilter() != null) || (stream.getDocument().getLinearization() == null)))
			return false;
		return true;
	}

	int shouldDecryptOrEncrypt(ASName type, ASName cryptFilter, boolean encrypting)
		throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		int retVal = 0;
		boolean encryptMetadata = shouldEncryptMetadata(type, cryptFilter, encrypting);
		if (!mInSave)
			retVal = encryptMetadata ? 1 : -1;
		else if (!encryptMetadata)
			retVal = -1;
		else if (mIsNewEncryption)
			retVal = 1;
		return retVal;
	}

	public SecurityManager getDecryptionSecurityManager()
	{
		return (mEncryptionHandle != null) ? mEncryptionHandle.getDecryptionSecurityManager() : null;
	}

	public boolean authenticateDecryption(DecryptedState decryptedState)
		throws PDFSecurityException
	{
		SecurityHandler secHandler;
		try {
			secHandler = getDefaultDecryptionHandler();
		} catch (PDFCosParseException excp) {
			throw new PDFSecurityAuthorizationException("Authentication failed", excp);
		}
		return mEncryptionHandle.authenticateSecurity(secHandler, decryptedState);
	}

	public void setDecryptionSecurityManager(SecurityManager securityMgr)
		throws PDFSecurityException, PDFCosParseException, PDFIOException
	{
		try {
			setDecryptionSecurityManager(securityMgr, getDecryptionMap());
		} catch (IOException e) {
			throw new PDFIOException(e);
		}
	}

	void setDecryptionSecurityManager(SecurityManager securityMgr, Map encryptParams)
		throws PDFSecurityException, PDFCosParseException, PDFIOException, IOException
	{
		if (mEncryptionHandle == null)
			getEncryptionImpl();
		if (mEncryptionHandle.getDecryptionSecurityManager() == null) {
			setupDecryption();
			mEncryptionHandle.setDecryptionSecurityManager(securityMgr, encryptParams, getIDForEncryption());
			if (mEncryptionHandle.getEncryptionSecurityManager() == null) {
				setEncryptionSecurityManager(securityMgr, encryptParams);
				mEncryptionHandle.setEncryptionSecurityManager(securityMgr, encryptParams, getIDForEncryption());
				mIsNewEncryption = false;
			}
		} else {
			throw new PDFSecurityAuthorizationException("Cannot unlock again already unlocked document");
		}
	}

	public SecurityManager getEncryptionSecurityManager()
	{
		return (mEncryptionHandle != null) ? mEncryptionHandle.getEncryptionSecurityManager() : null;
	}

	public void setEncryptionSecurityManager(SecurityManager securityMgr, Map encryptParams)
	{
		getEncryptionImpl();
		mNeedsEncryption = true;
		mIsNewEncryption = true;
		mEncryptionManager = securityMgr;
		mEncryptionMap = encryptParams;
	}

	/**
	 * Sets Security Manager for decryption equal to the Security Manager for encryption.
	 * Resets Security Manager for decryption if the Security Manager for encryption is not set.
	 * @throws PDFSecurityException
	 * @throws PDFCosParseException
	 * @throws PDFIOException
	 */
	public void setDecryptionAsEncryption()
		throws PDFSecurityException, PDFCosParseException, PDFIOException
	{
		try {
			if (mDoc.isEncrypted()) {
				SecurityManager encryptionMgr = getEncryptionSecurityManager();
				Map encryptionMap = getEncryptionMap();
				resetDecryptionSecurityManager();
				if (encryptionMgr != null)
					setDecryptionSecurityManager(encryptionMgr, encryptionMap);
			} else {
				resetDecryptionSecurityManager();
			}
			mIsNewEncryption = false;
			mInSave = false;
			setupDecryption();
		} catch (IOException e) {
			throw new PDFIOException(e);
		}
	}

	/**
	 * Setup encryption
	 * @throws PDFCosParseException
	 * @throws PDFIOException
	 * @throws PDFSecurityException
	 * @throws RuntimeException, hard failure, if encryption was never requested
	 * This represents a "can't happen" situation
	 */
	public void setupEncryption()
		throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		try {
			mInSave = true;
			if (!mNeedsEncryption)
				return;
			getEncryptionImpl();
			mEncryptionHandle.setEncryptionSecurityManager(mEncryptionManager, mEncryptionMap, getIDForEncryption());
			if (mIsNewEncryption) {
				SecurityHandler defaultHandler = getDefaultHandler(mEncryptionMap, mEncryptionManager);
				mEncryptionHandle.authenticateSecurity(mEncryptionMap, getIDForEncryption(), defaultHandler, null);
			}
		} catch (IOException e) {
			throw new PDFIOException(e);
		}
	}

	public void resetEncryptionSecurityManager()
	{
		if (mEncryptionHandle != null)
			mEncryptionHandle.resetEncryptionSecurityManager();
		mNeedsEncryption = false;
		mEncryptionManager = null;
		mIsNewEncryption = true;
	}

	public void resetDecryptionSecurityManager()
	{
		if (mEncryptionHandle != null)
			mEncryptionHandle.resetDecryptionSecurityManager();
	}

	private int getEncryptVersion() {
		int version = 0;
		if (mEncryptionMap != null) {
			Integer versionObj = (Integer) mEncryptionMap.get(V);
			if (versionObj != null) {
				version = versionObj.intValue();
			}
		}
		return version;
	}

	private int getDecryptVersion() throws PDFSecurityException
	{
		int version = 0;
		if (mDecryptDictionary != null) {
			if (mDecryptDictionary.containsKey(ASName.k_V)) {
				try {
					version = mDecryptDictionary.getInt(ASName.k_V);
				} catch (PDFCosParseException e) {
					throw new PDFSecurityException("PDFCosParseException while reading /Encrypt/V ", e);
				} catch (PDFIOException e) {
					throw new PDFSecurityException("PDFIOException while reading /Encrypt/V ", e);
				}
			}
		}
		return version;
	}

	/**
	 * Get the encrypted bytes for a string
	 * @throws PDFUnsupportedFeatureException if no encryption implementation is in place
	 * @throws PDFSecurityException
	 */
	public byte[] encryptString(CosString strObj, byte[] content)
		throws PDFSecurityException
	{
		if (needsEncryption(strObj)) {
			byte[] addKey = (getEncryptVersion() < 5) ? strObj.getObjectEncryptionKey(true) : null;
			return getEncryptionImpl().encryptString(content, addKey);
		}
		return content;
	}

	/**
	 * Get the decrypted bytes for a string
	 * @throws PDFUnsupportedFeatureException
	 * @throws PDFSecurityException
	 * @throws PDFSecurityException
	 * @throws PDFUnsupportedFeatureException if no encryption implementation is in place
	 */
	public byte[] decryptString(CosString strObj, byte[] content)
		throws PDFSecurityException
	{
		if (needsDecryption(strObj)) {
			byte[] addKey = (getDecryptVersion() < 5) ? strObj.getObjectEncryptionKey(false): null;
			return getEncryptionImpl().decryptString(content, addKey);
		}
		return content;
	}

	/**
	 * Get the decrypted contents of a stream.  This stream becomes the property of the caller and
	 * <b>must</b> be closed by them.
	 * @throws PDFIOException
	 * @throws PDFSecurityException
	 * @throws PDFCosParseException
	 * @throws PDFUnsupportedFeatureException if no encryption implementation is in place
	 */
	public InputByteStream decryptStream(ASName cryptFilter, CosStream strObj, InputByteStream content)
		throws PDFIOException, PDFSecurityException, PDFCosParseException
	{
		try {
			getEncryptionImpl();
			OutputByteStream dest =
				strObj.getStreamManager().getOutputByteStreamEncryptedDocument(Fixed.GROWABLE, content.length());
			mEncryptionHandle.decryptStream((cryptFilter != null) ? cryptFilter.asString(true) : null,
							content, dest, getStreamEncryptionKey(strObj, false));
			return dest.closeAndConvert();
		} catch (IOException e) {
			throw new PDFIOException(e);
		}
	}

	byte[] getStreamEncryptionKey(CosStream stream, boolean write)
		throws PDFCosParseException, PDFIOException, PDFSecurityException, IOException
	{
		// Acrobat does not add object/generation numbers to the encryption key
		// when a particular stream has /Crypt filter defined in its dictionary.
		// So, let's do some dancing...
		byte[] addKey = null;
		if (stream.getCryptFilter() == null) {
			int version = (write) ? getEncryptVersion() : getDecryptVersion();
			if (version < 5) {
				addKey = stream.getObjectEncryptionKey(write);
			}
		}
		return addKey;
	}

	/**
	 * Get the encrypted contents of a stream.  This encrypts the entire contents of the <code>InputByteStream</code>.
	 * If the caller desires to enrypt only a portion of an <code>InputByteStream</code> they should <i>slice</i>
	 * the original <code>InputByteStream</code>. The returned encrypted stream becomes the property of the caller and
	 * <b>must</b> be closed by them.
	 * @throws PDFCosParseException
	 * @throws PDFIOException
	 * @throws PDFSecurityException
	 * @throws PDFUnsupportedFeatureException if no encryption implementation is in place
	 */
	public InputByteStream encryptStream(ASName cryptFilter, CosStream strObj, InputByteStream content)
		throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		try {
			getEncryptionImpl();
			// Guess the size based on the original stream
			OutputByteStream dest =
				strObj.getStreamManager().getOutputByteStreamEncryptedDocument(Fixed.GROWABLE, content.length());
			content.seek(0);	// make sure that we are at the beginning of the stream!
			mEncryptionHandle.encryptStream((cryptFilter != null) ? cryptFilter.asString(true) : null,
							content, dest, getStreamEncryptionKey(strObj, true));
			return dest.closeAndConvert();
		} catch (IOException e) {
			throw new PDFIOException(e);
		}
	}

	public boolean setEncryptionState(boolean doIt)
	{
		boolean old = getEncryptionState();
		mDoDecryption = doIt;
		return old;
	}

	public boolean getEncryptionState()
	{
		return mDoDecryption;
	}

	private byte[] getIDForEncryption()
		throws PDFCosParseException, PDFIOException, PDFSecurityException, IOException
	{
		byte[] id = null;
		CosDictionary trailer = mDoc.getTrailer();
		if (trailer != null) {
			CosObject idStr = trailer.get(ASName.k_ID);
			if ((idStr != null) && (idStr.getType() == CosObject.t_Array))
				id = ((CosString) ((CosArray) idStr).get(0)).byteArrayValue();
		}
		return id;
	}

	// EFF processing
	private boolean hasEFF()
		throws PDFSecurityAuthorizationException
	{
		boolean result = false;
		if (this.needsEncryption()) {
			Map encryptionParams = this.getEncryptionMap();
			if (encryptionParams != null && encryptionParams.containsKey("EFF"))
				result = true;
		}
		return result;
	}

	private boolean needsEncryption(CosString stringObj)
	{
		if (!needsEncryption())
			return false;
		CosObjectInfo strInfo = stringObj.getInfo();
		if (strInfo == null) {
			CosObject parent = stringObj.getParentObj();
			if (parent == null)
				return true;
			strInfo = parent.getInfo();
		}
		if (strInfo == null)
			return true;
		return mDoDecryption && !strInfo.isWriteCompressed();
	}

	private boolean needsDecryption(CosString stringObj)
	{
		if (!needsDecryption())
			return false;
		CosObjectInfo strInfo = stringObj.getInfo();
		if (strInfo == null) {
			CosObject parent = stringObj.getParentObj();
			if (parent == null)
				return true;
			strInfo = parent.getInfo();
		}
		if (strInfo == null)
			return true;
		return mDoDecryption && !strInfo.isCompressed();
	}

	/**
	 * If this document requires EFF processing, remove the Crypt filter
	 */
	CosArray checkEFFOutputFilter(CosStream stmObj)
		throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		CosArray outputFilters = stmObj.getOutputFiltersList();
		if (outputFilters == null && hasEFF() && stmObj.isDirty()) {
			outputFilters = stmObj.getInputFiltersList();
			if (outputFilters != null) {
				removeCryptFilter(outputFilters);
			}
		}
		return outputFilters;
	}

	/*
	 * Acrobat wants /Filter /Crypt when /Metadata false but only when there are
	 * Crypt filters there. Hence the following hack.
	 */
	CosArray checkMetadataStream(CosStream stmObj)
		throws PDFCosParseException, PDFIOException, PDFSecurityException, IOException
	{
		CosArray outputFilters = stmObj.getOutputFiltersList();
		CosObject type = stmObj.get(ASName.k_Type);
		if ((type != null) && needsEncryption() && ASName.k_Metadata.equals(type.nameValue())) {
			if (outputFilters == null)
				outputFilters = stmObj.getInputFiltersList();
			Map encryptParams = getEncryptionMap();
			if (encryptParams != null) {
				removeCryptFilter(outputFilters);
				if (encryptParams.containsKey(CF) && !shouldEncryptMetadata(type.nameValue(), null, true)) {
					CosArray cryptFilter = mDoc.createCosArray();
					cryptFilter.add(mDoc.createCosName(ASName.k_Crypt));
					cryptFilter.add(mDoc.createCosNull());
					outputFilters.add(0, cryptFilter);
				}
			}
		}
		return outputFilters;
	}

	private boolean removeCryptFilter(CosArray filters) throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		Iterator<CosObject> filtersIter = filters.iterator();
		CosArray filterAtIndex;
		while (filtersIter.hasNext()) {
			filterAtIndex = (CosArray) filtersIter.next();
			if (ASName.k_Crypt.equals(((CosName)(filterAtIndex.get(0))).nameValue())) {
				filtersIter.remove();
				return true;
			}
		}
		return false;
	}

	EncryptionHandler getStreamDecryptionHandler(String cryptName)
		throws PDFSecurityException
	{
		getEncryptionImpl();
		return mEncryptionHandle.getStreamDecryptionHandler(cryptName);
	}

	EncryptionHandler getStreamEncryptionHandler(String cryptName)
		throws PDFSecurityException
	{
		getEncryptionImpl();
		return mEncryptionHandle.getStreamEncryptionHandler(cryptName);
	}

	/**
	 * 
	 * This method creates and caches EncryptionHandlerState object for encryption suitable
	 * for the specified CosStream.
	 * It returns the same EncryptionHandlerState object for each {@link EncryptionHandler}
	 * regardless of how many streams use it. Different streams may not use the same EncryptionHandlerState
	 * instance concurently.
	 * @param stream
	 * @return EncryptionHandlerState object associated with the EncryptionHandler
	 * suitable for the specified CosStream.
	 * @throws PDFCosParseException
	 * @throws PDFIOException
	 * @throws PDFSecurityException
	 * @throws IOException
	 */
	EncryptionHandlerState getStreamDecryptionStateHandler(CosStream stream, InputByteStream content)
		throws PDFCosParseException, PDFIOException, PDFSecurityException, IOException
	{
		if (!stream.needsDecryption(content))
			return null;
		ASName cryptFilter = stream.getCryptFilter();
		String cryptName = cryptFilter == null ? null : cryptFilter.asString(true);
		getEncryptionImpl();
		EncryptionHandler handler = mEncryptionHandle.getStreamDecryptionHandler(cryptName);
		if (handler == null)
			return null;
		if (mDecryptionStateHandlers == null)
			mDecryptionStateHandlers = new HashMap();
		EncryptionHandlerState stateHandler = (EncryptionHandlerState) mDecryptionStateHandlers.get(handler);
		if (stateHandler == null) {
			stateHandler = handler.createEncryptionHandlerState();
			mDecryptionStateHandlers.put(handler, stateHandler);
		}
		return stateHandler;
	}


	/**
	 * 
	 * This method creates and caches EncryptionHandlerState object for decryption suitable
	 * for the specified CosStream.
	 * It returns the same EncryptionHandlerState object for each {@link EncryptionHandler}
	 * regardless of how many streams use it. Different streams may not use the same EncryptionHandlerState
	 * instance concurently.
	 * @param stream
	 * @return EncryptionHandlerState object associated with the EncryptionHandler
	 * suitable for the specified CosStream.
	 * @throws PDFCosParseException
	 * @throws PDFIOException
	 * @throws PDFSecurityException
	 * @throws IOException
	 */
	EncryptionHandlerState getStreamEncryptionStateHandler(CosStream stream)
		throws PDFCosParseException, PDFIOException, PDFSecurityException, IOException
	{
		if (!stream.needsEncryption())
			return null;
		getEncryptionImpl();
		ASName cryptFilter = stream.getCryptFilter();
		String cryptName = cryptFilter == null ? null : cryptFilter.asString(true);
		EncryptionHandler handler = mEncryptionHandle.getStreamEncryptionHandler(cryptName);
		if (handler == null)
			return null;
		if (mEncryptionStateHandlers == null)
			mEncryptionStateHandlers = new HashMap();
		EncryptionHandlerState stateHandler = (EncryptionHandlerState) mEncryptionStateHandlers.get(handler);
		if (stateHandler == null) {
			stateHandler = handler.createEncryptionHandlerState();
			mEncryptionStateHandlers.put(handler, stateHandler);
		}
		return stateHandler;
	}

	void resetDecryptionStateHandlers()
	{
		mDecryptionStateHandlers = null;
	}

	void resetEncryptionStateHandlers()
	{
		mEncryptionStateHandlers = null;
	}
}
