/*
 * File: CosList.java
 *
 * ****************************************************************************
 *
 *	ADOBE CONFIDENTIAL
 *	___________________
 *
 *	Copyright 2003-2006 Adobe Systems Incorporated
 *	All Rights Reserved.
 *
 *	NOTICE: All information contained herein is, and remains the property of
 *	Adobe Systems Incorporated and its suppliers, if any. The intellectual
 *	and technical concepts contained herein are proprietary to Adobe Systems
 *	Incorporated and its suppliers and may be covered by U.S. and Foreign
 *	Patents, patents in process, and are protected by trade secret or
 *	copyright law. Dissemination of this information or reproduction of this
 *	material is strictly forbidden unless prior written permission is obtained
 *	from Adobe Systems Incorporated.
 *
 * ***************************************************************************/

package com.adobe.internal.pdftoolkit.core.cos;

import java.util.Iterator;

/**
 * A simple, fast, list object for maintaining ordered lists of objects.
 */
public final class CosList
{
	// Parameters
	private static final int BLOCK_BITS = 6;					// Log base two object entries per block
	private static final int BLOCK_SIZE = 1 << BLOCK_BITS;				// Object entries per block
	private static final int BLOCK_MASK = (1 << BLOCK_BITS) - 1;			// Mask for index into blocks

	// Per object storage
	private int mLevels = 0;							// Current number of layers in block list
	private int mFirst = 0;								// First used entry in list
	private int mSize = 0;								// Size of used part of list
	private int mCount = 0;								// Count of non-null objects in list
	private Object[] mObjects = null;						// The object structure
	private Object[] mCurNode = null;						// Current iteration leaf node

	/**
	 * Constructor does nothing except allocate master block
	 * 
	 * @exclude
	 */
	public CosList()
	{
		mLevels = 1;
		mFirst = -1;
		mObjects = new Object[BLOCK_SIZE];
	}

	/**
	 * Adds specified object to this list.
	 * @exclude
	 */
	public void add(Object obj)
	{
		add(mSize, obj);
	}

	/*
	 * 
	 * @exclude
	 * Add a new item to the Object list. Create a block if there isn't one yet
	 * for this object index.
	 */
	public void add(int index, Object obj)
	{
		if (index < 0)
			return;
		if (obj == null) {
			delete(index);
		} else {
			while (index >= (1 << (BLOCK_BITS * mLevels))) {
				Object[] block = new Object[BLOCK_SIZE];
				block[0] = mObjects;
				mObjects = block;
				mLevels++;
			}
			int curLevel;
			int curIndex;
			Object[] nodeBlock = mObjects;
			for (curLevel = 0; curLevel < mLevels - 1; curLevel++) {
				curIndex = (index >> (BLOCK_BITS * (mLevels - curLevel - 1))) & BLOCK_MASK;
				if (nodeBlock[curIndex] == null)
					nodeBlock[curIndex] = new Object[BLOCK_SIZE];
				nodeBlock = (Object[])nodeBlock[curIndex];
			}
			curIndex = index & BLOCK_MASK;
			if (nodeBlock[curIndex] == null) {
				if (++mCount == 1) {
					mFirst = index;
					mSize = index + 1;
				} else if (index >= mSize) {
					mSize = index + 1;
				} else if (index < mFirst) {
					mFirst = index;
				}
			}
			nodeBlock[curIndex] = obj;
		}
	}

	/*
	 * 
	 * @exclude
	 * Copy all items from the source list, in order, to the end of this one.
	 */
	public void addAll(CosList src)
	{
		Iterator iter = src.iterator();
		while (iter.hasNext()) {
			add(iter.next());
		}
	}

	/*
	 * 
	 * @exclude
	 * Delete the item at index and any empty intermediate nodes.
	 */
	public void delete(int index)
	{
		if (index < 0 || index >= (1 << (BLOCK_BITS * mLevels)))
			return;
		int curLevel;
		int curIndex;
		Object[] blockStack = new Object[mLevels];
		int[] indexStack = new int[mLevels];
		Object[] nodeBlock = mObjects;
		for (curLevel = 0; curLevel < mLevels - 1; curLevel++) {
			blockStack[curLevel] = nodeBlock;
			curIndex = (index >> (BLOCK_BITS * (mLevels - curLevel - 1))) & BLOCK_MASK;
			indexStack[curLevel] = curIndex;
			nodeBlock = (Object[])nodeBlock[curIndex];
			if (nodeBlock == null)
				return;
		}
		blockStack[curLevel] = nodeBlock;
		curIndex = index & BLOCK_MASK;
		indexStack[curLevel] = curIndex;
		if (nodeBlock[curIndex] == null)
			return;
		if (--mCount == 0) {
			mLevels = 1;
			mFirst = -1;
			mSize = 0;
			mObjects = new Object[BLOCK_SIZE];
			return;
		}
		for (curLevel = mLevels - 1; curLevel >= 0; curLevel--) {
			nodeBlock = (Object[])blockStack[curLevel];
			curIndex = indexStack[curLevel];
			nodeBlock[curIndex] = null;
			for (curIndex = 0; curIndex < BLOCK_SIZE; curIndex++) {
				if (nodeBlock[curIndex] != null)
					break;
			}
			if (curIndex != BLOCK_SIZE)
				break;
		}
		if (index == mFirst)
			mFirst = findNext(index);
		else if (index == mSize - 1)
			mSize = findPrevious(index) + 1;
		while (mSize <= (1 << (BLOCK_BITS * (mLevels - 1))) && mLevels > 1) {
			mObjects = (Object[])mObjects[0];
			mLevels --;
		}			       
	}

	/*
	 * 
	 * @exclude
	 * Get an item from the Object list. Returns NULL if there is no
	 * block for this object number.
	 */
	public Object get(int index)
	{
		if (index < 0 || index >= (1 << (BLOCK_BITS * mLevels)))
			return null;
		int curLevel;
		int curIndex;
		Object[] nodeBlock = mObjects;
		for (curLevel = 0; curLevel < mLevels - 1; curLevel++) {
			curIndex = (index >> (BLOCK_BITS * (mLevels - curLevel - 1))) & BLOCK_MASK;
			nodeBlock = (Object[])nodeBlock[curIndex];
			if (nodeBlock == null)
				return null;
		}
		return nodeBlock[index & BLOCK_MASK];
	}

	/*
	 * 
	 * @exclude
	 * Return the current real size of the list. This will be the maximum
	 * index ever set, plus one.
	 */
	public int size()
	{
		return mSize;
	}

	/*
	 * 
	 * @exclude
	 * Return the current count of non-null objects in the list.
	 */
	public int count()
	{
		return mCount;
	}

	/*
	 * 
	 * @exclude
	 * True if there are NO non-null items on this list.
	 */
	public boolean isEmpty()
	{
		return (mSize == 0);
	}

	/*
	 * 
	 * @exclude
	 * True if this list index contains an object.
	 */
	public boolean containsIndex(int index)
	{
		return (get(index) != null);
	}

	/*
	 * 
	 * @exclude
	 * Make and return an exact deep copy of this list, but without making
	 * new instances of the objects it contains.
	 */
	public CosList copy()
	{
		CosList listCopy = new CosList();
		Iterator iter = this.iterator();
		while (iter.hasNext())
			listCopy.add(iter.next());
		return listCopy;
	}

	/*
	 * 
	 * @exclude
	 * Convenience method to get the first object on the list.
	 */
	public Object first()
	{
		return get(mFirst);
	}

	/*
	 * 
	 * @exclude
	 * Convenience method to get the last object on the list.
	 */
	public Object last()
	{
		if (mSize == 0)
			return null;
		return get(mSize - 1);
	}

	/*
	 * 
	 * @exclude
	 * Find next used index after this one or -1 if none.
	 */
	public int findNext(int index)
	{
		index++;
		if (index >= mSize)
			return -1;
		if (index < mFirst)
			index = mFirst;
		int curLevel = 0;
		int curIndex = index >> (BLOCK_BITS * (mLevels - 1));
		Object[] blockStack = new Object[mLevels - 1];
		int[] indexStack = new int[mLevels - 1];
		Object[] nodeBlock = mObjects;
		while (true) {
			if (curIndex == BLOCK_SIZE) {
				if (--curLevel < 0)
					return -1;
				nodeBlock = (Object[])blockStack[curLevel];
				curIndex = indexStack[curLevel];
				curIndex++;
			} else if (nodeBlock[curIndex] == null) {
				index = 0;
				curIndex++;
			} else {
				if (curLevel == mLevels - 1) {
					index = 0;
					for (curLevel = 0; curLevel < mLevels - 1; curLevel++) {
						index |= indexStack[curLevel];
						index <<= BLOCK_BITS;
					}
					index |= curIndex;
					mCurNode = nodeBlock;
					return index;
				} else {
					blockStack[curLevel] = nodeBlock;
					indexStack[curLevel] = curIndex;
					nodeBlock = (Object[])nodeBlock[curIndex];
					curLevel++;
					curIndex = (index >> (BLOCK_BITS * (mLevels - curLevel - 1))) & BLOCK_MASK;
				}
			}
		}
	}

	/*
	 * 
	 * @exclude
	 * Find previous used index before this one or -1 if none.
	 */
	public int findPrevious(int index)
	{
		index--;
		if (index < mFirst)
			return -1;
		if (index > mSize - 1)
		    index = mSize - 1;
		int curLevel = 0;
		int curIndex = index >> (BLOCK_BITS * (mLevels - 1));
		Object[] blockStack = new Object[mLevels - 1];
		int[] indexStack = new int[mLevels - 1];
		Object[] nodeBlock = mObjects;
		while (true) {
			if (curIndex == -1) {
				if (--curLevel < 0)
					return -1;
				nodeBlock = (Object[])blockStack[curLevel];
				curIndex = indexStack[curLevel];
				curIndex--;
			} else if (nodeBlock[curIndex] == null) {
				index = -1;
				curIndex--;
			} else {
				if (curLevel == mLevels - 1) {
					index = 0;
					for (curLevel = 0; curLevel < mLevels - 1; curLevel++) {
						index |= indexStack[curLevel];
						index <<= BLOCK_BITS;
					}
					index |= curIndex;
					return index;
				} else {
					blockStack[curLevel] = nodeBlock;
					indexStack[curLevel] = curIndex;
					nodeBlock = (Object[])nodeBlock[curIndex];
					curLevel++;
					curIndex = (index >> (BLOCK_BITS * (mLevels - curLevel - 1))) & BLOCK_MASK;
				}
			}
		}
	}

	/*
	 * 
	 * @exclude
	 * Nice simple iterator. Returns next non-null entry in the list.
	 * Caveat: NOT INDEPENDENT. You may only have one of these iterators
	 * in use at any one time per CosList object.
	 */
	public Iterator<Object> iterator()
	{
		return new CosListIterator();
	}

	class CosListIterator implements Iterator
	{
		private int mCurIndex;

		CosListIterator()
		{
			mCurIndex = -1;
			mCurNode = null;
		}

		public boolean hasNext()
		{
			if (mCurIndex < mSize - 1)
				return true;
			return false;
		}


		public Object next()
		{
			if (mCurNode != null) {
				int nodeIndex = (mCurIndex + 1) & BLOCK_MASK;
				if (nodeIndex != 0) {
					while (nodeIndex <= BLOCK_MASK) {
						if (mCurNode[nodeIndex] != null) {
							mCurIndex = (mCurIndex & ~BLOCK_MASK) + nodeIndex;
							return mCurNode[nodeIndex];
						}
						nodeIndex++;
					}
				}
			}
			mCurIndex = findNext(mCurIndex);
			if (mCurIndex < 0)
				return null;
			return mCurNode[mCurIndex & BLOCK_MASK];
		}

		public void remove()
		{
			delete(mCurIndex);
			mCurNode = null;
		}
	}
}
