/* ****************************************************************************
 *
 *	File: COSName.java
 *
 * ****************************************************************************
 *
 *	ADOBE CONFIDENTIAL
 *	___________________
 *
 *	Copyright 2003-2006 Adobe Systems Incorporated
 *	All Rights Reserved.
 *
 *	NOTICE: All information contained herein is, and remains the property of
 *	Adobe Systems Incorporated and its suppliers, if any. The intellectual
 *	and technical concepts contained herein are proprietary to Adobe Systems
 *	Incorporated and its suppliers and may be covered by U.S. and Foreign
 *	Patents, patents in process, and are protected by trade secret or
 *	copyright law. Dissemination of this information or reproduction of this
 *	material is strictly forbidden unless prior written permission is obtained
 *	from Adobe Systems Incorporated.
 *
 * ***************************************************************************/

package com.adobe.internal.pdftoolkit.core.cos;

import java.io.IOException;

import com.adobe.internal.io.stream.OutputByteStream;
import com.adobe.internal.pdftoolkit.core.exceptions.PDFCosParseException;
import com.adobe.internal.pdftoolkit.core.exceptions.PDFIOException;
import com.adobe.internal.pdftoolkit.core.exceptions.PDFSecurityException;
import com.adobe.internal.pdftoolkit.core.types.ASName;
import com.adobe.internal.pdftoolkit.core.util.StringOps;

/**
 * Represents a COS name object as defined in section 3.2.4 of the PDF
 * Reference Manual version 1.4.
 */
public class CosName extends CosScalar
{
	private ASName name;		// Atom represented by the name object

	/**
	 * Constructs a COS name object representing the specified atom.
	 *
	 * @param doc		Document containing the name object
	 * @param value		Atom being wrapped by the name object
	 * @param info		Info for the name object
	 */
	CosName(CosDocument doc, ASName value, CosObjectInfo info)
	{
		super(doc, info);
		name = value;
	}

	/**
	 * 
	 * return the type of this CosObject
	 */
	@Override
	public int getType()
	{
		return t_Name;
	}

	/**
	 * 
	 * Returns the atom that is represented by the COS name object.
	 *
	 * @return Atom wrapped by the name object.
	 */
	@Override
	public ASName nameValue()
	{
		return name;
	}

	
	@Override
	public String toString()
	{
		return name.toString();
	}
	
	@Override
	void writeOut(OutputByteStream outStream, boolean inString, boolean inDebugMode)
	throws PDFCosParseException, IOException, PDFSecurityException, PDFIOException
	{
		if (inString && inDebugMode)
			outStream.write(StringOps.toByteArray(name.asString(inDebugMode)));
		else
			name.write(outStream);
	}
	
//	protected String outputString()
//	{
//		return name.outputString();
//	}
//
	
	@Override
	public Object getValue()
	{
		return name.asString(true);
	}
	
	/**
	 *  This method returns true if both CosNames have same ASName inside.
	 *  Returns false if passed CosObject is not an instance of CosName.
	 *  @param value
	 *  @return boolean
	 */
	@Override
	public boolean equals(CosObject value){
		if(!(value instanceof CosName) || value.getDocument() != this.getDocument())
			return false;
		if(value == this)
			return true; 
		return (value.nameValue() == this.nameValue());
	}
}
