/* ****************************************************************************
 *
 *	File: CosObject.java
 *
 * ****************************************************************************
 *
 *	ADOBE CONFIDENTIAL
 *	___________________
 *
 *	Copyright 2003-2005 Adobe Systems Incorporated
 *	All Rights Reserved.
 *
 *	NOTICE: All information contained herein is, and remains the property of
 *	Adobe Systems Incorporated and its suppliers, if any. The intellectual
 *	and technical concepts contained herein are proprietary to Adobe Systems
 *	Incorporated and its suppliers and may be covered by U.S. and Foreign
 *	Patents, patents in process, and are protected by trade secret or
 *	copyright law. Dissemination of this information or reproduction of this
 *	material is strictly forbidden unless prior written permission is obtained
 *	from Adobe Systems Incorporated.
 *
 * ***************************************************************************/

package com.adobe.internal.pdftoolkit.core.cos;

import java.io.IOException;

import com.adobe.internal.io.ByteWriterFactory.Fixed;
import com.adobe.internal.io.stream.InputByteStream;
import com.adobe.internal.io.stream.OutputByteStream;
import com.adobe.internal.io.stream.StreamManager;
import com.adobe.internal.pdftoolkit.core.exceptions.PDFCosParseException;
import com.adobe.internal.pdftoolkit.core.exceptions.PDFCosUnexpectedTypeException;
import com.adobe.internal.pdftoolkit.core.exceptions.PDFException;
import com.adobe.internal.pdftoolkit.core.exceptions.PDFIOException;
import com.adobe.internal.pdftoolkit.core.exceptions.PDFSecurityException;
import com.adobe.internal.pdftoolkit.core.types.ASHexString;
import com.adobe.internal.pdftoolkit.core.types.ASName;
import com.adobe.internal.pdftoolkit.core.types.ASString;
import com.adobe.internal.pdftoolkit.core.util.ByteOps;

/**
 * Base class for all COS objects.
 */
public abstract class CosObject
{
	/*
	 * Every Cos Object has an ID.  For objects that can are indirectly referenced, it is
	 * unique to that object.  For objects that are direct objects, i.e., that are embedded
	 * in outer objects (such as dictionaries or arrays), the ID is null.  The fields of the ID
	 * specify where the object is in the xref table.  There is a one to one correspondence
	 * between ID's and xref table entries.  There is a simple to understand characterization
	 * of the ID:
	 *
	 * The ID specifies which object referenced by the xref table will have to be rewritten
	 * if this object becomes dirty.
	 */
	private CosDocument mDoc;		// Document containing this COS object
	private CosObjectInfo mInfo;		// Info for this object if indirect

	/*
	 * A numeric type identifier for each of the concrete subtypes
	 * of CosObject plus an additional constant that can be used to
	 * express the absence of a key/value pair.
	 */
	public static final int t_Null = 0;
	public static final int t_Numeric = 1;
	public static final int t_Boolean = 2;
	public static final int t_Name = 3;
	public static final int t_String = 4;
	public static final int t_Array = 5;
	public static final int t_Dictionary = 6;
	public static final int t_Stream = 7;
	public static final int t_ObjectRef = 8;
	public static final int t_KeyAbsent = 9;

	// An enum to identify whether an object being created is direct or indirect
	public static final int DIRECT = 0;
	public static final int INDIRECT = 1;

	/**
	 * This flag is set as true if this object was repaired.
	 */
	protected boolean repaired = false;
	/**
	 * Base class constructor.
	 *
	 * @param doc		Document containing the object
	 * @param info		Info for the object
	 */
	CosObject(CosDocument doc, CosObjectInfo info)
	{
		mDoc = doc;
		if (info != null && info.getObjNum() != 0) {
			mInfo = info;
			mInfo.setObject(this);
		}
	}
	
	/**
	 * This method write character representation of the Cos object's content to an 
	 * OutputByteStream. It is called from both writeOut(OBS) and toString() methods.
	 * Some implementations of this method may perform different 
	 * actions while debugging. See CosString and CosStream
	 * @param outStream stream to which to write
	 * @param inToString indicator that it is called from toString()
	 * @param inDebug indicator of the debugging mode.
	 * @throws PDFCosParseException
	 * @throws IOException
	 * @throws PDFSecurityException
	 * @throws PDFIOException 
	 */
	abstract void writeOut(OutputByteStream outStream, boolean inToString, boolean inDebug)
	throws PDFCosParseException, IOException, PDFSecurityException, PDFIOException;
	
	void writeOut(OutputByteStream outStream)
	throws PDFCosParseException, IOException, PDFSecurityException, PDFIOException
	{
		writeOut(outStream, false, false);
	}

	/**
	 * 
	 *
	 * Close the <code>CosObject</code> and release any and all resources associated with it.
	 * Any use of this <code>CosObject</code> after this call is considered an error.
	 * @throws IOException
	 */
	void close()
		throws IOException
	{
		mDoc = null;
		mInfo = null;
	}

	/**
	 * 
	 *
	 * Release resources used by the <code>CosObject</code> that can be recreated later.
	 * @throws IOException
	 */
	void release()
		throws IOException
	{
		markNotDirty();
	}

	/**
	 * 
	 * return which of the concrete subtypes of CosObject we are
	 * @return one of the t_Mumble static final int values defined above
	 */
	public abstract int getType();

	/**
	 * 
	 * @return Object value of CosObject
	 * @throws PDFCosParseException
	 * @throws PDFIOException
	 * @throws PDFSecurityException
	 */
	public abstract Object getValue()
		throws PDFCosParseException, PDFIOException, PDFSecurityException;

	/**
	 * 
	 */
	public boolean isIndirect()
	{
		return (mInfo != null);
	}

	/**
	 * 
	 * Indirect object number or zero if direct
	 */
	public int getObjNum()
	{
		if (mInfo != null)
			return mInfo.getObjNum();
		return 0;
	}

	/**
	 * 
	 * Indirect object generation or -1 if direct
	 */
	public int getObjGen()
	{
		if (mInfo != null)
			return mInfo.getObjGen();
		return -1;
	}

	/**
	 * 
	 * Return EOF of object's update section or zero if cannot be determined.
	 */
	public long getObjEOF()
	{
		if (mInfo == null) {
			CosObject parentObj = null;
			if (this instanceof CosContainer)
				parentObj = ((CosContainer)this).getParentObj();
			if (this instanceof CosString)
				parentObj = ((CosString)this).getParentObj();
			if (parentObj == null)
				return 0;
			return mDoc.getObjEOF(parentObj);
		}
		return mDoc.getObjEOF(this);
	}

	/**
	 * 
	 * Return index of object's update section or -1 if cannot be determined.
	 */
	public int getObjRevision()
	{
		if (mInfo == null) {
			CosObject parentObj = null;
			if (this instanceof CosContainer)
				parentObj = ((CosContainer)this).getParentObj();
			if (this instanceof CosString)
				parentObj = ((CosString)this).getParentObj();
			if (parentObj == null)
				return -1;
			return mDoc.getObjRevision(parentObj);
		}
		return mDoc.getObjRevision(this);
	}

	/**
	 * 
	 * Return object's position or zero if cannot be determined.
	 */
	public long getObjPos()
	{
		if (mInfo == null) {
			CosObject parentObj = null;
			if (this instanceof CosContainer)
				parentObj = ((CosContainer)this).getParentObj();
			if (this instanceof CosString)
				parentObj = ((CosString)this).getParentObj();
			if (parentObj == null)
				return 0;
			return mDoc.getObjPos(parentObj);
		}
		return mDoc.getObjPos(this);
	}

	/**
	 * 
	 * Returns true if this INDIRECT object currently resides in an object stream.
	 * Returns false always if direct.
	 */
	public boolean isCompressed()
	{
		if (mInfo == null)
			return false;
		return mInfo.isCompressed();
	}

	/**
	 * 
	 * If this object is indirect and can be flushed, flush it.
	 * Returns false if direct or dirty.
	 */
	boolean flushCachedObject()
	{
		if (mInfo == null)
			return false;
		if (!mInfo.flushCachedObject())
			return false;
		mInfo = null;
		return true;
	}

	/**
	 * 
	 * Mark object as not dirty if it was dirty.
	 * Returns false if direct or not dirty.
	 */
	boolean markNotDirty()
	{
		if (mInfo == null)
			return false;
		return mInfo.markNotDirty();
	}

	/**
	 * 
	 * Accessor functions for scalar CosObjects.  These are defined in the
	 * base class to avoid having to make an explicit cast after determining
	 * the subclass by either getType() or instanceof.  The ones defined in
	 * the base class all throw RuntimeException. For each scalar subclass
	 * the appropriate method is overridden, so getting here represents a
	 * "can't happen" situation.
	 */
	public int intValue()
	{
		throw new PDFCosUnexpectedTypeException("expected CosNumeric");
	}

	/**
	 * Returns long value if it's type of {@link CosNumeric} else throws exception.
	 */
	public long longValue()
	{
		throw new PDFCosUnexpectedTypeException("expected CosNumeric");
	}

	/**
	 * Returns double value if it's type of {@link CosNumeric} else throws exception.
	 */
	public double doubleValue()
	{
		throw new PDFCosUnexpectedTypeException("expected CosNumeric");
	}

	/**
	 * Returns number value if it's type of {@link CosNumeric} else throws exception.
	 */
	public Number numberValue()
	{
		throw new PDFCosUnexpectedTypeException("expected CosNumeric");
	}

	/**
	 * Returns boolean value if it's type of {@link CosBoolean} else throws exception.
	 */
	public boolean booleanValue()
	{
		throw new PDFCosUnexpectedTypeException("expected CosBoolean");
	}

	/**
	 * Returns name value if it's type of {@link CosName} else throws exception.
	 */
	public ASName nameValue()
	{
		throw new PDFCosUnexpectedTypeException("expected CosName");
	}

	/**
	 * Returns string value if it's type of {@link CosString} else throws exception.
	 * @throws PDFSecurityException
	 */
	public ASString stringValue()
		throws PDFSecurityException
	{
		throw new PDFCosUnexpectedTypeException("expected CosString");
	}

	/**
	 * Returns hex string value if it's type of {@link CosString} else throws exception.
	 * @throws PDFSecurityException
	 */
	public ASHexString hexStringValue() 
	throws PDFSecurityException
	{
		throw new PDFCosUnexpectedTypeException("expected CosString");
	}
	
	/**
	 * Returns text value if it's type of {@link CosString} else throws exception.
	 * @throws PDFSecurityException
	 */
	public String textValue()
		throws PDFSecurityException
	{
		throw new PDFCosUnexpectedTypeException("expected CosString or CosName");
	}

	
	byte[] getObjectEncryptionKey(boolean write)
	{
		byte[] key = new byte[5];
		int objNum = getInfo().getObjNum();
		int gen = getInfo().getObjGen();
		if (!write) {
			CosLinearization cosLin = getDocument().getLinearization();
			if (cosLin != null && !(this instanceof CosObjectStream)) {
				objNum = cosLin.mapNewToOldObjNum(objNum);
				gen = cosLin.mapNewToOldObjGen(objNum);
			}
		}
		System.arraycopy(ByteOps.splitInt2Bytes(objNum, 3), 0, key, 0, 3);
		System.arraycopy(ByteOps.splitInt2Bytes(gen, 2), 0, key, 3, 2);
		return key;
	}

	/**
	 * Obtain the COS document that contains this object.
	 *
	 * @return COSDocument containing this object.
	 */
	public CosDocument getDocument()
	{
		return mDoc;
	}

	/**
	 * Obtain the StreamManager for the document containing this object.
	 *
	 * @return StreamManager to use for this COS object.
	 */
	public StreamManager getStreamManager()
	{
		return this.getDocument().getStreamManager();
	}

	
	CosObjectInfo getInfo()
	{
		return mInfo;
	}

	
	void setInfo(CosObjectInfo info)
	{
		mInfo = info;
	}

	/**
	 * Returns if this cosobject is dirty, else false.
	 */
	public boolean isDirty()
	{
		if (mInfo != null)
			return mInfo.isDirty();
		return false;
	}
	
	@Override
	public String toString()
	{
		return toString(false);
	}
	
	/**
	 * This returns character representation of the Cos object's content as Java String object. 
	 * Some implementations of this method may perform different 
	 * actions while debugging. See CosString and CosStream
	 * @param inDebug indicator of the debugging mode.
	 * @return String representation of the Cos object's content
	 */
	protected String toString(boolean inDebug)
	{
		InputByteStream stream = null;
		try {
			OutputByteStream outStream = getStreamManager().getOutputByteStreamClearTemp(Fixed.GROWABLE, 10000);
			writeOut(outStream, true, inDebug);
			stream = outStream.closeAndConvert();
			StringBuilder string = new StringBuilder((int) stream.length());
			while (stream.bytesAvailable() > 0)
			{
				string.append((char) stream.read());
			}
			return string.toString();
		} catch (PDFException e) {
			return e.toString();
		} catch (IOException e) {
			return e.toString();
		} finally {
			if (stream != null)
			{
				try
				{
					stream.close();
				} catch (IOException e) {
					return e.toString();
				}
			}
		}
	}

	/**
	 *  This method returns true if either they have same reference or have same 
	 *  data inside.
	 *  Returns false if passed CosObject is not on same document.
	 *  @param value
	 *  @return boolean
	 */
	public abstract boolean equals(CosObject value);
	
	
	/**
	 *  Object.equals(Object) is overridden here.
	 */
	@Override
	public boolean equals(Object value){
		if(!(value instanceof CosObject))
			return false;
		return this.equals((CosObject)value);
	}
	
	/**
	 * Sets true if the object was repaired.
	 * @param repaired
	 */
	void setRepaired(boolean repaired){
		this.repaired = repaired;
		mDoc.setDocumentCosLevelRepaired();
	}
	
	/**
	 * Returns true if the object was repaired.
	 * @return boolean
	 */
	boolean isRepaired(){
		return this.repaired;
	}
}
