/*
* File: CosOpenOptions.java
*
*************************************************************************
** ADOBE CONFIDENTIAL
* ___________________
*
*  Copyright 2012 Adobe Systems Incorporated
*  All Rights Reserved.
*
* NOTICE:  All information contained herein is, and remains
* the property of Adobe Systems Incorporated and its suppliers,
* if any.  The intellectual and technical concepts contained
* herein are proprietary to Adobe Systems Incorporated and its
* suppliers and are protected by trade secret or copyright law.
* Dissemination of this information or reproduction of this material
* is strictly forbidden unless prior written permission is obtained
  from Adobe Systems Incorporated.
**************************************************************************/
package com.adobe.internal.pdftoolkit.core.cos;

import java.util.Iterator;
import java.util.List;
import java.util.Locale;

import com.adobe.internal.io.ByteWriterFactory;
import com.adobe.internal.io.SimpleByteWriterFactory;
import com.adobe.internal.pdftoolkit.core.filter.CustomFilterRegistry;
import com.adobe.internal.pdftoolkit.core.filter.spi.CustomFilter;
import com.adobe.internal.pdftoolkit.core.fontset.PDFFontSet;

/**
 * Provide options to the {@link com.adobe.internal.pdftoolkit.core.cos.CosDocument CosDocument}
 * when it is created.
 */
public class CosOpenOptions
{
	private ByteWriterFactory mByteWriterFactory;
	private long mEofValue = Long.MAX_VALUE;
	private boolean mSaveInPlace = false;
	private boolean mNoPreloadXRef = false;
	private PDFFontSet fontSet;
	private Locale docLocale = Locale.US;
	private CustomFilterRegistry filterRegistry;
	private boolean mRepairEnabled = true;
	private boolean mLateRepairEnabled = true;
	private boolean mSkipCorruptObjects = false;
	private boolean mDoNotEmbedFonts = false; //This flag takes care that font is not embedded in XObject generation, Appearances.
	

	/**
	 * This method returns true if the corrupted objects shall be skipped while parsing the document.
	 * In that case, the objects which couldn't be parsed shall be treated as {@link CosNull}.
	 * @return the mSkipCorruptObjects
	 */
	public boolean skipCorruptObjects() {
		return mSkipCorruptObjects;
	}

	/**
	 * This method sets true if the corrupted objects are to be skipped while parsing the document.
	 * In that case, the objects which couldn't be parsed shall be treated as {@link CosNull}.
	 * @param skipCorruptObjects
	 */
	public void setSkipCorruptObjects(boolean skipCorruptObjects) {
		this.mSkipCorruptObjects = skipCorruptObjects;
	}

	/**
	 * 
	 * Default constructor.
	 */
	protected CosOpenOptions()
	{
		mByteWriterFactory = new SimpleByteWriterFactory();
	}

	/**
	 * 
	 * Copy constructor.
	 * @param original the one to copy
  	 */
	protected CosOpenOptions(CosOpenOptions original)
	{
		this.mByteWriterFactory = original.mByteWriterFactory;
		this.mEofValue = original.mEofValue;
		this.mSaveInPlace = original.mSaveInPlace;
		this.mNoPreloadXRef = original.mNoPreloadXRef;
		this.fontSet = original.fontSet;
		this.docLocale = original.docLocale;
		this.filterRegistry = original.filterRegistry;
		this.mRepairEnabled = original.mRepairEnabled;
		this.mLateRepairEnabled = original.mLateRepairEnabled;
		this.mSkipCorruptObjects = original.mSkipCorruptObjects;
		this.mDoNotEmbedFonts = original.mDoNotEmbedFonts;
	}

	/**
	 * 
	 * Get the setting for the <code>ByteWriterFactory</code>.
	 * @return the byte writer factory
	 */
	protected ByteWriterFactory getByteWriterFactory()
	{
		return mByteWriterFactory;
	}

	/**
	 * Set the <code>ByteWriterFactory</code> to be used.  Note that this 
	 * <code>ByteWriterFactory</code> should only be used by one document.
	 * @param byteWriterFactory	the factory object for creating <code>ByteWriter</code>s
	 */
	public void setByteWriterFactory(ByteWriterFactory byteWriterFactory)
	{
		mByteWriterFactory = byteWriterFactory;
	}

	/**
	 * 
	 * @return Returns the eofValue.
	 */
	protected long getEofValue()
	{
		return mEofValue;
	}

	/**
	 * 
	 * @param eofValue The eofValue to set.
	 */
	public void setEofValue(long eofValue)
	{
		mEofValue = eofValue;
	}

	/**
	 * 
	 * @return true if in-place incremental save should be allowed
	 */
	protected boolean getSaveInPlace()
	{
		return mSaveInPlace;
	}

	/**
	 * Enables the document to be incrementally saved to its current stream.
	 * 
	 * @param saveInPlace true if in-place incremental save should be allowed
	 */
	public void setSaveInPlace(boolean saveInPlace)
	{
		mSaveInPlace = saveInPlace;
	}

	/**
	 * 
	 * @return true if preloading of the main XRef section is disabled
	 */
	protected boolean getNoPreloadXRef()
	{
		return mNoPreloadXRef;
	}

	/**
	 * 
	 * @param noPreloadXRef true if preloading of the main XRef section is disabled
	 */
	public void setNoPreloadXRef(boolean noPreloadXRef)
	{
		mNoPreloadXRef = noPreloadXRef;
	}

	public Locale getDocLocale() {
		return docLocale;
	}

	/**
	 * The primary locale of the document. 
	 */
	public void setDocLocale(Locale docLocale) {
		this.docLocale = docLocale;
	}

	public PDFFontSet getFontSet() {
		return fontSet;
	}

	/**
	 * The font set that should be used while processing the document.
	 */
	public void setFontSet(PDFFontSet fontSet) {
		this.fontSet = fontSet;
	}
	
	/**
	 * Get list of registered custom filters
	 * @return list of registered custom filters.
	 */
	public List/*<CustomFilter List>*/ getRegisteredFilters()
	{
		return getCustomFilterRegistry().getRegisteredFilters();
	}
	

	/**
	 * Supply lists that contains custom stream filter implementations for encoding and decoding.
	 * @param customFilters - list of custom filters to be used during stream encoding or decoding.
	 * 
	 */
	public void registerCustomFilters(List customFilters/*<CustomFilter List>*/)
	{
		if (customFilters == null)
			return;	
		
	
		Iterator customFilterIter =  customFilters.iterator();
		while (customFilterIter.hasNext())
		{
			CustomFilter customFilter = (CustomFilter) customFilterIter.next();
			procureFilterRegistry().registerFilter(customFilter);			
		}
	}
	
	/**
	 * 
	 * Return the FilterRegistry that contains custom stream filter
	 * implementations 
	 * @return filter registry
	 */
	public CustomFilterRegistry getCustomFilterRegistry() 
	{
		/*if (filterRegistry == null) 
			filterRegistry = procureFilterRegistry();*/
		
		return filterRegistry;
	}


	private CustomFilterRegistry procureFilterRegistry() 
	{
		if (filterRegistry == null)
		{
			filterRegistry = new CustomFilterRegistry();			
		}
		return filterRegistry;
	}
	
	/**
	 * Set/get state of repair enable boolean, default is true
	 */
	public void setRepairEnabled(boolean enableRepair)
	{
		mRepairEnabled = enableRepair;
	}

	public boolean getRepairEnabled()
	{
		return mRepairEnabled;
	}
	
	/**
	 * Set/get state of late repair enable boolean, default is true
	 */
	public void setLateRepairEnabled(boolean enableLateRepair)
	{
		mLateRepairEnabled = enableLateRepair;
	}

	public boolean getLateRepairEnabled()
	{
		return mLateRepairEnabled;
	}
	
	public boolean getDoNotEmbedFonts() {
		return mDoNotEmbedFonts;
	}

	public void setDoNotEmbedFonts(boolean mDoNotEmbedFonts) {
		this.mDoNotEmbedFonts = mDoNotEmbedFonts;
	}
}
