/*
 * File: CosPDFOptimizer.java
 *
 * ****************************************************************************
 *
 *	ADOBE CONFIDENTIAL
 *	___________________
 *
 *	Copyright 2005-2006 Adobe Systems Incorporated
 *	All Rights Reserved.
 *
 *	NOTICE: All information contained herein is, and remains the property of
 *	Adobe Systems Incorporated and its suppliers, if any. The intellectual
 *	and technical concepts contained herein are proprietary to Adobe Systems
 *	Incorporated and its suppliers and may be covered by U.S. and Foreign
 *	Patents, patents in process, and are protected by trade secret or
 *	copyright law. Dissemination of this information or reproduction of this
 *	material is strictly forbidden unless prior written permission is obtained
 *	from Adobe Systems Incorporated.
 *
 * ***************************************************************************/
package com.adobe.internal.pdftoolkit.core.cos;

import java.io.IOException;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;

import com.adobe.internal.io.stream.InputByteStream;
import com.adobe.internal.pdftoolkit.core.exceptions.PDFCosParseException;
import com.adobe.internal.pdftoolkit.core.exceptions.PDFIOException;
import com.adobe.internal.pdftoolkit.core.exceptions.PDFSecurityException;
import com.adobe.internal.pdftoolkit.core.types.ASName;
import com.adobe.internal.pdftoolkit.core.types.ASString;

/**
 * Provides methods for optimizing PDF files prior to saving.
 * Examples of such optimizations are unreferenced object removal, duplicate
 * resource object removal, embedded font coalescing, and rearrangements of the
 * objects placed in object streams. The idea is to <i>not</i> make these available as
 * save options because many of them are rather expensive and clients need to
 * see the price they're paying up front.
 *
 * @author peters
 */
final class CosPDFOptimizer
{
	// Object types
	private static final byte kNullType = 0;
	private static final byte kIntegerType = 1;
	private static final byte kRealType = 2;
	private static final byte kBooleanType = 3;
	private static final byte kNameType = 4;
	private static final byte kStringType = 5;
	private static final byte kDictionaryType = 6;
	private static final byte kArrayType = 7;
	private static final byte kStreamType = 8;

	private CosDocument mCosDoc;

	// Variables for freeDuplicateResources
	MessageDigest mDigester;
	Digest mNullDigest;
	HashMap mObjToDigest;
	HashMap mDigestToObj;

	/**
	 * The constructor, requires a PDF document to work with
	 */
	private CosPDFOptimizer(CosDocument cosDocument)
	{
		mCosDoc = cosDocument;
	}

	/**
	 * Encapsulates the constructor.
	 * @param cosDocument a PDF document.
	 * @return The constructed PDFOptimizer object.
	 */
	protected static CosPDFOptimizer newInstance(CosDocument cosDocument)
	{
		return new CosPDFOptimizer(cosDocument);
	}

	/**
	 * Parse the document to identify and remove duplicate resources
	 * referenced in the page resource object trees. Replaces
	 * duplicates with the previous identical object in our cache.
	 * Duplicates are then removed.
	 * @throws PDFSecurityException 
	 * @throws PDFIOException 
	 * @throws PDFCosParseException 
	 */
	protected void freeDuplicateResources() 
	throws PDFCosParseException, IOException, PDFIOException, PDFSecurityException
	{
		mDigester = newDigesterInstance();
		if (mDigester == null)
			return;
		mNullDigest = new Digest(null);
		mObjToDigest = new HashMap();
		mDigestToObj = new HashMap();
		CosDictionary node;
		node = safeGetDict(mCosDoc.getRoot(), ASName.k_Pages);
		freeDupPageTreeNode(node);
	}

	// Here we process a page tree node
	private void freeDupPageTreeNode(CosDictionary node) 
	throws PDFCosParseException, IOException, PDFIOException, PDFSecurityException
	{
		if (node == null)
			return;
		CosArray kids = safeGetArray(node, ASName.k_Kids);
		if (kids != null) {
			Iterator iter = kids.iterator();
			while (iter.hasNext()) {
				freeDupPageTreeNode((CosDictionary)iter.next());
			}
		} else {
			freeDupPageRes(safeGetDict(node, ASName.k_Resources));
		}
	}

	// Here we process a page resource dictionary
	private void freeDupPageRes(CosDictionary pageRes)
	throws PDFCosParseException, IOException, PDFIOException, PDFSecurityException
	{
		if (pageRes == null)
			return;
		Iterator iter = pageRes.keyIterator();
		while (iter.hasNext()) {
			ASName className = (ASName)iter.next();
			CosDictionary resClass = safeGetDict(pageRes, className);
			freeDupResObj(resClass);
		}
	}

	/** Here to process a resource object at any level. If the passed object
	 * is indirect, we create a new digester for it. When we're done we save
	 * the digest specific to that object (tree) and then pop back a level,
	 * merging the resulting value into the digest at the previous level. As
	 * we go we check to see if any objects we already saw had the same digest
	 * as this one and use them instead if they did. That's how we get rid
	 * of duplicate objects (trees) at whatever level duplication occurs.
	 * @throws PDFSecurityException 
	 * @throws PDFIOException 
	 * @throws PDFCosParseException 
	 */
	private CosObject freeDupResObj(CosObject resource) 
	throws PDFCosParseException, PDFIOException, IOException, PDFSecurityException
	{
		if (resource == null || resource instanceof CosNull) {
			updateByte(kNullType);
			return null;
		}
		byte[] localDigest;
		MessageDigest savedDigester = mDigester;
		int objNum = resource.getObjNum();
		Integer objNumInt = Integer.valueOf(objNum);
		if (objNum != 0) {
			if (mObjToDigest.containsKey(objNumInt)) {
				localDigest = ((Digest)mObjToDigest.get(objNumInt)).getBytes();
				if (localDigest != null)
					mDigester.update(localDigest);
				return null;
			} else {
				mDigester = newDigesterInstance();
				mObjToDigest.put(objNumInt, mNullDigest);
			}
		}
			if (resource instanceof CosNumeric) {
				updateNumber(resource.numberValue());
			} else if (resource instanceof CosName) {
				updateName(resource.nameValue());
			} else if (resource instanceof CosBoolean) {
				updateBoolean(resource.booleanValue());
			} else if (resource instanceof CosString) {
				updateString(resource.stringValue());
			} else if (resource instanceof CosDictionary) {
				if (resource instanceof CosStream)
					updateByte(kStreamType);
				else
					updateByte(kDictionaryType);
				updateInteger(((CosDictionary)resource).size());
				ArrayList keys = new ArrayList(((CosDictionary)resource).keySet());
				Collections.sort(keys);
				Iterator iter = keys.iterator();
				while (iter.hasNext()) {
					ASName name = (ASName)iter.next();
					updateName(name);
					CosObject value = ((CosDictionary)resource).get(name);
					CosObject prevValue = freeDupResObj(value);
					if (prevValue != null)
						((CosDictionary)resource).put(name, prevValue);
				}
				if (resource instanceof CosStream) {
					int type = 0;
					long length = 0;
					InputByteStream stream = ((CosStream)resource).getStreamEncoded();
					if (stream != null && stream.bytesAvailable() != 0) {
						type = 1;
						length = stream.bytesAvailable();
					} else {
						stream = ((CosStream)resource).getStreamDecoded();
						if (stream != null && stream.bytesAvailable() != 0) {
							type = 2;
							length = stream.bytesAvailable();
						}
					}
					updateInteger(type);
					updateInteger((int)length);
					byte[] buffer = new byte[4096];
					while (length > 0) {
						int bytesRead = stream.read(buffer, 0, buffer.length);
						mDigester.update(buffer, 0, bytesRead);
						length -= bytesRead;
					}
				}
			} else if (resource instanceof CosArray) {
				updateByte(kArrayType);
				int size = ((CosArray)resource).size();
				updateInteger(size);
				for (int i = 0; i < size; i++) {
					CosObject value = ((CosArray)resource).get(i);
					CosObject prevValue = freeDupResObj(value);
					if (prevValue != null)
						((CosArray)resource).set(i, prevValue);
				}
			}
			
		if (objNum != 0) {
			localDigest = mDigester.digest();
			mDigester = savedDigester;
			mDigester.update(localDigest);
			Digest digestObj = new Digest(localDigest);
			if (mDigestToObj.containsKey(digestObj))
				return (CosObject)mDigestToObj.get(digestObj);
			mDigestToObj.put(digestObj, resource);
			mObjToDigest.put(objNumInt, digestObj);
		}
		return null;
	}

	// Digest a number, includes tag
	private final void updateNumber(Number num)
	{
		if (num instanceof Integer) {
			updateByte(kIntegerType);
			updateInteger(num.intValue());
		} else {
			updateByte(kRealType);
			long bits = Double.doubleToLongBits(num.doubleValue());
			updateInteger((int)(bits >> 32));
			updateInteger((int)bits);
		}
	}

	// Digest a boolean, includes tag
	private final void updateBoolean(boolean bool)
	{
		updateByte(kBooleanType);
		if (bool)
			updateByte((byte)1);
		else
			updateByte((byte)0);
	}

	// Digest a name value, includes tag
	private final void updateName(ASName name)
	{
		updateByte(kNameType);
		byte[] nameContent = name.getBytes();
		updateInteger(nameContent.length);
		updateByteArray(nameContent);
	}

	// Digest a string, includes tag
	private final void updateString(ASString string)
	{
		updateByte(kStringType);
		byte[] stringBytes = string.getBytes();
		updateInteger(stringBytes.length);
		updateByteArray(stringBytes);
	}

	// Digest a byte, does NOT include tag
	private final void updateByte(byte b)
	{
		mDigester.update(b);
	}

	// Digest a byte array, does NOT include tag
	private final void updateByteArray(byte[] data)
	{
		mDigester.update(data);
	}

	// Digest an integer, does NOT include tag
	private final void updateInteger(int num)
	{
		byte[] data = new byte[4];
		data[0] = (byte)(num >> 24);
		data[1] = (byte)(num >> 16);
		data[2] = (byte)(num >> 8);
		data[3] = (byte)num;
		updateByteArray(data);
	}

	// Create a new digester instance (for each indirect object)
	private static final MessageDigest newDigesterInstance()
	{
		try {
			return MessageDigest.getInstance("MD5");
		}
		catch (NoSuchAlgorithmException e) {
			return null;
		}
	}

	// Get a dictionary item and return null if the item isn't a dictionary
	private static final CosDictionary safeGetDict(CosDictionary dict, ASName key) 
	throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		CosObject obj;
			obj = dict.get(key);
		if (obj instanceof CosDictionary)
			return (CosDictionary)obj;
		return null;
	}

	// Get a dictionary item and return null if the item isn't an array
	private static final CosArray safeGetArray(CosDictionary dict, ASName key) 
	throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		CosObject obj;

		obj = dict.get(key);
		if (obj instanceof CosArray)
			return (CosArray)obj;
		return null;
	}

	// Utility class for hash map of digests
	static private final class Digest
	{
		byte[] mDigest;

		Digest(byte[] digest) {mDigest = digest;}

		byte[] getBytes(){return mDigest;}

		@Override
		public boolean equals(Object digest)
		{
			if(!(digest instanceof Digest))
				return false;
			byte[] digBytes = ((Digest)digest).getBytes();
			if (mDigest == null && digBytes == null)
				return true;
			if (mDigest == null || digBytes == null)
				return false;
			int length = mDigest.length;
			if (digBytes.length != length)
				return false;
			for (int i = 0; i < length; i++)
				if (digBytes[i] != mDigest[i])
					return false;
			return true;
		}

		// Hash algorithm borrowed from ASName.java
		@Override
		public int hashCode()
		{
			if (mDigest == null)
				return 0;
			int hash = 0;
			int len = mDigest.length;
			for (int i = 0; i < len; i++)
				hash ^= mDigest[i] * 0x9E3779B9;
			return hash;
		}
	}
}
