/* ****************************************************************************
 *
 *	File: CosSaveParams.java
 *
 * ****************************************************************************
 *
 *	ADOBE CONFIDENTIAL
 *	___________________
 *
 *	Copyright 2004-2005 Adobe Systems Incorporated
 *	All Rights Reserved.
 *
 *	NOTICE: All information contained herein is, and remains the property of
 *	Adobe Systems Incorporated and its suppliers, if any. The intellectual
 *	and technical concepts contained herein are proprietary to Adobe Systems
 *	Incorporated and its suppliers and may be covered by U.S. and Foreign
 *	Patents, patents in process, and are protected by trade secret or
 *	copyright law. Dissemination of this information or reproduction of this
 *	material is strictly forbidden unless prior written permission is obtained
 *	from Adobe Systems Incorporated.
 *
 * ***************************************************************************/

package com.adobe.internal.pdftoolkit.core.cos;

import java.util.Map;

import com.adobe.internal.io.ByteWriter;
import com.adobe.internal.pdftoolkit.core.exceptions.PDFInvalidParameterException;

/**
 * @author peters
 *
 * Class CosSaveParams - parameters specifying how to perform a save.
 * An instance of this is a required parameter for CosDocument.save().
 */
public class CosSaveParams implements Cloneable
{
	// xrefstyle selections enum
	public static final int CURRENT_XREF = 0;
	public static final int XREFTABLE = 1;
	public static final int XREFSTREAM = 2;
	public static final int XREFHYBRID = 3;

	// save style selections enum
	public static final int SAVE_STYLE_DEFAULT = 0;
	public static final int SAVE_STYLE_INCREMENTAL = 1;
	public static final int SAVE_STYLE_FULL = 2;
	public static final int SAVE_STYLE_LINEAR = 3;

	// the parameters and their default values
	private String mHeader = null;
	private String mHeaderToken = null;
	private String mVersion = null;
	private Map mExtensions = null;
	private int mXrefStyle = CURRENT_XREF;
	private int mSaveStyle = SAVE_STYLE_DEFAULT;
	private ByteWriter mTempByteWriter = null;
	private boolean mCloseAfterSave = false;
	private boolean mSaveToCopy = false;
	private boolean mForceCompress = false;

	/**
	 * Constructor with specified save style.
	 */
	public CosSaveParams(int saveStyle)
	{
		mSaveStyle = saveStyle;
	}

	// get/set methods for parameters

	/**
	 * Sets header to sue while saving.
	 */
	public void setHeader(String header)
	{
		mHeader = header;
	}

	/**
	 * Returns header.
	 */
	public String getHeader()
	{
		return mHeader;
	}

	/**
	 * Sets header token.
	 */
	public void setHeaderToken(String headerToken)
		throws PDFInvalidParameterException
	{
		mHeaderToken = headerToken;
		if (mHeaderToken != null) {
			if (mHeaderToken.length() > 10)
				throw new PDFInvalidParameterException("Header token too long");
			for (int i = 0; i < mHeaderToken.length(); i++) {
				char c = mHeaderToken.charAt(i);
				if (c == '\r' || c == '\n' || (c & 0xFF00) != 0)
					throw new PDFInvalidParameterException("Illegal header token");
			}
		}
	}

	/**
	 * Returns header token.
	 */
	public String getHeaderToken()
	{
		return mHeaderToken;
	}

	/**
	 * Sets version to save with.
	 */
	public void setVersion(String version)
	{
		mVersion = version;
	}

	/**
	 * Returns version to save with.
	 */
	public String getVersion()
	{
		return mVersion;
	}

	/**
	 * Returns version extensions.
	 */
	public Map getExtensions() {
		return mExtensions;
	}

	/**
	 * Sets version extensions to use.
	 */
	public void setExtensions(Map extensions) {
		mExtensions = extensions;
	}

	/**
	 * 
	 * @param xrefstyle - int enum for specifying full save xref style:
	 * CURRENT_XREF - use existing style (default for existing document)
	 * XREFTABLE - xref table (default for new document)
	 * XREFSTREAM - xref stream (required if compressed object streams)
	 * XREFHYBRID - hybrid (compressed object streams only visible to v 1.5 or later)
	 *
	 */
	public void setXrefStyle(int xrefstyle)
	{
		mXrefStyle = xrefstyle;
	}

	/**
	 * Returns xref style to save with.
	 */
	public int getXrefStyle()
	{
		return mXrefStyle;
	}

	/**
	 * Sets save style to save with.
	 */
	public void setSaveStyle(int saveStyle)
	{
		mSaveStyle = saveStyle;
	}

	/**
	 * Returns save style to save with.
	 */
	public int getSaveStyle()
	{
		return mSaveStyle;
	}

	/**
	 * Sets true if document to be closed after save.
	 */
	public void setCloseAfterSave(boolean closeAfterSave)
	{
		mCloseAfterSave = closeAfterSave;
	}

	/**
	 * Returns true if document to be closed after save.
	 */
	public boolean getCloseAfterSave()
	{
		return mCloseAfterSave;
	}

	/**
	 * Specifies if force compression to be used.
	 */
	public void setForceCompress(boolean forceCompress)
	{
		mForceCompress = forceCompress;
	}

	/**
	 * Returns true if force compression to be used.
	 */
	public boolean getForceCompress()
	{
		return mForceCompress;
	}

	/**
	 * Only for internal engineering use. This api can change without notice.
	 */
	public void setTempByteWriter(ByteWriter tempByteWriter)
	{
		mTempByteWriter = tempByteWriter;
	}

	/**
	 * Only for internal engineering use. This api can change without notice.
	 */
	public ByteWriter getTempByteWriter()
	{
		return mTempByteWriter;
	}
	
	/**
	 * 
	 * Save the CosDocument to the ByteWriter given during save but
	 * do NOT make it the backing store and continue to use the existing one.
	 * 
	 * @param saveToCopy will document be saved as a copy
	 */
	public void setSaveToCopy(boolean saveToCopy)
	{
		mSaveToCopy = saveToCopy;
	}

	/**
	 * 
	 * Save the CosDocument to the ByteWriter given during save but
	 * do NOT make it the backing store and continue to use the existing one.
	 */
	public boolean getSaveToCopy()
	{
		return mSaveToCopy;
	}

	/**
	 * Produce a simple, shallow clone.
	 */
	@Override
	public Object clone()
	{
		try {
			return super.clone();
		} catch (CloneNotSupportedException e) {
			return null;
		}
	}
}
